% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PAC_deseq.R
\name{PAC_deseq}
\alias{PAC_deseq}
\title{A wrapper to DESeq2 that apply differential expression analysis on a PAC
object}
\usage{
PAC_deseq(
  PAC,
  model,
  deseq_norm = FALSE,
  test = "Wald",
  fitType = "local",
  threads = 1,
  pheno_target = NULL
)
}
\arguments{
\item{PAC}{PAC object containing a Pheno data.frame with samples as row
names, an Anno data.frame with sequences as row names and a Counts table
with raw counts. The Counts table must have the sample names as column
names and sequences as row names.}

\item{model}{Character vector describing the statistical model based on
column names in Pheno.}

\item{deseq_norm}{Logical whether to return deseq normalized values or raw
counts (default=FALSE).}

\item{test}{Character parsed directly to \code{\link[DESeq2]{DESeq}} that
controls what type of statistical test that should be used. Alternatives
are either "Wald" (Wald significance test) or "LTR" (likelihood ratio
test/chi-squared test). See \code{\link[DESeq2]{DESeq}} for more details.
(Default="Wald")}

\item{fitType}{Character parsed directly to \code{\link[DESeq2]{DESeq}} that
controls what type of dispersion fit that should be used. Alternatives are
either "parametric" (dispersion-mean relation), "local" (local regression
of log dispersions), "mean" (mean of gene-wise dispersion). See
\code{\link[DESeq2]{DESeq}} for more details. (Default="local")}

\item{threads}{Integer number of threads to run in parallel.}

\item{pheno_target}{(optional) List with: 1st object being a character
  indicating the main target column in Pheno. 2nd object being a character
  vector of the target group(s) in the target Pheno column (1st object).

  Important: In \code{PAC_deseq}, pheno_target controls the main comparative
  factor category from which a summarized table and plots will be generated.
  If, for instance, a target column named "groups" in pheno(PAC) contains
  "control" and "treatment" categories, setting pheno_target=list("groups",
  c("treatment", "controls") ensures that "treatment" is presented 1st in the
  factor levels, making for example log2FC appear as "treatment vs control".
  As default, pheno_target=NULL will result in the factor levels being
  automatically sorted in ascending order, which in the example above would
  result in control vs treatment. If no pheno_target is given, the
  first feature in the model will also be the main comparison presented in
  the graphs and summary tables.}
}
\value{
A list of objects: 

   1st object - Summarized result table merged with anno(PAC) 
   
   2nd object - Target graphs (p-val distribution and volcano) 
   
   3rd object - All output from \code{\link[DESeq2]{DESeq}}
}
\description{
\code{PAC_deseq} DESeq2 analysis on PAC_object.
}
\details{
Given a PAC object this function will apply a differential expression
analysis using DESeq2.
}
\examples{


# Note, these examples will generate some warnings since data is based on
# heavily down-sampled fastq files, where many sequences receives low counts
# in specific groups.

## Load test data
load(system.file("extdata", "drosophila_sRNA_pac_filt_anno.Rdata", 
                package = "seqpac", mustWork = TRUE))

## Simple model with embryonic stages using Wald test with local fit (default)
table(pheno(pac)$stage)
output_deseq <- suppressWarnings(PAC_deseq(pac, model= ~stage, threads=1))

## Batch corrected, graphs are generated for 'stage' (=first in the model)
output_deseq <- suppressWarnings(PAC_deseq(pac, model= ~stage + batch,
                                          threads=1))

## Using pheno_target 
output_deseq <- suppressWarnings(PAC_deseq(pac,model= ~stage + batch, 
                                          pheno_target=list("batch"),
                                          threads=1))

## With pheno_target we can change the direction for the comparison
# Stage5 vs Stage3 (reverse order):
output_deseq <- suppressWarnings(PAC_deseq(pac, model= ~stage + batch, 
                         pheno_target = list("stage", c("Stage5", "Stage3")),
                         threads=1))  

## In the output you find PAC merged results, target plots and output_deseq   
names(output_deseq)
tibble::as_tibble(output_deseq$result)

}
\seealso{
\url{https://github.com/OestLab/seqpac} for updates on the current
  package.

Other PAC analysis: 
\code{\link{PAC_covplot}()},
\code{\link{PAC_filter}()},
\code{\link{PAC_filtsep}()},
\code{\link{PAC_gtf}()},
\code{\link{PAC_jitter}()},
\code{\link{PAC_mapper}()},
\code{\link{PAC_nbias}()},
\code{\link{PAC_norm}()},
\code{\link{PAC_pca}()},
\code{\link{PAC_pie}()},
\code{\link{PAC_saturation}()},
\code{\link{PAC_sizedist}()},
\code{\link{PAC_stackbar}()},
\code{\link{PAC_summary}()},
\code{\link{PAC_trna}()},
\code{\link{as.PAC}()},
\code{\link{filtsep_bin}()},
\code{\link{map_rangetype}()},
\code{\link{tRNA_class}()}
}
\concept{PAC analysis}
