# Analysis overview

<script>
document.addEventListener("click", function (event) {
    if (event.target.classList.contains("rebook-collapse")) {
        event.target.classList.toggle("active");
        var content = event.target.nextElementSibling;
        if (content.style.display === "block") {
            content.style.display = "none";
        } else {
            content.style.display = "block";
        }
    }
})
</script>

<style>
.rebook-collapse {
  background-color: #eee;
  color: #444;
  cursor: pointer;
  padding: 18px;
  width: 100%;
  border: none;
  text-align: left;
  outline: none;
  font-size: 15px;
}

.rebook-content {
  padding: 0 18px;
  display: none;
  overflow: hidden;
  background-color: #f1f1f1;
}
</style>

## Outline

This chapter provides an overview of the framework of a typical scRNA-seq analysis workflow (Figure \@ref(fig:scworkflow)).

<div class="figure">
<img src="images/workflow.png" alt="Schematic of a typical scRNA-seq analysis workflow. Each stage (separated by dashed lines) consists of a number of specific steps, many of which operate on and modify a `SingleCellExperiment` instance." width="735" />
<p class="caption">(\#fig:scworkflow)Schematic of a typical scRNA-seq analysis workflow. Each stage (separated by dashed lines) consists of a number of specific steps, many of which operate on and modify a `SingleCellExperiment` instance.</p>
</div>

In the simplest case, the workflow has the following form:

1. We compute quality control metrics to remove low-quality cells that would interfere with downstream analyses.
These cells may have been damaged during processing or may not have been fully captured by the sequencing protocol.
Common metrics includes the total counts per cell, the proportion of spike-in or mitochondrial reads and the number of detected features.
2. We convert the counts into normalized expression values to eliminate cell-specific biases (e.g., in capture efficiency).
This allows us to perform explicit comparisons across cells in downstream steps like clustering.
We also apply a transformation, typically log, to adjust for the mean-variance relationship. 
3. We perform feature selection to pick a subset of interesting features for downstream analysis.
This is done by modelling the variance across cells for each gene and retaining genes that are highly variable.
The aim is to reduce computational overhead and noise from uninteresting genes.
4. We apply dimensionality reduction to compact the data and further reduce noise.
Principal components analysis is typically used to obtain an initial low-rank representation for more computational work,
followed by more aggressive methods like $t$-stochastic neighbor embedding for visualization purposes.
5. We cluster cells into groups according to similarities in their (normalized) expression profiles.
This aims to obtain groupings that serve as empirical proxies for distinct biological states.
We typically interpret these groupings by identifying differentially expressed marker genes between clusters.

Subsequent chapters will describe each analysis step in more detail.

## Quick start (simple)

Here, we use the a droplet-based retina dataset from @macosko2015highly, provided in the *[scRNAseq](https://bioconductor.org/packages/3.23/scRNAseq)* package.
This starts from a count matrix and finishes with clusters (Figure \@ref(fig:quick-start-umap)) in preparation for biological interpretation.
Similar workflows are available in abbreviated form in later parts of the book.


``` r
library(scRNAseq)
sce <- MacoskoRetinaData()

# Quality control (using mitochondrial genes).
library(scater)
is.mito <- grepl("^MT-", rownames(sce))
qcstats <- perCellQCMetrics(sce, subsets=list(Mito=is.mito))
filtered <- quickPerCellQC(qcstats, percent_subsets="subsets_Mito_percent")
sce <- sce[, !filtered$discard]

# Normalization.
sce <- logNormCounts(sce)

# Feature selection.
library(scran)
dec <- modelGeneVar(sce)
hvg <- getTopHVGs(dec, prop=0.1)

# PCA.
library(scater)
set.seed(1234)
sce <- runPCA(sce, ncomponents=25, subset_row=hvg)

# Clustering.
library(bluster)
colLabels(sce) <- clusterCells(sce, use.dimred='PCA',
    BLUSPARAM=NNGraphParam(cluster.fun="louvain"))    

# Visualization.
sce <- runUMAP(sce, dimred = 'PCA')
plotUMAP(sce, colour_by="label")
```

<div class="figure">
<img src="analysis-overview_files/figure-html/quick-start-umap-1.png" alt="UMAP plot of the retina dataset, where each point is a cell and is colored by the assigned cluster identity." width="672" />
<p class="caption">(\#fig:quick-start-umap)UMAP plot of the retina dataset, where each point is a cell and is colored by the assigned cluster identity.</p>
</div>

``` r
# Marker detection.
markers <- findMarkers(sce, test.type="wilcox", direction="up", lfc=1)
```

## Quick start (multiple batches)

Here we use the pancreas Smart-seq2 dataset from @segerstolpe2016singlecell, again provided in the *[scRNAseq](https://bioconductor.org/packages/3.23/scRNAseq)* package.
This starts from a count matrix and finishes with clusters (Figure \@ref(fig:quick-start-umap)) with some additional tweaks to eliminate uninteresting batch effects between individuals.
Note that a more elaborate analysis of the same dataset with justifications for each step is available in [Workflow Chapter 8](http://bioconductor.org/books/3.23/OSCA.workflows/segerstolpe-human-pancreas-smart-seq2.html#segerstolpe-human-pancreas-smart-seq2).


``` r
sce <- SegerstolpePancreasData()

# Quality control (using ERCCs).
qcstats <- perCellQCMetrics(sce)
filtered <- quickPerCellQC(qcstats, percent_subsets="altexps_ERCC_percent")
sce <- sce[, !filtered$discard]

# Normalization.
sce <- logNormCounts(sce)

# Feature selection, blocking on the individual of origin.
dec <- modelGeneVar(sce, block=sce$individual)
hvg <- getTopHVGs(dec, prop=0.1)

# Batch correction.
library(batchelor)
set.seed(1234)
sce <- correctExperiments(sce, batch=sce$individual, 
    subset.row=hvg, correct.all=TRUE)

# Clustering.
colLabels(sce) <- clusterCells(sce, use.dimred='corrected')

# Visualization.
sce <- runUMAP(sce, dimred = 'corrected')
gridExtra::grid.arrange(
    plotUMAP(sce, colour_by="label"),
    plotUMAP(sce, colour_by="individual"),
    ncol=2
)
```

<div class="figure">
<img src="analysis-overview_files/figure-html/quick-start2-umap-1.png" alt="UMAP plot of the pancreas dataset, where each point is a cell and is colored by the assigned cluster identity (left) or the individual of origin (right)." width="960" />
<p class="caption">(\#fig:quick-start2-umap)UMAP plot of the pancreas dataset, where each point is a cell and is colored by the assigned cluster identity (left) or the individual of origin (right).</p>
</div>

``` r
# Marker detection, blocking on the individual of origin.
markers <- findMarkers(sce, test.type="wilcox", direction="up", lfc=1)
```

## Session Info {-}

<button class="rebook-collapse">View session info</button>
<div class="rebook-content">
```
R Under development (unstable) (2025-10-20 r88955)
Platform: x86_64-pc-linux-gnu
Running under: Ubuntu 24.04.3 LTS

Matrix products: default
BLAS:   /home/biocbuild/bbs-3.23-bioc/R/lib/libRblas.so 
LAPACK: /usr/lib/x86_64-linux-gnu/lapack/liblapack.so.3.12.0  LAPACK version 3.12.0

locale:
 [1] LC_CTYPE=en_US.UTF-8       LC_NUMERIC=C              
 [3] LC_TIME=en_GB              LC_COLLATE=C              
 [5] LC_MONETARY=en_US.UTF-8    LC_MESSAGES=en_US.UTF-8   
 [7] LC_PAPER=en_US.UTF-8       LC_NAME=C                 
 [9] LC_ADDRESS=C               LC_TELEPHONE=C            
[11] LC_MEASUREMENT=en_US.UTF-8 LC_IDENTIFICATION=C       

time zone: America/New_York
tzcode source: system (glibc)

attached base packages:
[1] stats4    stats     graphics  grDevices utils     datasets  methods  
[8] base     

other attached packages:
 [1] batchelor_1.27.0            bluster_1.21.0             
 [3] scran_1.39.0                scater_1.39.0              
 [5] ggplot2_4.0.1               scuttle_1.21.0             
 [7] scRNAseq_2.25.0             SingleCellExperiment_1.33.0
 [9] SummarizedExperiment_1.41.0 Biobase_2.71.0             
[11] GenomicRanges_1.63.1        Seqinfo_1.1.0              
[13] IRanges_2.45.0              S4Vectors_0.49.0           
[15] BiocGenerics_0.57.0         generics_0.1.4             
[17] MatrixGenerics_1.23.0       matrixStats_1.5.0          
[19] BiocStyle_2.39.0            rebook_1.21.0              

loaded via a namespace (and not attached):
  [1] RColorBrewer_1.1-3        jsonlite_2.0.0           
  [3] CodeDepends_0.6.6         magrittr_2.0.4           
  [5] ggbeeswarm_0.7.3          GenomicFeatures_1.63.1   
  [7] gypsum_1.7.0              farver_2.1.2             
  [9] rmarkdown_2.30            BiocIO_1.21.0            
 [11] vctrs_0.6.5               DelayedMatrixStats_1.33.0
 [13] memoise_2.0.1             Rsamtools_2.27.0         
 [15] RCurl_1.98-1.17           htmltools_0.5.9          
 [17] S4Arrays_1.11.1           AnnotationHub_4.1.0      
 [19] curl_7.0.0                BiocNeighbors_2.5.0      
 [21] Rhdf5lib_1.33.0           SparseArray_1.11.10      
 [23] rhdf5_2.55.12             sass_0.4.10              
 [25] alabaster.base_1.11.1     bslib_0.9.0              
 [27] alabaster.sce_1.11.0      httr2_1.2.2              
 [29] cachem_1.1.0              ResidualMatrix_1.21.0    
 [31] GenomicAlignments_1.47.0  igraph_2.2.1             
 [33] lifecycle_1.0.4           pkgconfig_2.0.3          
 [35] rsvd_1.0.5                Matrix_1.7-4             
 [37] R6_2.6.1                  fastmap_1.2.0            
 [39] digest_0.6.39             AnnotationDbi_1.73.0     
 [41] RSpectra_0.16-2           dqrng_0.4.1              
 [43] irlba_2.3.5.1             ExperimentHub_3.1.0      
 [45] RSQLite_2.4.5             beachmat_2.27.1          
 [47] labeling_0.4.3            filelock_1.0.3           
 [49] httr_1.4.7                abind_1.4-8              
 [51] compiler_4.6.0            bit64_4.6.0-1            
 [53] withr_3.0.2               S7_0.2.1                 
 [55] BiocParallel_1.45.0       viridis_0.6.5            
 [57] DBI_1.2.3                 HDF5Array_1.39.0         
 [59] alabaster.ranges_1.11.0   alabaster.schemas_1.11.0 
 [61] rappdirs_0.3.3            DelayedArray_0.37.0      
 [63] rjson_0.2.23              tools_4.6.0              
 [65] vipor_0.4.7               otel_0.2.0               
 [67] beeswarm_0.4.0            glue_1.8.0               
 [69] h5mread_1.3.1             restfulr_0.0.16          
 [71] rhdf5filters_1.23.3       grid_4.6.0               
 [73] cluster_2.1.8.1           gtable_0.3.6             
 [75] ensembldb_2.35.0          metapod_1.19.1           
 [77] BiocSingular_1.27.1       ScaledMatrix_1.19.0      
 [79] XVector_0.51.0            RcppAnnoy_0.0.22         
 [81] ggrepel_0.9.6             BiocVersion_3.23.1       
 [83] pillar_1.11.1             limma_3.67.0             
 [85] dplyr_1.1.4               BiocFileCache_3.1.0      
 [87] lattice_0.22-7            FNN_1.1.4.1              
 [89] rtracklayer_1.71.3        bit_4.6.0                
 [91] tidyselect_1.2.1          locfit_1.5-9.12          
 [93] Biostrings_2.79.2         knitr_1.50               
 [95] gridExtra_2.3             bookdown_0.46            
 [97] ProtGenerics_1.43.0       edgeR_4.9.1              
 [99] xfun_0.55                 statmod_1.5.1            
[101] UCSC.utils_1.7.1          lazyeval_0.2.2           
[103] yaml_2.3.12               evaluate_1.0.5           
[105] codetools_0.2-20          cigarillo_1.1.0          
[107] tibble_3.3.0              alabaster.matrix_1.11.0  
[109] BiocManager_1.30.27       graph_1.89.1             
[111] cli_3.6.5                 uwot_0.2.4               
[113] jquerylib_0.1.4           dichromat_2.0-0.1        
[115] Rcpp_1.1.0.8.1            GenomeInfoDb_1.47.2      
[117] dir.expiry_1.19.0         dbplyr_2.5.1             
[119] png_0.1-8                 XML_3.99-0.20            
[121] parallel_4.6.0            blob_1.2.4               
[123] AnnotationFilter_1.35.0   sparseMatrixStats_1.23.0 
[125] bitops_1.0-9              viridisLite_0.4.2        
[127] alabaster.se_1.11.0       scales_1.4.0             
[129] crayon_1.5.3              rlang_1.1.6              
[131] cowplot_1.2.0             KEGGREST_1.51.1          
```
</div>
