---
output:
  html_document
bibliography: ref.bib
---

# (PART) Basic usage {-}

# Introduction

<script>
document.addEventListener("click", function (event) {
    if (event.target.classList.contains("rebook-collapse")) {
        event.target.classList.toggle("active");
        var content = event.target.nextElementSibling;
        if (content.style.display === "block") {
            content.style.display = "none";
        } else {
            content.style.display = "block";
        }
    }
})
</script>

<style>
.rebook-collapse {
  background-color: #eee;
  color: #444;
  cursor: pointer;
  padding: 18px;
  width: 100%;
  border: none;
  text-align: left;
  outline: none;
  font-size: 15px;
}

.rebook-content {
  padding: 0 18px;
  display: none;
  overflow: hidden;
  background-color: #f1f1f1;
}
</style>

## Motivation

The Bioconductor package *[SingleR](https://bioconductor.org/packages/3.23/SingleR)* implements an automatic annotation method 
for single-cell RNA sequencing (scRNA-seq) data [@aran2019reference].
Given a reference dataset of samples (single-cell or bulk) with known labels, 
it assigns those labels to new cells from a test dataset based on similarities in their expression profiles.
This provides a convenient way of transferring biological knowledge across datasets,
allowing users to leverage the domain expertise implicit in the creation of each reference.
The most common application of *[SingleR](https://bioconductor.org/packages/3.23/SingleR)* involves predicting cell type (or "state", or "kind") in a new dataset,
a process that is facilitated by the availability of curated references and compatibility with user-supplied datasets.
In this manner, the burden of manually interpreting clusters and defining marker genes only has to be done once, for the reference dataset, and this knowledge can be propagated to new datasets in an automated manner.

## Method description

*[SingleR](https://bioconductor.org/packages/3.23/SingleR)* can be considered a robust variant of nearest-neighbors classification,
with some tweaks to improve resolution for closely related labels.
For each test cell:

1. We compute the Spearman correlation between its expression profile and that of each reference sample.
The use of Spearman's correlation provides a measure of robustness to batch effects across datasets.
The calculation only uses the union of marker genes identified by pairwise comparisons between labels in the reference data,
so as to improve resolution of separation between labels.
2. We define the per-label score as a fixed quantile (by default, 0.8) of the correlations across all samples with that label.
This accounts for differences in the number of reference samples for each label, 
which interferes with simpler flavors of nearest neighbor classification;
it also avoids penalizing classifications to heterogeneous labels by only requiring a good match to a minority of samples.
3. We repeat the score calculation for all labels in the reference dataset.
The label with the highest score is used as *[SingleR](https://bioconductor.org/packages/3.23/SingleR)*'s prediction for this cell.
4. We optionally perform a fine-tuning step to improve resolution between closely related labels.
The reference dataset is subsetted to only include labels with scores close to the maximum;
scores are recomputed using only marker genes for the subset of labels, thus focusing on the most relevant features;
and this process is iterated until only one label remains.

## Quick start

We will demonstrate the use of `SingleR()` on a well-known 10X Genomics dataset [@zheng2017massively]
with the Human Primary Cell Atlas dataset [@hpcaRef] as the reference.


``` r
# Loading test data.
library(TENxPBMCData)
new.data <- TENxPBMCData("pbmc4k")

# Loading reference data with Ensembl annotations.
library(celldex)
ref.data <- HumanPrimaryCellAtlasData(ensembl=TRUE)

# Performing predictions.
library(SingleR)
predictions <- SingleR(test=new.data, assay.type.test=1, 
    ref=ref.data, labels=ref.data$label.main)

table(predictions$labels)
```

```
## 
##           B_cell              CMP               DC              GMP 
##              606                8                1                2 
##         Monocyte          NK_cell        Platelets Pre-B_cell_CD34- 
##             1164              217                3               46 
##          T_cells 
##             2293
```

And that's it, really.

## Where to get help

Questions on the general use of *[SingleR](https://bioconductor.org/packages/3.23/SingleR)* should be posted to 
the [Bioconductor support site](https://support.bioconductor.org).
Please send requests for general assistance and advice to the
support site rather than to the individual authors.
Bug reports or feature requests should be made to the [GitHub repository](https://github.com/LTLA/SingleR);
well-considered suggestions for improvements are always welcome.

## Session information {-}

<button class="rebook-collapse">View session info</button>
<div class="rebook-content">
```
R Under development (unstable) (2025-10-20 r88955)
Platform: x86_64-pc-linux-gnu
Running under: Ubuntu 24.04.3 LTS

Matrix products: default
BLAS:   /home/biocbuild/bbs-3.23-bioc/R/lib/libRblas.so 
LAPACK: /usr/lib/x86_64-linux-gnu/lapack/liblapack.so.3.12.0  LAPACK version 3.12.0

locale:
 [1] LC_CTYPE=en_US.UTF-8       LC_NUMERIC=C              
 [3] LC_TIME=en_GB              LC_COLLATE=C              
 [5] LC_MONETARY=en_US.UTF-8    LC_MESSAGES=en_US.UTF-8   
 [7] LC_PAPER=en_US.UTF-8       LC_NAME=C                 
 [9] LC_ADDRESS=C               LC_TELEPHONE=C            
[11] LC_MEASUREMENT=en_US.UTF-8 LC_IDENTIFICATION=C       

time zone: America/New_York
tzcode source: system (glibc)

attached base packages:
[1] stats4    stats     graphics  grDevices utils     datasets  methods  
[8] base     

other attached packages:
 [1] SingleR_2.13.0              ensembldb_2.35.0           
 [3] AnnotationFilter_1.35.0     GenomicFeatures_1.63.1     
 [5] AnnotationDbi_1.73.0        celldex_1.21.0             
 [7] TENxPBMCData_1.29.0         HDF5Array_1.39.0           
 [9] h5mread_1.3.1               rhdf5_2.55.12              
[11] DelayedArray_0.37.0         SparseArray_1.11.9         
[13] S4Arrays_1.11.1             abind_1.4-8                
[15] Matrix_1.7-4                SingleCellExperiment_1.33.0
[17] SummarizedExperiment_1.41.0 Biobase_2.71.0             
[19] GenomicRanges_1.63.1        Seqinfo_1.1.0              
[21] IRanges_2.45.0              S4Vectors_0.49.0           
[23] BiocGenerics_0.57.0         generics_0.1.4             
[25] MatrixGenerics_1.23.0       matrixStats_1.5.0          
[27] BiocStyle_2.39.0            rebook_1.21.0              

loaded via a namespace (and not attached):
 [1] DBI_1.2.3                 bitops_1.0-9             
 [3] httr2_1.2.2               CodeDepends_0.6.6        
 [5] rlang_1.1.6               magrittr_2.0.4           
 [7] otel_0.2.0                gypsum_1.7.0             
 [9] compiler_4.6.0            RSQLite_2.4.5            
[11] dir.expiry_1.19.0         DelayedMatrixStats_1.33.0
[13] png_0.1-8                 vctrs_0.6.5              
[15] ProtGenerics_1.43.0       pkgconfig_2.0.3          
[17] crayon_1.5.3              fastmap_1.2.0            
[19] dbplyr_2.5.1              XVector_0.51.0           
[21] Rsamtools_2.27.0          rmarkdown_2.30           
[23] UCSC.utils_1.7.1          graph_1.89.1             
[25] purrr_1.2.0               bit_4.6.0                
[27] xfun_0.54                 beachmat_2.27.0          
[29] cachem_1.1.0              cigarillo_1.1.0          
[31] GenomeInfoDb_1.47.2       jsonlite_2.0.0           
[33] blob_1.2.4                rhdf5filters_1.23.3      
[35] BiocParallel_1.45.0       Rhdf5lib_1.33.0          
[37] parallel_4.6.0            R6_2.6.1                 
[39] bslib_0.9.0               rtracklayer_1.71.2       
[41] jquerylib_0.1.4           Rcpp_1.1.0.8.1           
[43] bookdown_0.46             knitr_1.50               
[45] tidyselect_1.2.1          yaml_2.3.12              
[47] codetools_0.2-20          curl_7.0.0               
[49] lattice_0.22-7            tibble_3.3.0             
[51] withr_3.0.2               KEGGREST_1.51.1          
[53] evaluate_1.0.5            BiocFileCache_3.1.0      
[55] alabaster.schemas_1.11.0  ExperimentHub_3.1.0      
[57] Biostrings_2.79.2         pillar_1.11.1            
[59] BiocManager_1.30.27       filelock_1.0.3           
[61] RCurl_1.98-1.17           BiocVersion_3.23.1       
[63] sparseMatrixStats_1.23.0  alabaster.base_1.11.1    
[65] glue_1.8.0                alabaster.ranges_1.11.0  
[67] lazyeval_0.2.2            alabaster.matrix_1.11.0  
[69] tools_4.6.0               beachmat.hdf5_1.9.0      
[71] BiocIO_1.21.0             AnnotationHub_4.1.0      
[73] BiocNeighbors_2.5.0       GenomicAlignments_1.47.0 
[75] XML_3.99-0.20             grid_4.6.0               
[77] restfulr_0.0.16           cli_3.6.5                
[79] rappdirs_0.3.3            dplyr_1.1.4              
[81] alabaster.se_1.11.0       sass_0.4.10              
[83] digest_0.6.39             rjson_0.2.23             
[85] memoise_2.0.1             htmltools_0.5.9          
[87] lifecycle_1.0.4           httr_1.4.7               
[89] bit64_4.6.0-1            
```
</div>
