# Unfiltered human PBMCs (10X Genomics)

<script>
document.addEventListener("click", function (event) {
    if (event.target.classList.contains("rebook-collapse")) {
        event.target.classList.toggle("active");
        var content = event.target.nextElementSibling;
        if (content.style.display === "block") {
            content.style.display = "none";
        } else {
            content.style.display = "block";
        }
    }
})
</script>

<style>
.rebook-collapse {
  background-color: #eee;
  color: #444;
  cursor: pointer;
  padding: 18px;
  width: 100%;
  border: none;
  text-align: left;
  outline: none;
  font-size: 15px;
}

.rebook-content {
  padding: 0 18px;
  display: none;
  overflow: hidden;
  background-color: #f1f1f1;
}
</style>

## Introduction

Here, we describe a brief analysis of the peripheral blood mononuclear cell (PBMC) dataset from 10X Genomics [@zheng2017massively].
The data are publicly available from the [10X Genomics website](https://support.10xgenomics.com/single-cell-gene-expression/datasets/2.1.0/pbmc4k),
from which we download the raw gene/barcode count matrices, i.e., before cell calling from the _CellRanger_ pipeline.

## Data loading


``` r
library(DropletTestFiles)
raw.path <- getTestFile("tenx-2.1.0-pbmc4k/1.0.0/raw.tar.gz")
out.path <- file.path(tempdir(), "pbmc4k")
untar(raw.path, exdir=out.path)

library(DropletUtils)
fname <- file.path(out.path, "raw_gene_bc_matrices/GRCh38")
sce.pbmc <- read10xCounts(fname, col.names=TRUE)
```


``` r
library(scater)
rownames(sce.pbmc) <- uniquifyFeatureNames(
    rowData(sce.pbmc)$ID, rowData(sce.pbmc)$Symbol)

library(EnsDb.Hsapiens.v86)
location <- mapIds(EnsDb.Hsapiens.v86, keys=rowData(sce.pbmc)$ID, 
    column="SEQNAME", keytype="GENEID")
```

## Quality control

We perform cell detection using the `emptyDrops()` algorithm, as discussed in [Advanced Section 7.2](http://bioconductor.org/books/3.23/OSCA.advanced/droplet-processing.html#qc-droplets).


``` r
set.seed(100)
e.out <- emptyDrops(counts(sce.pbmc))
sce.pbmc <- sce.pbmc[,which(e.out$FDR <= 0.001)]
```


``` r
unfiltered <- sce.pbmc
```

We use a relaxed QC strategy and only remove cells with large mitochondrial proportions, using it as a proxy for cell damage.
This reduces the risk of removing cell types with low RNA content, especially in a heterogeneous PBMC population with many different cell types.


``` r
stats <- perCellQCMetrics(sce.pbmc, subsets=list(Mito=which(location=="MT")))
high.mito <- isOutlier(stats$subsets_Mito_percent, type="higher")
sce.pbmc <- sce.pbmc[,!high.mito]
```


``` r
summary(high.mito)
```

```
##    Mode   FALSE    TRUE 
## logical    4080     322
```


``` r
colData(unfiltered) <- cbind(colData(unfiltered), stats)
unfiltered$discard <- high.mito

gridExtra::grid.arrange(
    plotColData(unfiltered, y="sum", colour_by="discard") +
        scale_y_log10() + ggtitle("Total count"),
    plotColData(unfiltered, y="detected", colour_by="discard") +
        scale_y_log10() + ggtitle("Detected features"),
    plotColData(unfiltered, y="subsets_Mito_percent",
        colour_by="discard") + ggtitle("Mito percent"),
    ncol=2
)
```

<div class="figure">
<img src="tenx-unfiltered-pbmc4k_files/figure-html/unref-unfiltered-pbmc-qc-1.png" alt="Distribution of various QC metrics in the PBMC dataset after cell calling. Each point is a cell and is colored according to whether it was discarded by the mitochondrial filter." width="672" />
<p class="caption">(\#fig:unref-unfiltered-pbmc-qc)Distribution of various QC metrics in the PBMC dataset after cell calling. Each point is a cell and is colored according to whether it was discarded by the mitochondrial filter.</p>
</div>


``` r
plotColData(unfiltered, x="sum", y="subsets_Mito_percent",
    colour_by="discard") + scale_x_log10()
```

<div class="figure">
<img src="tenx-unfiltered-pbmc4k_files/figure-html/unref-unfiltered-pbmc-mito-1.png" alt="Proportion of mitochondrial reads in each cell of the PBMC dataset compared to its total count." width="672" />
<p class="caption">(\#fig:unref-unfiltered-pbmc-mito)Proportion of mitochondrial reads in each cell of the PBMC dataset compared to its total count.</p>
</div>

## Normalization


``` r
library(scran)
set.seed(1000)
clusters <- quickCluster(sce.pbmc)
sce.pbmc <- computeSumFactors(sce.pbmc, cluster=clusters)
sce.pbmc <- logNormCounts(sce.pbmc)
```


``` r
summary(sizeFactors(sce.pbmc))
```

```
##    Min. 1st Qu.  Median    Mean 3rd Qu.    Max. 
##  0.0034  0.7107  0.8792  1.0000  1.1036 11.8046
```


``` r
plot(librarySizeFactors(sce.pbmc), sizeFactors(sce.pbmc), pch=16,
    xlab="Library size factors", ylab="Deconvolution factors", log="xy")
```

<div class="figure">
<img src="tenx-unfiltered-pbmc4k_files/figure-html/unref-unfiltered-pbmc-norm-1.png" alt="Relationship between the library size factors and the deconvolution size factors in the PBMC dataset." width="672" />
<p class="caption">(\#fig:unref-unfiltered-pbmc-norm)Relationship between the library size factors and the deconvolution size factors in the PBMC dataset.</p>
</div>

## Variance modelling


``` r
set.seed(1001)
dec.pbmc <- modelGeneVarByPoisson(sce.pbmc)
top.pbmc <- getTopHVGs(dec.pbmc, prop=0.1)
```


``` r
plot(dec.pbmc$mean, dec.pbmc$total, pch=16, cex=0.5,
    xlab="Mean of log-expression", ylab="Variance of log-expression")
curfit <- metadata(dec.pbmc)
curve(curfit$trend(x), col='dodgerblue', add=TRUE, lwd=2)
```

<div class="figure">
<img src="tenx-unfiltered-pbmc4k_files/figure-html/unref-unfiltered-pbmc-var-1.png" alt="Per-gene variance as a function of the mean for the log-expression values in the PBMC dataset. Each point represents a gene (black) with the mean-variance trend (blue) fitted to simulated Poisson counts." width="672" />
<p class="caption">(\#fig:unref-unfiltered-pbmc-var)Per-gene variance as a function of the mean for the log-expression values in the PBMC dataset. Each point represents a gene (black) with the mean-variance trend (blue) fitted to simulated Poisson counts.</p>
</div>

## Dimensionality reduction


``` r
set.seed(10000)
sce.pbmc <- denoisePCA(sce.pbmc, subset.row=top.pbmc, technical=dec.pbmc)

set.seed(100000)
sce.pbmc <- runTSNE(sce.pbmc, dimred="PCA")

set.seed(1000000)
sce.pbmc <- runUMAP(sce.pbmc, dimred="PCA")
```

We verify that a reasonable number of PCs is retained.


``` r
ncol(reducedDim(sce.pbmc, "PCA"))
```

```
## [1] 8
```

## Clustering


``` r
g <- buildSNNGraph(sce.pbmc, k=10, use.dimred = 'PCA')
clust <- igraph::cluster_walktrap(g)$membership
colLabels(sce.pbmc) <- factor(clust)
```


``` r
table(colLabels(sce.pbmc))
```

```
## 
##   1   2   3   4   5   6   7   8   9  10  11  12  13  14  15  16  17  18  19 
## 249 559 367  46 173 914 450 285 266 111 157 130 137  46  41  85  19  16  29
```


``` r
plotTSNE(sce.pbmc, colour_by="label")
```

<div class="figure">
<img src="tenx-unfiltered-pbmc4k_files/figure-html/unref-unfiltered-pbmc-tsne-1.png" alt="Obligatory $t$-SNE plot of the PBMC dataset, where each point represents a cell and is colored according to the assigned cluster." width="672" />
<p class="caption">(\#fig:unref-unfiltered-pbmc-tsne)Obligatory $t$-SNE plot of the PBMC dataset, where each point represents a cell and is colored according to the assigned cluster.</p>
</div>

## Interpretation




``` r
markers <- findMarkers(sce.pbmc, pval.type="some", direction="up")
```

We examine the markers for cluster 7 in more detail.
High expression of _CD14_, _CD68_ and _MNDA_ combined with low expression of _FCGR3A_ (_CD16_) suggests that this cluster contains monocytes,
compared to macrophages in cluster 16 (Figure \@ref(fig:unref-mono-pbmc-markers)).


``` r
marker.set <- markers[["7"]]
as.data.frame(marker.set[1:30,1:3])
```

```
##                  p.value        FDR summary.logFC
## CSTA          1.119e-228 3.769e-224        2.3449
## S100A12       4.309e-219 7.259e-215        2.9671
## VCAN          3.002e-196 3.371e-192        2.2075
## MNDA          2.592e-190 2.183e-186        2.4570
## FCN1          1.091e-189 7.349e-186        2.6310
## TYMP          5.656e-149 3.176e-145        2.0055
## LGALS2        2.886e-145 1.389e-141        1.9130
## AIF1          8.929e-138 3.761e-134        2.4806
## MS4A6A        1.733e-135 6.486e-132        1.4980
## RP11-1143G9.4 1.269e-132 4.276e-129        2.8184
## FGL2          5.369e-132 1.644e-128        1.3690
## CD14          7.158e-121 2.010e-117        1.4279
## CFD           1.961e-114 5.082e-111        1.3114
## AP1S2         4.262e-107 1.026e-103        1.7835
## SERPINA1      3.393e-105 7.621e-102        1.3844
## CYBB          6.123e-104 1.290e-100        1.2483
## CLEC7A         7.224e-97  1.432e-93        1.0774
## KLF4           9.440e-95  1.767e-91        1.1718
## TNFSF13B       6.880e-92  1.220e-88        1.0524
## S100A8         1.526e-89  2.571e-86        4.7220
## NAMPT          3.679e-88  5.903e-85        1.0970
## CD36           3.426e-86  5.246e-83        1.0487
## MPEG1          2.504e-84  3.668e-81        0.9781
## CD68           1.330e-82  1.867e-79        0.9343
## CD302          1.506e-80  2.030e-77        0.8891
## CSF3R          6.392e-73  8.283e-70        0.8596
## RBP7           1.216e-72  1.517e-69        0.9072
## BLVRB          1.804e-72  2.171e-69        0.9711
## S100A11        1.824e-71  2.119e-68        1.8392
## CFP            3.688e-71  4.142e-68        1.0117
```




``` r
plotExpression(sce.pbmc, features=c("CD14", "CD68",
    "MNDA", "FCGR3A"), x="label", colour_by="label")
```

<div class="figure">
<img src="tenx-unfiltered-pbmc4k_files/figure-html/unref-mono-pbmc-markers-1.png" alt="Distribution of expression values for monocyte and macrophage markers across clusters in the PBMC dataset." width="672" />
<p class="caption">(\#fig:unref-mono-pbmc-markers)Distribution of expression values for monocyte and macrophage markers across clusters in the PBMC dataset.</p>
</div>

## Session Info {-}

<button class="rebook-collapse">View session info</button>
<div class="rebook-content">
```
R Under development (unstable) (2025-10-20 r88955)
Platform: x86_64-pc-linux-gnu
Running under: Ubuntu 24.04.3 LTS

Matrix products: default
BLAS:   /home/biocbuild/bbs-3.23-bioc/R/lib/libRblas.so 
LAPACK: /usr/lib/x86_64-linux-gnu/lapack/liblapack.so.3.12.0  LAPACK version 3.12.0

locale:
 [1] LC_CTYPE=en_US.UTF-8       LC_NUMERIC=C              
 [3] LC_TIME=en_GB              LC_COLLATE=C              
 [5] LC_MONETARY=en_US.UTF-8    LC_MESSAGES=en_US.UTF-8   
 [7] LC_PAPER=en_US.UTF-8       LC_NAME=C                 
 [9] LC_ADDRESS=C               LC_TELEPHONE=C            
[11] LC_MEASUREMENT=en_US.UTF-8 LC_IDENTIFICATION=C       

time zone: America/New_York
tzcode source: system (glibc)

attached base packages:
[1] stats4    stats     graphics  grDevices utils     datasets  methods  
[8] base     

other attached packages:
 [1] scran_1.39.0                EnsDb.Hsapiens.v86_2.99.0  
 [3] ensembldb_2.35.0            AnnotationFilter_1.35.0    
 [5] GenomicFeatures_1.63.1      AnnotationDbi_1.73.0       
 [7] scater_1.39.0               ggplot2_4.0.1              
 [9] scuttle_1.21.0              DropletUtils_1.31.0        
[11] SingleCellExperiment_1.33.0 SummarizedExperiment_1.41.0
[13] Biobase_2.71.0              GenomicRanges_1.63.1       
[15] Seqinfo_1.1.0               IRanges_2.45.0             
[17] S4Vectors_0.49.0            BiocGenerics_0.57.0        
[19] generics_0.1.4              MatrixGenerics_1.23.0      
[21] matrixStats_1.5.0           DropletTestFiles_1.21.0    
[23] BiocStyle_2.39.0            rebook_1.21.0              

loaded via a namespace (and not attached):
  [1] RColorBrewer_1.1-3        jsonlite_2.0.0           
  [3] CodeDepends_0.6.6         magrittr_2.0.4           
  [5] ggbeeswarm_0.7.3          farver_2.1.2             
  [7] rmarkdown_2.30            BiocIO_1.21.0            
  [9] vctrs_0.6.5               memoise_2.0.1            
 [11] Rsamtools_2.27.0          DelayedMatrixStats_1.33.0
 [13] RCurl_1.98-1.17           htmltools_0.5.9          
 [15] S4Arrays_1.11.1           AnnotationHub_4.1.0      
 [17] curl_7.0.0                BiocNeighbors_2.5.0      
 [19] Rhdf5lib_1.33.0           SparseArray_1.11.9       
 [21] rhdf5_2.55.12             sass_0.4.10              
 [23] bslib_0.9.0               httr2_1.2.2              
 [25] cachem_1.1.0              GenomicAlignments_1.47.0 
 [27] igraph_2.2.1              lifecycle_1.0.4          
 [29] pkgconfig_2.0.3           rsvd_1.0.5               
 [31] Matrix_1.7-4              R6_2.6.1                 
 [33] fastmap_1.2.0             digest_0.6.39            
 [35] RSpectra_0.16-2           dqrng_0.4.1              
 [37] irlba_2.3.5.1             ExperimentHub_3.1.0      
 [39] RSQLite_2.4.5             beachmat_2.27.0          
 [41] labeling_0.4.3            filelock_1.0.3           
 [43] httr_1.4.7                abind_1.4-8              
 [45] compiler_4.6.0            bit64_4.6.0-1            
 [47] withr_3.0.2               S7_0.2.1                 
 [49] BiocParallel_1.45.0       viridis_0.6.5            
 [51] DBI_1.2.3                 HDF5Array_1.39.0         
 [53] R.utils_2.13.0            rappdirs_0.3.3           
 [55] DelayedArray_0.37.0       bluster_1.21.0           
 [57] rjson_0.2.23              tools_4.6.0              
 [59] vipor_0.4.7               otel_0.2.0               
 [61] beeswarm_0.4.0            R.oo_1.27.1              
 [63] glue_1.8.0                h5mread_1.3.1            
 [65] restfulr_0.0.16           rhdf5filters_1.23.3      
 [67] grid_4.6.0                Rtsne_0.17               
 [69] cluster_2.1.8.1           gtable_0.3.6             
 [71] R.methodsS3_1.8.2         metapod_1.19.1           
 [73] BiocSingular_1.27.1       ScaledMatrix_1.19.0      
 [75] XVector_0.51.0            ggrepel_0.9.6            
 [77] BiocVersion_3.23.1        pillar_1.11.1            
 [79] limma_3.67.0              dplyr_1.1.4              
 [81] BiocFileCache_3.1.0       lattice_0.22-7           
 [83] FNN_1.1.4.1               rtracklayer_1.71.2       
 [85] bit_4.6.0                 tidyselect_1.2.1         
 [87] locfit_1.5-9.12           Biostrings_2.79.2        
 [89] knitr_1.50                gridExtra_2.3            
 [91] bookdown_0.46             ProtGenerics_1.43.0      
 [93] edgeR_4.9.1               xfun_0.54                
 [95] statmod_1.5.1             UCSC.utils_1.7.1         
 [97] lazyeval_0.2.2            yaml_2.3.12              
 [99] cigarillo_1.1.0           evaluate_1.0.5           
[101] codetools_0.2-20          tibble_3.3.0             
[103] BiocManager_1.30.27       graph_1.89.1             
[105] cli_3.6.5                 uwot_0.2.4               
[107] jquerylib_0.1.4           GenomeInfoDb_1.47.2      
[109] dichromat_2.0-0.1         Rcpp_1.1.0.8.1           
[111] dir.expiry_1.19.0         dbplyr_2.5.1             
[113] png_0.1-8                 XML_3.99-0.20            
[115] parallel_4.6.0            blob_1.2.4               
[117] sparseMatrixStats_1.23.0  bitops_1.0-9             
[119] viridisLite_0.4.2         scales_1.4.0             
[121] purrr_1.2.0               crayon_1.5.3             
[123] rlang_1.1.6               cowplot_1.2.0            
[125] KEGGREST_1.51.1          
```
</div>
