# Lawlor human pancreas (SMARTer)

<script>
document.addEventListener("click", function (event) {
    if (event.target.classList.contains("rebook-collapse")) {
        event.target.classList.toggle("active");
        var content = event.target.nextElementSibling;
        if (content.style.display === "block") {
            content.style.display = "none";
        } else {
            content.style.display = "block";
        }
    }
})
</script>

<style>
.rebook-collapse {
  background-color: #eee;
  color: #444;
  cursor: pointer;
  padding: 18px;
  width: 100%;
  border: none;
  text-align: left;
  outline: none;
  font-size: 15px;
}

.rebook-content {
  padding: 0 18px;
  display: none;
  overflow: hidden;
  background-color: #f1f1f1;
}
</style>

## Introduction

This performs an analysis of the @lawlor2017singlecell dataset,
consisting of human pancreas cells from various donors.

## Data loading


``` r
library(scRNAseq)
sce.lawlor <- LawlorPancreasData()
```


``` r
library(AnnotationHub)
edb <- AnnotationHub()[["AH73881"]]
anno <- select(edb, keys=rownames(sce.lawlor), keytype="GENEID", 
    columns=c("SYMBOL", "SEQNAME"))
rowData(sce.lawlor) <- anno[match(rownames(sce.lawlor), anno[,1]),-1]
```

## Quality control


``` r
unfiltered <- sce.lawlor
```


``` r
library(scater)
stats <- perCellQCMetrics(sce.lawlor, 
    subsets=list(Mito=which(rowData(sce.lawlor)$SEQNAME=="MT")))
qc <- quickPerCellQC(stats, percent_subsets="subsets_Mito_percent",
    batch=sce.lawlor$`islet unos id`)
sce.lawlor <- sce.lawlor[,!qc$discard]
```


``` r
colData(unfiltered) <- cbind(colData(unfiltered), stats)
unfiltered$discard <- qc$discard

gridExtra::grid.arrange(
    plotColData(unfiltered, x="islet unos id", y="sum", colour_by="discard") +
        scale_y_log10() + ggtitle("Total count") +
        theme(axis.text.x = element_text(angle = 90)),
    plotColData(unfiltered, x="islet unos id", y="detected", 
        colour_by="discard") + scale_y_log10() + ggtitle("Detected features") +
        theme(axis.text.x = element_text(angle = 90)), 
    plotColData(unfiltered, x="islet unos id", y="subsets_Mito_percent",
        colour_by="discard") + ggtitle("Mito percent") +
        theme(axis.text.x = element_text(angle = 90)),
    ncol=2
)
```

<div class="figure">
<img src="lawlor-pancreas_files/figure-html/unref-lawlor-qc-dist-1.png" alt="Distribution of each QC metric across cells from each donor of the Lawlor pancreas dataset. Each point represents a cell and is colored according to whether that cell was discarded." width="960" />
<p class="caption">(\#fig:unref-lawlor-qc-dist)Distribution of each QC metric across cells from each donor of the Lawlor pancreas dataset. Each point represents a cell and is colored according to whether that cell was discarded.</p>
</div>


``` r
plotColData(unfiltered, x="sum", y="subsets_Mito_percent",
    colour_by="discard") + scale_x_log10()
```

<div class="figure">
<img src="lawlor-pancreas_files/figure-html/unref-lawlor-qc-comp-1.png" alt="Percentage of mitochondrial reads in each cell in the 416B dataset compared to the total count. Each point represents a cell and is colored according to whether that cell was discarded." width="672" />
<p class="caption">(\#fig:unref-lawlor-qc-comp)Percentage of mitochondrial reads in each cell in the 416B dataset compared to the total count. Each point represents a cell and is colored according to whether that cell was discarded.</p>
</div>


``` r
colSums(as.matrix(qc))
```

```
##              low_lib_size            low_n_features high_subsets_Mito_percent 
##                         9                         5                        25 
##                   discard 
##                        34
```

## Normalization


``` r
library(scran)
set.seed(1000)
clusters <- quickCluster(sce.lawlor)
sce.lawlor <- computeSumFactors(sce.lawlor, clusters=clusters)
sce.lawlor <- logNormCounts(sce.lawlor)
```


``` r
summary(sizeFactors(sce.lawlor))
```

```
##    Min. 1st Qu.  Median    Mean 3rd Qu.    Max. 
##   0.295   0.781   0.963   1.000   1.182   2.629
```


``` r
plot(librarySizeFactors(sce.lawlor), sizeFactors(sce.lawlor), pch=16,
    xlab="Library size factors", ylab="Deconvolution factors", log="xy")
```

<div class="figure">
<img src="lawlor-pancreas_files/figure-html/unref-lawlor-norm-1.png" alt="Relationship between the library size factors and the deconvolution size factors in the Lawlor pancreas dataset." width="672" />
<p class="caption">(\#fig:unref-lawlor-norm)Relationship between the library size factors and the deconvolution size factors in the Lawlor pancreas dataset.</p>
</div>

## Variance modelling

Using age as a proxy for the donor.


``` r
dec.lawlor <- modelGeneVar(sce.lawlor, block=sce.lawlor$`islet unos id`)
chosen.genes <- getTopHVGs(dec.lawlor, n=2000)
```


``` r
par(mfrow=c(4,2))
blocked.stats <- dec.lawlor$per.block
for (i in colnames(blocked.stats)) {
    current <- blocked.stats[[i]]
    plot(current$mean, current$total, main=i, pch=16, cex=0.5,
        xlab="Mean of log-expression", ylab="Variance of log-expression")
    curfit <- metadata(current)
    curve(curfit$trend(x), col='dodgerblue', add=TRUE, lwd=2)
}
```

<div class="figure">
<img src="lawlor-pancreas_files/figure-html/unnamed-chunk-4-1.png" alt="Per-gene variance as a function of the mean for the log-expression values in the Lawlor pancreas dataset. Each point represents a gene (black) with the mean-variance trend (blue) fitted separately for each donor." width="672" />
<p class="caption">(\#fig:unnamed-chunk-4)Per-gene variance as a function of the mean for the log-expression values in the Lawlor pancreas dataset. Each point represents a gene (black) with the mean-variance trend (blue) fitted separately for each donor.</p>
</div>

## Dimensionality reduction


``` r
library(BiocSingular)
set.seed(101011001)
sce.lawlor <- runPCA(sce.lawlor, subset_row=chosen.genes, ncomponents=25)
sce.lawlor <- runTSNE(sce.lawlor, dimred="PCA")
```

## Clustering


``` r
snn.gr <- buildSNNGraph(sce.lawlor, use.dimred="PCA")
colLabels(sce.lawlor) <- factor(igraph::cluster_walktrap(snn.gr)$membership)
```


``` r
table(colLabels(sce.lawlor), sce.lawlor$`cell type`)
```

```
##    
##     Acinar Alpha Beta Delta Ductal Gamma/PP None/Other Stellate
##   1      1     0    1    13      2       16          2        0
##   2      0     0   75     1      0        0          0        0
##   3      0   161    1     0      0        1          2        0
##   4      0     1    0     1      0        0          5       19
##   5     22     0    0     0      0        0          0        0
##   6      0     0  174     4      1        0          1        0
##   7      0    76    1     0      0        0          0        0
##   8      0     0    0     1     20        0          2        0
```


``` r
table(colLabels(sce.lawlor), sce.lawlor$`islet unos id`)
```

```
##    
##     ACCG268 ACCR015A ACEK420A ACEL337 ACHY057 ACIB065 ACIW009 ACJV399
##   1       8        2        2       4       4       4       9       2
##   2      13        3        2      33       3       2       4      16
##   3      36       23       14      13      14      14      21      30
##   4       7        1        0       1       0       4       9       4
##   5       0        2       13       0       0       0       5       2
##   6      34       10        4      39       7      23      24      39
##   7      33       12        0       5       6       7       4      10
##   8       1        1        2       1       2       1      12       3
```


``` r
gridExtra::grid.arrange(
    plotTSNE(sce.lawlor, colour_by="label"),
    plotTSNE(sce.lawlor, colour_by="islet unos id"),
    ncol=2
)
```

<div class="figure">
<img src="lawlor-pancreas_files/figure-html/unref-grun-tsne-1.png" alt="Obligatory $t$-SNE plots of the Lawlor pancreas dataset. Each point represents a cell that is colored by cluster (left) or batch (right)." width="672" />
<p class="caption">(\#fig:unref-grun-tsne)Obligatory $t$-SNE plots of the Lawlor pancreas dataset. Each point represents a cell that is colored by cluster (left) or batch (right).</p>
</div>

## Session Info {-}

<button class="rebook-collapse">View session info</button>
<div class="rebook-content">
```
R Under development (unstable) (2025-10-20 r88955)
Platform: x86_64-pc-linux-gnu
Running under: Ubuntu 24.04.3 LTS

Matrix products: default
BLAS:   /home/biocbuild/bbs-3.23-bioc/R/lib/libRblas.so 
LAPACK: /usr/lib/x86_64-linux-gnu/lapack/liblapack.so.3.12.0  LAPACK version 3.12.0

locale:
 [1] LC_CTYPE=en_US.UTF-8       LC_NUMERIC=C              
 [3] LC_TIME=en_GB              LC_COLLATE=C              
 [5] LC_MONETARY=en_US.UTF-8    LC_MESSAGES=en_US.UTF-8   
 [7] LC_PAPER=en_US.UTF-8       LC_NAME=C                 
 [9] LC_ADDRESS=C               LC_TELEPHONE=C            
[11] LC_MEASUREMENT=en_US.UTF-8 LC_IDENTIFICATION=C       

time zone: America/New_York
tzcode source: system (glibc)

attached base packages:
[1] stats4    stats     graphics  grDevices utils     datasets  methods  
[8] base     

other attached packages:
 [1] BiocSingular_1.27.1         scran_1.39.0               
 [3] scater_1.39.0               ggplot2_4.0.1              
 [5] scuttle_1.21.0              ensembldb_2.35.0           
 [7] AnnotationFilter_1.35.0     GenomicFeatures_1.63.1     
 [9] AnnotationDbi_1.73.0        AnnotationHub_4.1.0        
[11] BiocFileCache_3.1.0         dbplyr_2.5.1               
[13] scRNAseq_2.25.0             SingleCellExperiment_1.33.0
[15] SummarizedExperiment_1.41.0 Biobase_2.71.0             
[17] GenomicRanges_1.63.1        Seqinfo_1.1.0              
[19] IRanges_2.45.0              S4Vectors_0.49.0           
[21] BiocGenerics_0.57.0         generics_0.1.4             
[23] MatrixGenerics_1.23.0       matrixStats_1.5.0          
[25] BiocStyle_2.39.0            rebook_1.21.0              

loaded via a namespace (and not attached):
  [1] RColorBrewer_1.1-3       jsonlite_2.0.0           CodeDepends_0.6.6       
  [4] magrittr_2.0.4           ggbeeswarm_0.7.3         gypsum_1.7.0            
  [7] farver_2.1.2             rmarkdown_2.30           BiocIO_1.21.0           
 [10] vctrs_0.6.5              memoise_2.0.1            Rsamtools_2.27.0        
 [13] RCurl_1.98-1.17          htmltools_0.5.9          S4Arrays_1.11.1         
 [16] curl_7.0.0               BiocNeighbors_2.5.0      Rhdf5lib_1.33.0         
 [19] SparseArray_1.11.9       rhdf5_2.55.12            sass_0.4.10             
 [22] alabaster.base_1.11.1    bslib_0.9.0              alabaster.sce_1.11.0    
 [25] httr2_1.2.2              cachem_1.1.0             GenomicAlignments_1.47.0
 [28] igraph_2.2.1             lifecycle_1.0.4          pkgconfig_2.0.3         
 [31] rsvd_1.0.5               Matrix_1.7-4             R6_2.6.1                
 [34] fastmap_1.2.0            digest_0.6.39            dqrng_0.4.1             
 [37] irlba_2.3.5.1            ExperimentHub_3.1.0      RSQLite_2.4.5           
 [40] beachmat_2.27.0          labeling_0.4.3           filelock_1.0.3          
 [43] httr_1.4.7               abind_1.4-8              compiler_4.6.0          
 [46] bit64_4.6.0-1            withr_3.0.2              S7_0.2.1                
 [49] BiocParallel_1.45.0      viridis_0.6.5            DBI_1.2.3               
 [52] HDF5Array_1.39.0         alabaster.ranges_1.11.0  alabaster.schemas_1.11.0
 [55] rappdirs_0.3.3           DelayedArray_0.37.0      bluster_1.21.0          
 [58] rjson_0.2.23             tools_4.6.0              vipor_0.4.7             
 [61] otel_0.2.0               beeswarm_0.4.0           glue_1.8.0              
 [64] h5mread_1.3.1            restfulr_0.0.16          rhdf5filters_1.23.3     
 [67] grid_4.6.0               Rtsne_0.17               cluster_2.1.8.1         
 [70] gtable_0.3.6             metapod_1.19.1           ScaledMatrix_1.19.0     
 [73] XVector_0.51.0           ggrepel_0.9.6            BiocVersion_3.23.1      
 [76] pillar_1.11.1            limma_3.67.0             dplyr_1.1.4             
 [79] lattice_0.22-7           rtracklayer_1.71.2       bit_4.6.0               
 [82] tidyselect_1.2.1         locfit_1.5-9.12          Biostrings_2.79.2       
 [85] knitr_1.50               gridExtra_2.3            bookdown_0.46           
 [88] ProtGenerics_1.43.0      edgeR_4.9.1              xfun_0.54               
 [91] statmod_1.5.1            UCSC.utils_1.7.1         lazyeval_0.2.2          
 [94] yaml_2.3.12              evaluate_1.0.5           codetools_0.2-20        
 [97] cigarillo_1.1.0          tibble_3.3.0             alabaster.matrix_1.11.0 
[100] BiocManager_1.30.27      graph_1.89.1             cli_3.6.5               
[103] jquerylib_0.1.4          dichromat_2.0-0.1        Rcpp_1.1.0.8.1          
[106] GenomeInfoDb_1.47.2      dir.expiry_1.19.0        png_0.1-8               
[109] XML_3.99-0.20            parallel_4.6.0           blob_1.2.4              
[112] bitops_1.0-9             viridisLite_0.4.2        alabaster.se_1.11.0     
[115] scales_1.4.0             purrr_1.2.0              crayon_1.5.3            
[118] rlang_1.1.6              cowplot_1.2.0            KEGGREST_1.51.1         
```
</div>
