# Bach mouse mammary gland (10X Genomics)

<script>
document.addEventListener("click", function (event) {
    if (event.target.classList.contains("rebook-collapse")) {
        event.target.classList.toggle("active");
        var content = event.target.nextElementSibling;
        if (content.style.display === "block") {
            content.style.display = "none";
        } else {
            content.style.display = "block";
        }
    }
})
</script>

<style>
.rebook-collapse {
  background-color: #eee;
  color: #444;
  cursor: pointer;
  padding: 18px;
  width: 100%;
  border: none;
  text-align: left;
  outline: none;
  font-size: 15px;
}

.rebook-content {
  padding: 0 18px;
  display: none;
  overflow: hidden;
  background-color: #f1f1f1;
}
</style>

## Introduction

This performs an analysis of the @bach2017differentiation 10X Genomics dataset,
from which we will consider a single sample of epithelial cells from the mouse mammary gland during gestation.

## Data loading


``` r
library(scRNAseq)
sce.mam <- BachMammaryData(samples="G_1")
```


``` r
library(scater)
rownames(sce.mam) <- uniquifyFeatureNames(
    rowData(sce.mam)$Ensembl, rowData(sce.mam)$Symbol)

library(AnnotationHub)
ens.mm.v97 <- AnnotationHub()[["AH73905"]]
rowData(sce.mam)$SEQNAME <- mapIds(ens.mm.v97, keys=rowData(sce.mam)$Ensembl,
    keytype="GENEID", column="SEQNAME")
```

## Quality control


``` r
unfiltered <- sce.mam
```


``` r
is.mito <- rowData(sce.mam)$SEQNAME == "MT"
stats <- perCellQCMetrics(sce.mam, subsets=list(Mito=which(is.mito)))
qc <- quickPerCellQC(stats, percent_subsets="subsets_Mito_percent")
sce.mam <- sce.mam[,!qc$discard]
```


``` r
colData(unfiltered) <- cbind(colData(unfiltered), stats)
unfiltered$discard <- qc$discard

gridExtra::grid.arrange(
    plotColData(unfiltered, y="sum", colour_by="discard") + 
        scale_y_log10() + ggtitle("Total count"),
    plotColData(unfiltered, y="detected", colour_by="discard") + 
        scale_y_log10() + ggtitle("Detected features"),
    plotColData(unfiltered, y="subsets_Mito_percent", 
        colour_by="discard") + ggtitle("Mito percent"),
    ncol=2
)
```

<div class="figure">
<img src="bach-mammary_files/figure-html/unref-bach-qc-dist-1.png" alt="Distribution of each QC metric across cells in the Bach mammary gland dataset. Each point represents a cell and is colored according to whether that cell was discarded." width="672" />
<p class="caption">(\#fig:unref-bach-qc-dist)Distribution of each QC metric across cells in the Bach mammary gland dataset. Each point represents a cell and is colored according to whether that cell was discarded.</p>
</div>


``` r
plotColData(unfiltered, x="sum", y="subsets_Mito_percent", 
    colour_by="discard") + scale_x_log10()
```

<div class="figure">
<img src="bach-mammary_files/figure-html/unref-bach-qc-comp-1.png" alt="Percentage of mitochondrial reads in each cell in the Bach mammary gland dataset compared to its total count. Each point represents a cell and is colored according to whether that cell was discarded." width="672" />
<p class="caption">(\#fig:unref-bach-qc-comp)Percentage of mitochondrial reads in each cell in the Bach mammary gland dataset compared to its total count. Each point represents a cell and is colored according to whether that cell was discarded.</p>
</div>


``` r
colSums(as.matrix(qc))
```

```
##              low_lib_size            low_n_features high_subsets_Mito_percent 
##                         0                         0                       143 
##                   discard 
##                       143
```

## Normalization 


``` r
library(scran)
set.seed(101000110)
clusters <- quickCluster(sce.mam)
sce.mam <- computeSumFactors(sce.mam, clusters=clusters)
sce.mam <- logNormCounts(sce.mam)
```


``` r
summary(sizeFactors(sce.mam))
```

```
##    Min. 1st Qu.  Median    Mean 3rd Qu.    Max. 
##   0.264   0.520   0.752   1.000   1.207  10.790
```


``` r
plot(librarySizeFactors(sce.mam), sizeFactors(sce.mam), pch=16,
    xlab="Library size factors", ylab="Deconvolution factors", log="xy")
```

<div class="figure">
<img src="bach-mammary_files/figure-html/unref-bach-norm-1.png" alt="Relationship between the library size factors and the deconvolution size factors in the Bach mammary gland dataset." width="672" />
<p class="caption">(\#fig:unref-bach-norm)Relationship between the library size factors and the deconvolution size factors in the Bach mammary gland dataset.</p>
</div>

## Variance modelling

We use a Poisson-based technical trend to capture more genuine biological variation in the biological component.


``` r
set.seed(00010101)
dec.mam <- modelGeneVarByPoisson(sce.mam)
top.mam <- getTopHVGs(dec.mam, prop=0.1)
```


``` r
plot(dec.mam$mean, dec.mam$total, pch=16, cex=0.5,
    xlab="Mean of log-expression", ylab="Variance of log-expression")
curfit <- metadata(dec.mam)
curve(curfit$trend(x), col='dodgerblue', add=TRUE, lwd=2)
```

<div class="figure">
<img src="bach-mammary_files/figure-html/unref-bach-var-1.png" alt="Per-gene variance as a function of the mean for the log-expression values in the Bach mammary gland dataset. Each point represents a gene (black) with the mean-variance trend (blue) fitted to simulated Poisson counts." width="672" />
<p class="caption">(\#fig:unref-bach-var)Per-gene variance as a function of the mean for the log-expression values in the Bach mammary gland dataset. Each point represents a gene (black) with the mean-variance trend (blue) fitted to simulated Poisson counts.</p>
</div>

## Dimensionality reduction


``` r
library(BiocSingular)
set.seed(101010011)
sce.mam <- denoisePCA(sce.mam, technical=dec.mam, subset.row=top.mam)
sce.mam <- runTSNE(sce.mam, dimred="PCA")
```


``` r
ncol(reducedDim(sce.mam, "PCA"))
```

```
## [1] 15
```

## Clustering 

We use a higher `k` to obtain coarser clusters (for use in `doubletCluster()` later).


``` r
snn.gr <- buildSNNGraph(sce.mam, use.dimred="PCA", k=25)
colLabels(sce.mam) <- factor(igraph::cluster_walktrap(snn.gr)$membership)
```


``` r
table(colLabels(sce.mam))
```

```
## 
##   1   2   3   4   5   6   7   8   9  10 
## 550 847 639 477  54  88  39  22  32  24
```


``` r
plotTSNE(sce.mam, colour_by="label")
```

<div class="figure">
<img src="bach-mammary_files/figure-html/unref-bach-tsne-1.png" alt="Obligatory $t$-SNE plot of the Bach mammary gland dataset, where each point represents a cell and is colored according to the assigned cluster." width="672" />
<p class="caption">(\#fig:unref-bach-tsne)Obligatory $t$-SNE plot of the Bach mammary gland dataset, where each point represents a cell and is colored according to the assigned cluster.</p>
</div>

## Session Info {-}

<button class="rebook-collapse">View session info</button>
<div class="rebook-content">
```
R Under development (unstable) (2025-10-20 r88955)
Platform: x86_64-pc-linux-gnu
Running under: Ubuntu 24.04.3 LTS

Matrix products: default
BLAS:   /home/biocbuild/bbs-3.23-bioc/R/lib/libRblas.so 
LAPACK: /usr/lib/x86_64-linux-gnu/lapack/liblapack.so.3.12.0  LAPACK version 3.12.0

locale:
 [1] LC_CTYPE=en_US.UTF-8       LC_NUMERIC=C              
 [3] LC_TIME=en_GB              LC_COLLATE=C              
 [5] LC_MONETARY=en_US.UTF-8    LC_MESSAGES=en_US.UTF-8   
 [7] LC_PAPER=en_US.UTF-8       LC_NAME=C                 
 [9] LC_ADDRESS=C               LC_TELEPHONE=C            
[11] LC_MEASUREMENT=en_US.UTF-8 LC_IDENTIFICATION=C       

time zone: America/New_York
tzcode source: system (glibc)

attached base packages:
[1] stats4    stats     graphics  grDevices utils     datasets  methods  
[8] base     

other attached packages:
 [1] BiocSingular_1.27.1         scran_1.39.0               
 [3] AnnotationHub_4.1.0         BiocFileCache_3.1.0        
 [5] dbplyr_2.5.1                scater_1.39.0              
 [7] ggplot2_4.0.1               scuttle_1.21.0             
 [9] ensembldb_2.35.0            AnnotationFilter_1.35.0    
[11] GenomicFeatures_1.63.1      AnnotationDbi_1.73.0       
[13] scRNAseq_2.25.0             SingleCellExperiment_1.33.0
[15] SummarizedExperiment_1.41.0 Biobase_2.71.0             
[17] GenomicRanges_1.63.1        Seqinfo_1.1.0              
[19] IRanges_2.45.0              S4Vectors_0.49.0           
[21] BiocGenerics_0.57.0         generics_0.1.4             
[23] MatrixGenerics_1.23.0       matrixStats_1.5.0          
[25] BiocStyle_2.39.0            rebook_1.21.0              

loaded via a namespace (and not attached):
  [1] RColorBrewer_1.1-3       jsonlite_2.0.0           CodeDepends_0.6.6       
  [4] magrittr_2.0.4           ggbeeswarm_0.7.3         gypsum_1.7.0            
  [7] farver_2.1.2             rmarkdown_2.30           BiocIO_1.21.0           
 [10] vctrs_0.6.5              memoise_2.0.1            Rsamtools_2.27.0        
 [13] RCurl_1.98-1.17          htmltools_0.5.9          S4Arrays_1.11.1         
 [16] curl_7.0.0               BiocNeighbors_2.5.0      Rhdf5lib_1.33.0         
 [19] SparseArray_1.11.9       rhdf5_2.55.12            sass_0.4.10             
 [22] alabaster.base_1.11.1    bslib_0.9.0              alabaster.sce_1.11.0    
 [25] httr2_1.2.2              cachem_1.1.0             GenomicAlignments_1.47.0
 [28] igraph_2.2.1             lifecycle_1.0.4          pkgconfig_2.0.3         
 [31] rsvd_1.0.5               Matrix_1.7-4             R6_2.6.1                
 [34] fastmap_1.2.0            digest_0.6.39            dqrng_0.4.1             
 [37] irlba_2.3.5.1            ExperimentHub_3.1.0      RSQLite_2.4.5           
 [40] beachmat_2.27.0          labeling_0.4.3           filelock_1.0.3          
 [43] httr_1.4.7               abind_1.4-8              compiler_4.6.0          
 [46] bit64_4.6.0-1            withr_3.0.2              S7_0.2.1                
 [49] BiocParallel_1.45.0      viridis_0.6.5            DBI_1.2.3               
 [52] HDF5Array_1.39.0         alabaster.ranges_1.11.0  alabaster.schemas_1.11.0
 [55] rappdirs_0.3.3           DelayedArray_0.37.0      bluster_1.21.0          
 [58] rjson_0.2.23             tools_4.6.0              vipor_0.4.7             
 [61] otel_0.2.0               beeswarm_0.4.0           glue_1.8.0              
 [64] h5mread_1.3.1            restfulr_0.0.16          rhdf5filters_1.23.3     
 [67] grid_4.6.0               Rtsne_0.17               cluster_2.1.8.1         
 [70] gtable_0.3.6             metapod_1.19.1           ScaledMatrix_1.19.0     
 [73] XVector_0.51.0           ggrepel_0.9.6            BiocVersion_3.23.1      
 [76] pillar_1.11.1            limma_3.67.0             dplyr_1.1.4             
 [79] lattice_0.22-7           rtracklayer_1.71.2       bit_4.6.0               
 [82] tidyselect_1.2.1         locfit_1.5-9.12          Biostrings_2.79.2       
 [85] knitr_1.50               gridExtra_2.3            bookdown_0.46           
 [88] ProtGenerics_1.43.0      edgeR_4.9.1              xfun_0.54               
 [91] statmod_1.5.1            UCSC.utils_1.7.1         lazyeval_0.2.2          
 [94] yaml_2.3.12              evaluate_1.0.5           codetools_0.2-20        
 [97] cigarillo_1.1.0          tibble_3.3.0             alabaster.matrix_1.11.0 
[100] BiocManager_1.30.27      graph_1.89.1             cli_3.6.5               
[103] jquerylib_0.1.4          dichromat_2.0-0.1        Rcpp_1.1.0.8.1          
[106] GenomeInfoDb_1.47.2      dir.expiry_1.19.0        png_0.1-8               
[109] XML_3.99-0.20            parallel_4.6.0           blob_1.2.4              
[112] bitops_1.0-9             viridisLite_0.4.2        alabaster.se_1.11.0     
[115] scales_1.4.0             purrr_1.2.0              crayon_1.5.3            
[118] rlang_1.1.6              cowplot_1.2.0            KEGGREST_1.51.1         
```
</div>
