% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils.R
\name{tof_spade_density}
\alias{tof_spade_density}
\title{Estimate cells' local densities as done in Spanning-tree Progression Analysis of Density-normalized Events (SPADE)}
\usage{
tof_spade_density(
  tof_tibble,
  distance_cols = where(tof_is_numeric),
  distance_function = c("euclidean", "cosine", "l2", "ip"),
  num_alpha_cells = 2000L,
  alpha_multiplier = 5,
  max_neighbors = round(0.01 * nrow(tof_tibble)),
  normalize = TRUE,
  ...
)
}
\arguments{
\item{tof_tibble}{A `tof_tbl` or a `tibble`.}

\item{distance_cols}{Unquoted names of the columns in `tof_tibble` to use in
calculating cell-to-cell distances during the local density estimation for
each cell. Defaults to all numeric columns in `tof_tibble`.}

\item{distance_function}{A string indicating which distance function to use
for calculating cell-to-cell distances during local density estimation. Options
include "euclidean" (the default) and "cosine".}

\item{num_alpha_cells}{An integer indicating how many cells from `tof_tibble`
should be randomly sampled from `tof_tibble` in order to estimate `alpha`, the
radius of the sphere constructed around each cell during local density
estimation. Alpha is calculated by taking the median nearest-neighbor distance
from the `num_alpha_cells` randomly-sampled cells and multiplying it by
`alpha_multiplier`. Defaults to 2000.}

\item{alpha_multiplier}{An numeric value indicating the multiplier that should be used
when calculating `alpha`, the radius of the sphere constructed around each
cell during local density estimation. Alpha is calculated by taking
the median nearest-neighbor distance from the `num_alpha_cells` cells
randomly-sampled from `tof_tibble` and multiplying it by
`alpha_multiplier`. Defaults to 5.}

\item{max_neighbors}{An integer indicating the maximum number of neighbors
that can be counted within the sphere surrounding any given cell. Implemented
to reduce the density estimation procedure's speed and memory requirements.
Defaults to 1\% of the number of rows in `tof_tibble`.}

\item{normalize}{A boolean value indicating if the vector of local density
estimates should be normalized to values between 0 and 1. Defaults to TRUE.}

\item{...}{Additional optional arguments to pass to \code{\link{tof_find_knn}}.}
}
\value{
A tibble with a single column named ".spade_density" containing the
local density estimates for each input cell in `tof_tibble`.
}
\description{
This function uses the algorithm described in
\href{https://pubmed.ncbi.nlm.nih.gov/21964415/}{Qiu et al., (2011)} to estimate
the local density of each cell in a `tof_tbl` or `tibble` containing high-dimensional cytometry data.
Briefly, this algorithm involves counting the number of neighboring cells
within  a sphere of radius alpha surrounding each cell. Here, we do so using
the \code{\link[RANN]{nn2}} function.
}
\examples{
sim_data <-
    dplyr::tibble(
        cd45 = rnorm(n = 1000),
        cd38 = rnorm(n = 1000),
        cd34 = rnorm(n = 1000),
        cd19 = rnorm(n = 1000)
    )

# perform the density estimation
tof_spade_density(tof_tibble = sim_data)

# perform the density estimation using cosine distance
tof_spade_density(
    tof_tibble = sim_data,
    distance_function = "cosine",
    alpha_multiplier = 2
)

# perform the density estimation with a smaller search radius around
# each cell
tof_spade_density(
    tof_tibble = sim_data,
    alpha_multiplier = 2
)

}
\seealso{
Other local density estimation functions: 
\code{\link{tof_estimate_density}()},
\code{\link{tof_knn_density}()}
}
\concept{local density estimation functions}
