% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spatial_enrich.R
\name{SpatialEnrichment}
\alias{SpatialEnrichment}
\alias{sf_var}
\alias{spatial_enrich}
\alias{query_enrich}
\alias{ovl_enrich}
\alias{graph_line}
\title{Identifying spatially enriched or depleted biomolecules}
\usage{
sf_var(
  data,
  feature,
  ft.sel = NULL,
  variable = NULL,
  var.sel = NULL,
  com.by = "ft"
)

spatial_enrich(
  data,
  method = c("edgeR"),
  norm = "TMM",
  m.array = FALSE,
  pairwise = FALSE,
  log2.fc = 1,
  p.adjust = "BH",
  fdr = 0.05,
  outliers = 0,
  aggr = "mean",
  log2.trans = TRUE,
  verbose = TRUE
)

query_enrich(res, query, other = FALSE, data.rep = FALSE)

ovl_enrich(
  res,
  type = "up",
  plot = "matrix",
  order.by = "freq",
  nintersects = 40,
  point.size = 3,
  line.size = 1,
  mb.ratio = c(0.6, 0.4),
  text.scale = 1.5,
  upset.arg = list(),
  show.plot = TRUE,
  venn.arg = list(),
  axis.agl = 45,
  font.size = 5,
  cols = c("lightcyan3", "darkorange")
)

graph_line(
  data,
  scale = "none",
  x.title = "Samples",
  y.title = "Assay values",
  linewidth = 1,
  text.size = 15,
  text.angle = 60,
  lgd.pos = "right",
  lgd.guide = guides(color = guide_legend(nrow = 1, byrow = TRUE, title = NULL))
)
}
\arguments{
\item{data}{\describe{
 \item{\code{sf_var}}{
  A \code{SummarizedExperiment} object. The \code{colData} slot is required to contain at least two columns of spatial features and experiment variables respectively.   
 }
 \item{\code{spatial_enrich}}{
 A \code{SummarizedExperiment} object returned by \code{sf_var}.
 } 
 \item{\code{graph_line}}{
 A \code{data.frame}, where rows are biomolecules and columns are spatial features.
 } 
}}

\item{feature}{The column name in the \code{colData} slot of \code{SummarizedExperiment} that contains spatial features.}

\item{ft.sel}{A vector of spatial features to choose.}

\item{variable}{The column name in the \code{colData} slot of \code{SummarizedExperiment} that contains experiment variables.}

\item{var.sel}{A vector of variables to choose.}

\item{com.by}{One of \code{ft}, \code{var}, or \code{ft.var}. If \code{ft}, the enrichment is performed for each spatial feature and the variables are treated as replicates. If \code{var} the enrichment is performed for each variable and spatial features are treated as replicates. If \code{ft.var}, spatial features (tissue1, tissue2) and variables (var1, var2) are combined such as tissue1__var1, tissue1_var2, tissue2__var1, tissue2_var2. The enrichment is performed for each combination.}

\item{method}{One of \code{edgeR}, \code{limma}, and \code{DESeq2}.}

\item{norm}{The normalization method (\code{TMM}, \code{RLE}, \code{upperquartile}, \code{none}) in edgeR. The default is \code{TMM}. Details: https://www.rdocumentation.org/packages/edgeR/versions/3.14.0/topics/calcNormFactors.}

\item{m.array}{Logical. `TRUE` and `FALSE` indicate the input are microarray and count data respectively.}

\item{pairwise}{Logical. If `TRUE`, pairwise comparisons will be performed starting dispersion estimation. If `FALSE` (default), all samples are fitted into a GLM model together, then pairwise comparisons are performed through contrasts.}

\item{log2.fc}{The log2-fold change cutoff. The default is 1.}

\item{p.adjust}{The method (\code{holm}, \code{hochberg}, \code{hommel}, \code{bonferroni}, \code{BH}, \code{BY}, \code{fdr}, or \code{none}) for adjusting p values in multiple hypothesis testing. The default is \code{BH}.}

\item{fdr}{The FDR cutoff. The default is 0.05.}

\item{outliers}{The number of outliers allowed in the references. If there are too many references, there might be no enriched/depleted biomolecules in the query feature. To avoid this, set a certain number of outliers.}

\item{aggr}{One of \code{mean} (default) or \code{median}. The method to aggregated replicates in the assay data.}

\item{log2.trans}{Logical. If \code{TRUE} (default), the aggregated data (see \code{aggr}) is transformed to log2-scale and will be further used for plotting SHMs.}

\item{verbose}{Logical. If `TRUE` (default), intermmediate messages will be printed.}

\item{res}{Enrichment results returned by \code{spatial_enrich}.}

\item{query}{A spatial feature for query.}

\item{other}{Logical (default is `FALSE`). If `TRUE` other genes that are neither enriched or depleted will also be returned.}

\item{data.rep}{Logical. If `TRUE` normalized data before aggregating replicates will be returned. If `FALSE`, normalized data after aggretating replicates will be returned.}

\item{type}{One of \code{up} (default) or \code{down}, which refers to up- or down-regulated biomolecules.}

\item{plot}{One of \code{upset}, \code{matrix}, or \code{venn}, corresponding to upset plot, overlap matrix, or Venn diagram respectively.}

\item{order.by}{How the intersections in the matrix should be ordered by. Options include frequency (entered as "freq"), degree, or both in any order.}

\item{nintersects}{Number of intersections to plot. If set to NA, all intersections will be plotted.}

\item{point.size}{Size of points in matrix plot}

\item{line.size}{The line thickness in overlap matrix.}

\item{mb.ratio}{Ratio between matrix plot and main bar plot (Keep in terms of hundredths)}

\item{text.scale}{Numeric, value to scale the text sizes, applies to all axis labels, tick labels, and numbers above bar plot. Can be a universal scale, or a vector containing individual scales
in the following format: c(intersection size title, intersection size tick labels, set size title, set size tick labels, set names, numbers above bars)}

\item{upset.arg}{A \code{list} of additional arguments passed to \code{\link[UpSetR]{upset}}.}

\item{show.plot}{Logical flag indicating whether the plot should be
    displayed.  If false, simply returns the group count matrix.}

\item{venn.arg}{A \code{list} of additional arguments passed to \code{\link[gplots]{venn}}.}

\item{axis.agl}{The angle of axis text in overlap matrix.}

\item{font.size}{The font size of all text in overlap matrix.}

\item{cols}{A vector of two colors indicating low and high values in the overlap matrix respectively. The default is \code{c("lightcyan3", "darkorange")}.}

\item{scale}{The method to scale the data. If \code{none} (default), no scaling. If \code{row}, each row is scaled independently. If \code{all}, all rows are scaled as a whole.}

\item{x.title, y.title}{The title of X-axis and Y-axis respectively.}

\item{linewidth}{The line width.}

\item{text.size}{The font size of all text.}

\item{text.angle}{The angle of axis text.}

\item{lgd.pos}{The position of legend. The default is \code{right}.}

\item{lgd.guide}{The \code{\link[ggplot2]{guides}} function in \code{ggplot2} for customizing legends.}
}
\value{
\describe{
 \item{`sf_var`}{
    A \code{SummarizedExperiment} object.
   }
 \item{`spatial_enrich`}{
    A \code{list} object.
   }
 \item{`query_enrich`}{
    A \code{SummarizedExperiment} object.
   }
 \item{`ovl_enrich`}{
    An UpSet plot, overlap matrix plot, or Venn diagram.
   }
 \item{`graph_line`}{
    A ggplot.
   }
}
}
\description{
The spatial enrichment (SpEn) is designed to detect spatially enriched or depleted biomolecules (genes, proteins, etc) for chosen spatial features (cellular compartments, tissues, organs, \emph{etc}). It compares each feature with all other reference features. The biomolecules significantly up- or down-regulated in one feature relative to reference features are denoted spatially enriched or depleted respectively. The underlying differential expression analysis methods include edgeR (Robinson et al, 2010), limma (Ritchie et al, 2015), and DESeq2 (Love et al, 2014). By querying a feature of interest from the enrichment results, the enriched or depleted biomolecules will be returned. \cr In addition, the SpEn is also able to identify biomolecules enriched or depleted in experiment vairables in a similar manner.   

`sf_var()` subsets data according to given spatial features and variables.

`spatial_enrich()` detects enriched or depleted biomolecules for each given spatial feature.

`query_enrich()` queries enriched or depleted biomolecules in the enrichment results returned by \code{spatial_enrich} for a chosen spatial feature.

`ovl_enrich()` plots overlap of enrichment results across spatial features in form an upset plot, overlap matrix, or Venn diagram.

`graph_line()` plots expression values of chosen biomolecules in a line graph.
}
\examples{

## In the following examples, the toy data come from an RNA-seq analysis on development of 7
## chicken organs under 9 time points (Cardoso-Moreira et al. 2019). For conveninece, it is
## included in this package. The complete raw count data are downloaded using the R package
## ExpressionAtlas (Keays 2019) with the accession number "E-MTAB-6769".   

library(SummarizedExperiment) 
# Access the count table. 
cnt.chk <- read.table(system.file('extdata/shinyApp/data/count_chicken.txt', package='spatialHeatmap'), header=TRUE, row.names=1,sep='\t')
cnt.chk[1:3, 1:5]
# A targets file describing spatial features and conditions is required for toy data. It should be made
# based on the experiment design, which is accessible through the accession number 
# "E-MTAB-6769" in the R package ExpressionAtlas. An example targets file is included in this
# package and accessed below. 

# Access the example targets file. 
tar.chk <- read.table(system.file('extdata/shinyApp/data/target_chicken.txt', package='spatialHeatmap'), header=TRUE, row.names=1, sep='\t') 
# Every column in count table corresponds with a row in targets file. 
tar.chk[1:5, ]
# Store count data and targets file in "SummarizedExperiment".
se.chk <- SummarizedExperiment(assay=cnt.chk, colData=tar.chk)
# The "rowData" slot can store a data frame of gene metadata, but not required. Only the 
# column named "metadata" will be recognized. 
# Pseudo row metadata.
metadata <- paste0('meta', seq_len(nrow(cnt.chk))); metadata[1:3]
rowData(se.chk) <- DataFrame(metadata=metadata)

# Subset the count data by features (brain, heart, kidney) and variables (day10, day12).
# By setting com.by='ft', the subsequent spatial enrichment will be performed across 
# features with the variables as replicates. 
data.sub <- sf_var(data=se.chk, feature='organism_part', ft.sel=c('brain', 'kidney',
 'heart', 'liver'), variable='age', var.sel=c('day10', 'day35'), com.by='ft')

## As conventions, raw sequencing count data should be normalized and filtered to
## reduce noise. Since normalization will be performed in spatial enrichment, only filtering
## is required.  

# Filter out genes with low counts and low variance. Genes with counts over 5 in
# at least 10\% samples (pOA), and coefficient of variance (CV) between 3.5 and 100 are 
# retained.
data.sub.fil <- filter_data(data=data.sub, sam.factor='organism_part', con.factor='age',
pOA=c(0.1, 5), CV=c(0.7, 100))

# Spatial enrichment for every spatial feature with 1 outlier allowed.  
enr.res <- spatial_enrich(data.sub.fil, method=c('edgeR'), norm='TMM', log2.fc=1, fdr=0.05, outliers=1)
# Overlaps of enriched genes across features.
ovl_enrich(enr.res, type='up', plot='upset')
# Query the results for brain.
en.brain <- query_enrich(enr.res, 'brain')
rowData(en.brain)[1:3, c('type', 'total', 'method')] 

# Read aSVG image into an "SVG" object.
svg.chk <- system.file("extdata/shinyApp/data", "gallus_gallus.svg", 
package="spatialHeatmap")
svg.chk <- read_svg(svg.chk)
# Plot an enrichment SHM.
dat.enrich <- SPHM(svg=svg.chk, bulk=en.brain)
shm(data=dat.enrich, ID=rownames(en.brain)[1], legend.r=1, legend.nrow=7, sub.title.size=10, ncol=2, bar.width=0.09, lay.shm='gene')
# Line graph of gene expression profile.
graph_line(assay(en.brain[1, , drop=FALSE]), lgd.pos='bottom')
}
\references{
Cardoso-Moreira, Margarida, Jean Halbert, Delphine Valloton, Britta Velten, Chunyan Chen, Yi Shao, Angélica Liechti, et al. 2019. “Gene Expression Across Mammalian Organ Development.” Nature 571 (7766): 505–9
Keays, Maria. 2019. ExpressionAtlas: Download Datasets from EMBL-EBI Expression Atlas
Martin Morgan, Valerie Obenchain, Jim Hester and Hervé Pagès (2018). SummarizedExperiment: SummarizedExperiment container. R package version 1.10.1
Robinson MD, McCarthy DJ and Smyth GK (2010). edgeR: a Bioconductor package for differential expression analysis of digital gene expression data. Bioinformatics 26, 139-140
Ritchie, M.E., Phipson, B., Wu, D., Hu, Y., Law, C.W., Shi, W., and Smyth, G.K. (2015). limma powers differential expression analyses for RNA-sequencing and microarray studies. Nucleic Acids Research 43(7), e47.
Love, M.I., Huber, W., Anders, S. Moderated estimation of fold change and dispersion for RNA-seq data with DESeq2 Genome Biology 15(12):550 (2014)
Nils Gehlenborg (2019). UpSetR: A More Scalable Alternative to Venn and Euler Diagrams for Visualizing Intersecting Sets. R package version 1.4.0. https://CRAN.R-project.org/package=UpSetR
H. Wickham. ggplot2: Elegant Graphics for Data Analysis. Springer-Verlag New York, 2016.
Hadley Wickham (2007). Reshaping Data with the reshape Package. Journal of Statistical Software, 21(12), 1-20. URL http://www.jstatsoft.org/v21/i12/.
}
\author{
Jianhai Zhang \email{jzhan067@ucr.edu} \cr Dr. Thomas Girke \email{thomas.girke@ucr.edu}
}
