% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cutoff_adjacency.R
\name{cutoff_adjacency}
\alias{cutoff_adjacency}
\title{Threshold Adjacency Matrices Based on Shuffled Network Quantiles}
\usage{
cutoff_adjacency(
  count_matrices,
  weighted_adjm_list,
  n,
  method = c("GENIE3", "GRNBoost2", "JRF"),
  quantile_threshold = 0.99,
  weight_function = "mean",
  nCores = 1,
  grnboost_modules = NULL,
  debug = FALSE
)
}
\arguments{
\item{count_matrices}{A \linkS4class{MultiAssayExperiment} object
containing expression data from multiple experiments or conditions.}

\item{weighted_adjm_list}{A \linkS4class{SummarizedExperiment} object
containing weighted adjacency matrices (one per experiment) to threshold.}

\item{n}{Integer. Number of shuffled replicates generated per original
expression matrix.}

\item{method}{Character string. One of \code{"GENIE3"}, \code{"GRNBoost2"},
or \code{"JRF"}.}

\item{quantile_threshold}{Numeric. The quantile used to define the cutoff.
Default is \code{0.99}.}

\item{weight_function}{Character string or function used to symmetrize
adjacency matrices (\code{"mean"}, \code{"max"}, etc.).}

\item{nCores}{Integer. Number of CPU cores to use for
parallelization. Default is the number of workers in the current
\pkg{BiocParallel} backend. Note: JRF uses C implementation and
does not use this parameter.}

\item{grnboost_modules}{Python modules needed for \code{GRNBoost2} if
using reticulate.}

\item{debug}{Logical. If \code{TRUE}, prints detailed progress messages.
Default is \code{FALSE}.}
}
\value{
A \linkS4class{SummarizedExperiment} object where each assay is a
binary (thresholded) adjacency matrix corresponding to an input weighted
matrix. Metadata includes cutoff values and method parameters.
}
\description{
Applies a cutoff to weighted adjacency matrices using a percentile
estimated from shuffled versions of the original expression matrices.
Supports inference methods \code{"GENIE3"}, \code{"GRNBoost2"},
and \code{"JRF"}.
}
\details{
For each input expression matrix, \code{n} shuffled versions are
generated by randomly permuting each gene’s expression across cells.
Network inference is performed on the shuffled matrices, and a cutoff
is determined as the specified quantile (\code{quantile_threshold}) of
the resulting edge weights. The original weighted adjacency matrices
are then thresholded using these estimated cutoffs.

Parallelization is handled via \pkg{BiocParallel}.

The methods are based on:
\itemize{
\item \strong{GENIE3}: Random Forest-based inference (Huynh-Thu et
al., 2010).
\item \strong{GRNBoost2}: Gradient boosting trees using arboreto
(Moerman et al., 2019).
\item \strong{JRF}: Joint Random Forests across multiple conditions
(Petralia et al., 2015).
}
}
\examples{
data(toy_counts)


# Infer networks (toy_counts is already a MultiAssayExperiment)
networks <- infer_networks(
    count_matrices_list = toy_counts,
    method = "GENIE3",
    nCores = 1
)
head(networks[[1]])

# Generate adjacency matrices
wadj_se <- generate_adjacency(networks)
swadj_se <- symmetrize(wadj_se, weight_function = "mean")

# Apply cutoff
binary_se <- cutoff_adjacency(
    count_matrices = toy_counts,
    weighted_adjm_list = swadj_se,
    n = 1,
    method = "GENIE3",
    quantile_threshold = 0.95,
    nCores = 1,
    debug = TRUE
)
head(binary_se[[1]])
}
