% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitting_functions_multiple_genes.R
\name{fitRegMultiple}
\alias{fitRegMultiple}
\alias{fitNullMultiple}
\alias{fitFullMultiple}
\title{Fit models.}
\usage{
fitRegMultiple(
  rs_results,
  rs_data,
  alpha = 1,
  lambda_choice = c("lambda.min", "lambda.1se"),
  seed = NULL,
  ...
)

fitNullMultiple(rs_results, rs_data, seed = NULL, ...)

fitFullMultiple(rs_results, rs_data, seed = NULL, ...)
}
\arguments{
\item{rs_results}{\code{\linkS4class{RegspliceResults}} object, which will be used to 
store results. Initialized using the constructor function \code{RegspliceResults()}.
See \code{\linkS4class{RegspliceResults}} for details.}

\item{rs_data}{\code{\linkS4class{RegspliceData}} object. In the case of RNA-seq read 
count data, this has been pre-transformed with \code{\link{runVoom}}. Contains 
\code{counts} and \code{weights} data matrices, and vector of experimental 
conditions for each biological sample stored in \code{colData}. See 
\code{\linkS4class{RegspliceData}} for details.}

\item{alpha}{Elastic net parameter \code{alpha} for \code{glmnet} model fitting 
functions. Must be between 0 (ridge regression) and 1 (lasso). Default is 1 (lasso).
See \code{glmnet} documentation for more details.}

\item{lambda_choice}{Parameter to select which optimal \code{lambda} value to choose 
from the \code{cv.glmnet} cross validation fit. Choices are "lambda.min" (model with
minimum cross-validated error) and "lambda.1se" (most regularized model with 
cross-validated error within one standard error of minimum). Default is 
"lambda.min". See \code{glmnet} documentation for more details.}

\item{seed}{Random seed (integer). Default is NULL. Provide an integer value to set 
the random seed for reproducible results.}

\item{...}{Other arguments to pass to \code{cv.glmnet}, \code{glmnet}, or \code{glm}.}
}
\value{
Returns a \code{\linkS4class{RegspliceResults}} object containing deviance and
  degrees of freedom of the fitted models. See \code{\linkS4class{RegspliceResults}}
  for details.
}
\description{
Model fitting functions for \code{regsplice} package.
}
\details{
There are three model fitting functions:

\code{fitRegMultiple} fits regularized (lasso) models containing an optimal subset of 
exon:condition interaction terms for each gene. The model fitting procedure penalizes 
the interaction terms only, so that the main effect terms for exons and samples are 
always included. This ensures that the null model is nested, allowing likelihood ratio
tests to be calculated.

\code{fitNullMultiple} fits the null models, which do not contain any interaction 
terms.

\code{fitFullMultiple} fits full models, which contain all exon:condition interaction
terms for each gene.

See \code{\link{createDesignMatrix}} for more details about the terms in each model.

The fitting functions fit models for all genes in the data set.

A random seed can be provided with the \code{seed} argument, to generate reproducible
results.

If the \code{rs_data} object does not contain a weights matrix, all exon bins are 
weighted equally.

Previous step: Initialize \code{\linkS4class{RegspliceResults}} object with 
\code{\link{initializeResults}}.
Next step: Calculate likelihood ratio tests with \code{\link{LRTests}}.
}
\examples{
file_counts <- system.file("extdata/vignette_counts.txt", package = "regsplice")
data <- read.table(file_counts, header = TRUE, sep = "\t", stringsAsFactors = FALSE)
head(data)

counts <- data[, 2:7]
tbl_exons <- table(sapply(strsplit(data$exon, ":"), function(s) s[[1]]))
gene_IDs <- names(tbl_exons)
n_exons <- unname(tbl_exons)
condition <- rep(c("untreated", "treated"), each = 3)

rs_data <- RegspliceData(counts, gene_IDs, n_exons, condition)

rs_data <- filterZeros(rs_data)
rs_data <- filterLowCounts(rs_data)
rs_data <- runNormalization(rs_data)
rs_data <- runVoom(rs_data)

rs_results <- initializeResults(rs_data)

rs_results <- fitRegMultiple(rs_results, rs_data)
rs_results <- fitNullMultiple(rs_results, rs_data)
rs_results <- fitFullMultiple(rs_results, rs_data)

}
\seealso{
\code{\link{createDesignMatrix}} \code{\link{RegspliceResults}} 
  \code{\link{initializeResults}} \code{\link{LRTests}}

\code{\link[glmnet]{glmnet}} \code{\link[glmnet]{cv.glmnet}} 
  \code{\link[stats]{glm}}
}
