#' similar RegionSets
#'
#' @description
#'
#' Create a list of similar region sets to a reference region set.
#'
#' @details
#'
#' This function takes a region set as an input and a vector of desired randomized
#' fractions. For each fraction value, a new region set will be generated
#' where that fraction of the original regions in the input region set has been randomized.
#' In effect, this creates region sets that are "similar" to a controlled degree
#' to the original region set. This tool can be useful for validation purposes
#' and its use in the demonstration of the usage of this package can be
#' seen in the `RegioneReloaded` vignette.
#'
#' @usage similarRegionSet(GR, name, genome, vectorPerc)
#'
#' @param GR a [GRanges][GenomicRanges::GRanges] object with the input region set.
#' @param name character, name for the output region sets. The names will be generated by adding an underscore and the
#' fraction of similarity after the name of each region set generated. (default = "A")
#' @param genome genome of reference to generate the similar region sets. (default = "hg19)
#' @param vectorPerc numeric, vector of desired randomized fractions. (default = seq(.1,.9,.1))
#'
#' @return A list of [GRanges][GenomicRanges::GRanges] objects.
#'
#' @examples
#'
#' data("cw_Alien")
#'
#' A<-createRandomRegions(nregions = 20, length.mean = 1000, length.sd = 100,
#' genome = AlienGenome)
#'
#' similAList <- similarRegionSet(GR = A, genome = AlienGenome,
#' vectorPerc = seq(0.1,0.9,0.2), name = "test")
#'
#' summary (similAList)
#'
#' @seealso [GRanges][GenomicRanges::GRanges]
#'
#' @examples
#'
#' data("cw_Alien")
#'
#' regA <- createRandomRegions(
#'   nregions = 100,
#'   length.mean = 10,
#'  length.sd = 5,
#'  genome = AlienGenome
#' )
#'
#' listRegA <- similarRegionSet(GR = regA, genome = AlienGenome)
#' summary(listRegA)
#'
#' @export similarRegionSet
#'
similarRegionSet <- function(GR,
                             name = "A",
                             genome = "hg19",
                             vectorPerc = seq(.1,.9,.1))
  {

  GRL<-lapply(seq_along(vectorPerc),
              FUN = function(i, vectorPerc, GR, genome){
                GRL <- randomizeRegionsPerc(GR, genome = genome, frac = vectorPerc[i])},
              vectorPerc, GR, genome)

  vectorNames <- paste0(name, "_0", vectorPerc * 10)

  GRL <- c(GRL, GR)
  vectorNames <- c(vectorNames, name)
  names(GRL) <- vectorNames

  return(GRL)
}
