#' Create redirection pages
#'
#' Create HTML pages to redirect users to the latest version of the relevant Bioconductor book.
#' This is useful for preserving compatibility with old links when reorganizing the contents of a book.
#'
#' @param dir String containing the path to the output directory for the HTMLs.
#' @param name Character vector containing the name of each HTML.
#' @param pkg Character vector containing the name of the Bioconductor book package to redirect to.
#' @param page Character vector containing the name of the new chapter to redirect to.
#' @param file String containing the name of a comma-separate file with three unnamed columns, from which to derive \code{name}, \code{pkg} and \code{page}.
#' @param check Logical scalar indicating whether to check if the destination URL exists.
#' @param include.gif Logical scalar indicating whether a GIF should be included in the redirection notice.
#'
#' @details
#' This function is intended to be called inside the Makefile generated by \code{\link{configureBook}},
#' which will create the necessary HTMLs at package build time.
#' The expectation is that there is a file like \code{redirect.txt} that can be passed in as the \code{file} argument.
#' The default \code{dir} is the same as the final destination for all HTMLs that is defined in the Makefile.
#'
#' In \code{file}, the last column can be left empty for any row.
#' This will instruct \code{createRedirects} to re-use \code{name} as \code{page},
#' which is convenient when a chapter is simply moved to another package without a change in the HTML file name.
#'
#' It is probably a good idea to run with \code{check=TRUE} on occasion, to verify that the redirections are working.
#' This is not done by default to avoid a chicken-and-egg situation where two books cannot build because they redirect to each other.
#'
#' @return
#' HTMLs of the specified \code{name} are created in \code{dir}, redirected to the sites defined by their respective \code{pkg} and \code{page} entries.
#' A \code{NULL} is invisibly returned.
#'
#' @author Aaron Lun
#' 
#' @examples
#' tmp <- tempfile()
#' dir.create(tmp)
#' createRedirects("BLAH.html", pkg="OSCA.intro", page="installation.html", dir=tmp)
#' 
#' if (interactive()) {
#'     browseURL(file.path(tmp, "BLAH.html"))
#' }
#'
#' @export
#' @importFrom BiocStyle Biocbook
#' @importFrom utils read.csv
createRedirects <- function(name, pkg, page, dir="../inst/doc/book", file=NULL, check=FALSE, include.gif=TRUE) {
    if (!is.null(file)) {
        contents <- read.csv(file, header=FALSE)
        name <- contents[,1]
        pkg <- contents[,2]
        page <- contents[,3]
        keep <- page=="" | is.na(page)
        page[keep] <- name[keep]
    }

    gif.rotation <- c(
        `yuki nagato`="https://media3.giphy.com/media/kz6cm1kKle2MYkHtJF/giphy.gif",
        `ghost in the shell`="https://i.pinimg.com/originals/64/56/bd/6456bd0dc08f90dcb40d07c252e62435.gif",
        `ika musume`="https://www.animefeminist.com/wp-content/uploads/2020/04/type-computer-squid-girl.gif",
        `nene sakura`="https://i.pinimg.com/originals/b5/a6/5b/b5a65b848bc39642c5bae8e86b069560.gif"
    )

    for (i in seq_along(name)) {
        cur.pkg <- pkg[i]
        url <- sub("\\)", "", sub(".*\\(", "", Biocbook(cur.pkg, label="X")))
        url <- file.path(url, page[i])
        if (check) {
            if (!RCurl::url.exists(url)) {
                stop("destination URL '", url, "' does not exist")
            }
        }

        if (include.gif) {
            extra <- sprintf('<img src="%s" alt="typing", width="600", height="400">', sample(gif.rotation, 1))
        } else {
            extra <- ""
        }

        title <- pkg[i]

        blob <- sprintf('<!DOCTYPE html>
<meta charset="utf-8">
<title>Redirecting to %s</title>
<meta http-equiv="refresh" content="5; URL=%s">
<link rel="canonical" href="%s">

<html>
<head>
<link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.5.2/css/bootstrap.min.css" integrity="sha384-JcKb8q3iqJ61gNV9KGb8thSsNjpSL0n8PARn9HuZOnIxN0hoP+VmmDGMN5t9UJ0Z" crossorigin="anonymous">
</head>
<body class="d-flex flex-column h-100">
    <!-- Begin page content -->
    <main role="main" class="flex-shrink-0">
        <div class="container">
             <h1 class="mt-5">Redirecting to %s...</h1>

            <p class="lead">Maintainers should update their links to point to the new <a href="%s">location</a>.</p>

            <p>Wait a moment, we are redirecting you now:</p>

            %s
        </div>
    </main>
</body>
</html>', title, url, url, title, url, extra)
        write(file=file.path(dir, name[i]), blob)
    }

    invisible(NULL)
}
