% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/randRot.R
\name{pFdr}
\alias{pFdr}
\title{Calculate resampling based p-values and FDRs}
\usage{
pFdr(obj, method = "none", pooled = TRUE, na.rm = FALSE, beta = 0.05)
}
\arguments{
\item{obj}{A \code{rotateStat} object as returned by \code{\link[randRotation:rotateStat]{rotateStat}}.}

\item{method}{Can be either \code{"none"} (default), \code{"fdr.q"}, \code{"fdr.qu"} or any term that can be passed as \code{method} argument to \code{\link[stats:p.adjust]{stats::p.adjust}}, see \code{Details}. If \code{method = "none"}, resampling based
p-values without further adjustment are calculated.}

\item{pooled}{\code{logical}. \code{TRUE} (default) if marginal distributions are exchangeable for all features so that rotated stats can be pooled, see \code{Details}.}

\item{na.rm}{\code{logical}. \code{NA} values are ignored if set \code{TRUE}. \code{NA} values should be avoided and could e.g. be removed by imputation in original data or by removing features that contain \code{NA} values. Few \code{NA} values do not have a large effect, but many \code{NA} values can lead to wrong estimations of p-values and FDRs. We highly recommend avoiding \code{NA} values.}

\item{beta}{\code{numeric} between \code{0} and \code{1}. Corresponds to beta in \insertCite{Yekutieli1999}{randRotation}.}
}
\value{
A \code{numeric} matrix of corrected p-values or FDRs with dimension \code{dim(obj$s0)}.
}
\description{
This function calculates either (1) resampling based p-values with subsequent
p-value adjustment using \code{\link[stats:p.adjust]{stats::p.adjust}} or (2)
resampling based false-discovery-rates (FDRs) for rotated statistics from a
\code{\link[randRotation:rotateStat]{rotateStat}} object.
}
\details{
Larger values of obj$s0 are considered more significant when
  compared to the empirical distribution. E.g. for calculation of resampling
  based p-values (with \code{pooled = FALSE}) we in principle use
  \code{p.val <- (rowSums(obj$stats >= obj$s0)+1)/(ncol(obj$stats)+1)} according to
  \insertCite{Phipson2010}{randRotation}.

  \code{method = "fdr.q"} and \code{method = "fdr.qu"} are resampling based
  fdr estimates and can only be used with \code{pooled = TRUE}. \code{method
  = "fdr.q"} is the FDR local estimator and \code{method = "fdr.qu"} is the
  FDR upper limit, see \insertCite{Reiner2003,Yekutieli1999}{randRotation}.
  For all other \code{method} arguments resampling based p-values are
  calculated and passed to \code{\link[stats:p.adjust]{stats::p.adjust}} for
  p-value adjustment. So these methods provide resampling based p-values with
  (non-resampling based) p-value adjustment.
  \code{method = "fdr.q"} and \code{method = "fdr.qu"} were
  adapted from package \code{fdrame}
  \insertCite{Fdrame2019,Reiner2003}{randRotation}.

  When \code{pooled = TRUE},
  marginal distributions of the test statistics are considered exchangeable
  for all features. The resampling based p-values of each feature are then
  calculated from all rotated statistics (all features, all rotations). For
  these cases, if the number of features is reasonably large, usually only
  few resamples (argument \code{R} in
  \code{\link[randRotation:rotateStat]{rotateStat}}) are required.
  We want to emphasize that in order for the marginal distributions to be
  exchangeable, the statistics must be a pivotal quantity (i.e. it must be
  scale independent).
  Pivotal quantities are e.g. t values. Using e.g. linear models
  with \code{coef} as statistics is questionable if the different features
  are measured on different scales. The resampled coefficients then
  have different variances and \code{pooled = TRUE} is not applicable.
  We thus highly recommend using pivotal quantities as \code{statistics} in
  \code{\link[randRotation:rotateStat]{rotateStat}}
  if possible.

  When \code{pooled = FALSE} the resampling based p-values are calculcated
  for each feature separately. This is required if one expects the resampling
  based statistics to be distributed differently for individual features. For
  most common applications this should not be the case and the marginal
  distribution are exchangeable for all features, hence \code{pooled = TRUE}
  by default.

  If \code{method = "fdr.q"} or \code{method = "fdr.qu"} and
  \code{weights} were specified when initialising the random rotation
  object (see parameter \code{initialised.obj} in
  \code{\link[randRotation:rotateStat]{rotateStat}}), a warning is displayed.
  The correlation structure (dependence structure) of linear model
  coefficients between different features is not generally preserved if
  different weights are used for different features.
  Methods \code{fdr.q} and \code{fdr.qu} rely on preserved correlation
  structure of dependent statistics and thus should not be used if statistics
  based on model coefficients (e.g. t statistics of model coefficients) are
  used in combination with different weights.

  P-values and FDRs are calculated for each column of \code{obj$s0}
  separately.
}
\examples{
# See also '?rotateStat':

#set.seed(0)

# Dataframe of phenotype data (sample information)
# We simulate 2 sample classes processed in 3 batches
pdata <- data.frame(batch = rep(1:3, c(10,10,10)),
                   phenotype = rep(c("Control", "Cancer"), c(5,5)))
features <- 100

# Matrix with random gene expression data
edata <- matrix(rnorm(features * nrow(pdata)), features)
rownames(edata) <- paste("feature", 1:nrow(edata))

mod1 <- model.matrix(~phenotype, pdata)

# Initialisation of the random rotation class
init1 <- initBatchRandrot(Y = edata, X = mod1, coef.h = 2, batch = pdata$batch)
init1

# Definition of the batch effect correction procedure with subsequent calculation
# of two-sided test statistics
statistic <- function(., batch, mod, coef){

  # The "capture.output" and "suppressMessages" simply suppress any output
  capture.output(suppressMessages(
    Y.tmp <- sva::ComBat(., batch = batch, mod)
  ))

  fit1 <- lm.fit(mod, t(Y.tmp))
  abs(coef(fit1)[coef,])
}

# We calculate test statistics for the second coefficient

res1 <- rotateStat(initialised.obj = init1,
                    R = 10,
                    statistic = statistic,
                    batch = pdata$batch, mod = mod1, coef = 2)

hist(pFdr(res1))
}
\references{
\insertAllCited{}
}
\seealso{
\code{\link[randRotation:rotateStat]{rotateStat}}
}
\author{
Peter Hettegger
}
