% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/syntheticNucMap.R
\name{syntheticNucMap}
\alias{syntheticNucMap}
\title{Generates a synthetic nucleosome map}
\usage{
syntheticNucMap(
  wp.num = 100,
  wp.del = 10,
  wp.var = 20,
  fuz.num = 50,
  fuz.var = 50,
  max.cover = 20,
  nuc.len = 147,
  lin.len = 20,
  rnd.seed = NULL,
  as.ratio = FALSE,
  show.plot = FALSE
)
}
\arguments{
\item{wp.num}{Number of well-positioned (non overlapped) nucleosomes. They
are placed uniformly every \code{nuc.len+lin.len} basepairs.}

\item{wp.del}{Number of well-positioned nucleosomes (the ones generated by
\code{wp.num}) to remove. This will create an uncovered region.}

\item{wp.var}{Maximum variance in basepairs of the well-positioned
nucleosomes. This will create some variation in the position of the reads
describing the same nucleosome.}

\item{fuz.num}{Number of fuzzy nucleosomes. They are distributed randomly
over all the region. They could be overlapped with other well-positioned
or fuzzy nucleosomes.}

\item{fuz.var}{Maximum variance of the fuzzy nucleosomes. This allow to set
different variance in well-positioned and fuzzy nucleosome reads (using
\code{wp.var} and \code{fuz.var}).}

\item{max.cover}{Maximum coverage of a nucleosome, i.e., how many times a
nucleosome read can be repeated. The final coverage probably will be
higher by the addition of overlapping nucleosomes.}

\item{nuc.len}{Nucleosome length. It's not recomended change the default
147bp value.}

\item{lin.len}{Linker DNA length. Usually around 20 bp.}

\item{rnd.seed}{As this model uses random distributions for the placement,
by setting the rnd.seed to a known value allows to reproduce maps in
different executions or computers. If you don't need this, just left it in
default value.}

\item{as.ratio}{If \code{as.ratio=TRUE} this will create and return a synthetic
naked DNA control map and the ratio between it and the nucleosome
coverage. This can be used to simulate hybridization ratio data, like the
one in Tiling Arrays.}

\item{show.plot}{If \code{TRUE}, will plot the output coverage map, with the
nucleosome calls and optionally the calculated ratio.}
}
\value{
A list with the following elements:
\itemize{
\item wp.starts Start points of well-positioned nucleosomes
\item wp.nreads Number of repetitions of each well positioned read
\item wp.reads Well positioned nucleosome reads (\code{IRanges} format),
containing the repetitions
\item fuz.starts Start points of the fuzzy nucleosomes
\item fuz.nreads Number of repetitions of each fuzzy nucleosome read
\item fuz.reads Fuzzy nucleosome reads (\code{IRanges} format), containing all
the repetitions
\item syn.reads All synthetic nucleosome reads togheter (\code{IRanges} format)
}

The following elements will be only returned if \code{as.ratio=TRUE}:
\itemize{
\item ctr.reads The pseudo-naked DNA (control) reads (\code{IRanges} format)
\item syn.ratio The calculated ratio nucleosomal/control (\code{Rle} format)
}
}
\description{
This function generates a synthetic nucleosome map using the parameters
given by the user and returns the coverage (like NGS experiments) or a
pseudo-hybdridization ratio (like Tiling Arrays) toghether with the perfect
information about the well positioned and fuzzy nucleosome positions.
}
\examples{
# Generate a synthetic map with 50wp + 20fuzzy nucleosomes using fixed
# random seed=1
res <- syntheticNucMap(wp.num=50, fuz.num=20, show.plot=TRUE, rnd.seed=1)

# Increase the fuzzyness
res <- syntheticNucMap(
    wp.num=50, fuz.num=20, wp.var=70, fuz.var=150, show.plot=TRUE,
    rnd.seed=1
)

# Calculate also a random map and get the ratio between random and
# nucleosomal
res <- syntheticNucMap(
    wp.num=50, wp.del=0, fuz.num=20, as.ratio=TRUE, show.plot=TRUE,
    rnd.seed=1
)

print(res)

# Different reads can be accessed separately from results
# Let's use this to plot the nucleosomal + the random map
library(ggplot2)
as <- as.vector(coverage.rpm(res$syn.reads))
bs <- as.vector(coverage.rpm(res$ctr.reads))
cs <- as.vector(res$syn.ratio)
plot_data <- rbind(
    data.frame(x=seq_along(as), y=as, lab="nucleosomal"),
    data.frame(x=seq_along(bs), y=bs, lab="random"),
    data.frame(x=seq_along(cs), y=cs, lab="ratio")
)
qplot(x=x, y=y, data=plot_data, geom="area", xlab="position", ylab="") +
    facet_grid(lab~., scales="free_y")

}
\author{
Oscar Flores \email{oflores@mmb.pcb.ub.es}
}
\keyword{datagen}
