% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/NormRFit.R
\docType{class}
\name{NormRFit-class}
\alias{NormRFit-class}
\alias{NormRFit}
\alias{normRFit}
\alias{normrfit}
\alias{NormRfit}
\alias{exportR,NormRFit,character-method}
\alias{exportR}
\alias{plot,NormRFit,missing-method}
\alias{plot.NormRFit}
\alias{getCounts,NormRFit-method}
\alias{getCounts}
\alias{getRanges,NormRFit-method}
\alias{getRanges}
\alias{getPosteriors,NormRFit-method}
\alias{getPosteriors}
\alias{getEnrichment,NormRFit-method}
\alias{getEnrichment}
\alias{getPvalues,NormRFit-method}
\alias{getPvalues}
\alias{getQvalues,NormRFit-method}
\alias{getQvalues}
\alias{getClasses,NormRFit-method}
\alias{getClasses}
\alias{length,NormRFit-method}
\alias{print,NormRFit-method}
\alias{show,NormRFit-method}
\alias{summary,NormRFit-method}
\title{Container for a fit done with normR}
\usage{
\S4method{exportR}{NormRFit,character}(x, filename, fdr = 0.01, color = NA,
  type = c(NA, "bed", "bedGraph", "bigWig"))

\S4method{plot}{NormRFit,missing}(x, y, ...)

\S4method{getCounts}{NormRFit}(x)

\S4method{getRanges}{NormRFit}(x, fdr = NA, k = NULL)

\S4method{getPosteriors}{NormRFit}(x)

\S4method{getEnrichment}{NormRFit}(x, B = NA, F = NA, standardized = TRUE,
  procs = 1L)

\S4method{getPvalues}{NormRFit}(x, filtered = FALSE)

\S4method{getQvalues}{NormRFit}(x)

\S4method{getClasses}{NormRFit}(x, fdr = NA)

\S4method{length}{NormRFit}(x)

\S4method{print}{NormRFit}(x, digits = max(3L, getOption("digits") - 3L), ...)

\S4method{show}{NormRFit}(object)

\S4method{summary}{NormRFit}(object, print = TRUE, digits = 3, ...)
}
\arguments{
\item{x}{A \code{NormRFit} object.}

\item{filename}{A \code{character} specifying the file to write to.}

\item{fdr}{\code{NA} or a \code{numeric} between \code{0} and \code{1}
specifying a FDR-level. Only regions with a q-value smaller than \code{fdr}
will be returned. If set to \code{NA}, all regions analyzed will be
returned (\code{\link{getRanges}}), classes are assigned by Maximum A
Posteriori (\code{\link{exportR}}).}

\item{color}{Specified color(s) when printing a bed file. If \code{x@type ==
"enrichR"}, \code{color} should of length 1 and color shading will be done
on this color. If \code{x@type == "diffR"}, \code{color} should be of length
2 giving, firstly, the color for \code{control} and, secondly, the color for
\code{treatment}. If \code{x@type == "regimeR"}, \code{color} should be of
length \code{x@k-1}, specifying a color for each enrichment component. Per
default an appropriate color palette is used.}

\item{type}{A \code{character} specifying the filetype for exporting
results. If \code{NA}, format is guessed from \code{filename}'s extension.}

\item{y}{not used.}

\item{...}{optional arguments to be passed directly to the inherited
function without alteration and with the original names preserved.}

\item{k}{\code{NULL} or a \code{integer} specifying a model component for
which regions have to be returned. If set to \code{NULL}, regions are not
filtered on component assignments. If \code{fdr} is set and \code{k == x@B},
the function stops.}

\item{B}{An \code{integer} specifying the index of a mixture component. The
enrichment is calculated relative to this component used as a background
component. If \code{<NA>} (default), the background is determined by normR.}

\item{F}{An \code{integer} specifying the index of a mixture component. The
enrichment is calculated for this component over background \code{B}. If
\code{<NA>} (default), the component with theta closest to B is used
(\code{enrichR}, \code{regimeR}). For \code{diffR}, \code{F} is not
effective.}

\item{standardized}{A \code{logical} indicating if the enrichment should be
standardized betwen 0 and 1. A non-standardized enrichment is particular
useful when comparing intensities for ChIP-seq against the same antigen in
different conditions (default = TRUE).}

\item{procs}{An \code{integer} specifying the number of threads to use.}

\item{filtered}{A \code{logical} specifying if T-filtered or all
(default) P-values should be returned.}

\item{digits}{Number of digits to show in number formatting.}

\item{object}{A \code{NormRCountConfig} object.}

\item{print}{\code{logical()} indicating if summary should be print to
screen}
}
\value{
getCounts: A \code{list} of length 2 with \code{integer} for control
and treatment each.

getRanges: A \code{GenomicRanges} object.

getPosteriors: A \code{matrix} of posteriors for \code{x@k} mixture
components

getEnrichment: A \code{numeric} of length \code{length(x@n)} giving
the normR computed enrichment.

getPvalues: A \code{numeric} of length \code{length(x@n)} giving
the normR computed Pvalues.

getQvalues: A \code{numeric} of length \code{length(x@filteredT)}
giving the FDR-corrected q-values using Storey's method.

getClasses: A \code{integer} specifying assignments of regions to
the mixture model. If \code{x@type == "enrichR"}, it contains \code{1} for
enriched regions and \code{NA} for non-enriched regions. If \code{x@type ==
"diffR"}, it contains \code{1} for control-enriched regions, \code{2} for
treatment-enriched regions and \code{NA} for non-enriched regions. If
\code{x@type == "regimeR"}, it contains \code{>= 1} for regime-enriched
regions and \code{NA} for non-enriched regions.
}
\description{
This S4 class wraps a \code{\link{normR}} fit containing counts, fit
configuration and results of the fit. Herein, functions for printing,
summarization and accessing are provided. The
functions \code{\link{enrichR}}, \code{\link{diffR}} and
\code{\link{regimeR}} generate a container of this class to save results of
a normR binomial mixture fitting. Please refer to their documentation for
conventional usage of the normR package.
}
\details{
When working with instances of this S4 class, it is recommended to only use
functions to access contents of this object. Internally, the class holds a
map structure of unique elements to reduce memory requirements. #'
}
\section{Methods (by generic)}{
\itemize{
\item \code{exportR}: Export results of a normR fit to common file formats.

\item \code{plot}: Plot a NormRFit.

\item \code{getCounts}: Get count data for control and treatment.

\item \code{getRanges}: Get the genomic coordinates of regions analyzed with
information about component assignment.

\item \code{getPosteriors}: Get computed posteriors for each mixture component.

\item \code{getEnrichment}: Get normalized enrichment.

\item \code{getPvalues}: Get normR-computed P-values.

\item \code{getQvalues}: Get FDR-corrected q-values.

\item \code{getClasses}: Get component assignments for each region analyzed.

\item \code{length}: Returns the number of regions analyzed.

\item \code{print}: Prints a small summary on a NormRFit.

\item \code{show}: Shows a small summary on a NormRFit.

\item \code{summary}: Prints a concise summary of a NormRFit.
}}

\section{Slots}{

\describe{
\item{\code{type}}{A \code{character} representing the type of fit. One of
\code{c("enrichR","diffR", "regimeR")}.}

\item{\code{n}}{An \code{integer} specifying the number of regions.}

\item{\code{ranges}}{A \code{GenomicRanges} specifying the genomic coordinates of
the regions.}

\item{\code{k}}{An \code{integer} giving the number of binomial mixture components.}

\item{\code{B}}{An \code{integer} specifying the index of the background component.}

\item{\code{map}}{A vector of \code{integer} holding a map to map back
\code{counts}, \code{lnposteriors}, \code{lnenrichment}, \code{lnpvals},
\code{lnqvals} and \code{classes}. See low level function
\code{normr:::map2uniquePairs} for how the map is generated.}

\item{\code{counts}}{A \code{list} of length two containing a vector of
\code{integer} holding unique counts for control and treatment each. Use
\code{\link{getCounts}} to retrieve original count matrix.}

\item{\code{amount}}{A vector of \code{integer} specifying the number of occurences
of each unique control / treatment count pair.}

\item{\code{names}}{A \code{character} of length two specifying the names for
control and treatment.}

\item{\code{thetastar}}{A \code{numeric} giving the calculated naive background
estimation, \emph{i.e.} \code{sum(getCounts(obj)[2,])/sum(getCounts(obj))}}

\item{\code{theta}}{A \code{numeric} of length \code{k} giving the normR fitted
parametrization of \code{k} binomial mixture components.}

\item{\code{mixtures}}{A \code{numeric} of length \code{k} giving the normR fitted
mixture proportions of \code{k} binomial mixture components. Should add up
to one.}

\item{\code{lnL}}{A vector of \code{numeric} holding the log-likelihood-trace of
a normR model fit.}

\item{\code{eps}}{A \code{numeric} used as threshold for normR fit EM convergence.}

\item{\code{lnposteriors}}{A \code{matrix} with \code{length(amount)} rows and
\code{k} columns. It contains the ln posterior probabilities for each unique
control / treatment count pair. Use \code{\link{getPosteriors}} to get the
posterior matrix for the original data.}

\item{\code{lnenrichment}}{A \code{numeric} of length \code{length(amount)} holding
calculared normalized enrichment for each unique control / treatment count
pair. The enrichment is calculated with respect to the fitted component
\code{B}. Use \code{\link{getEnrichment}} to retrieve enrichment for the
original data.}

\item{\code{lnpvals}}{A \code{numeric} of length \code{length(amount)} holding ln
P-values for each unique control / treatment count pair. Given
\code{theta} of \code{B} the signifcane of enrichment is assigned. Use
\code{\link{getPvalues}} to retrieve Pvalues for original data.}

\item{\code{thresholdT}}{An \code{integer} giving the threshold used to filter
P-values for FDR correction. The T-Filter threshold is a calculated
population size for which the null hypothesis (\code{theta} of \code{B}) can
be rejected. \code{eps} specifies the significance level.}

\item{\code{filteredT}}{A vector of \code{integer} giving indices of P-values
passing \code{thresholdT}. Only these P-values will be considered for FDR
correction.}

\item{\code{lnqvals}}{A \code{numeric} of length \code{length(filteredT)} holding
ln q-values (FDR correction). P-values are corrected for multiple testing
using Storey's method.}

\item{\code{classes}}{A \code{integer} of length \code{length(amount)} specifying
the class assignments for each unique control / treatment count pair. These
class assignments are based on the normR model fit. For \code{type ==
"enrichR"}, this vector contains either \code{NA} (not enriched) or \code{1}
(enriched). For \code{type == "diffR"}, this vector contains \code{NA}
(unchanged), \code{1} (differential in ChIP-seq 1) and \code{2}
(differential in ChIP-seq 2). For \code{type == "regimeR"}, this vector
contains \code{NA} (not enriched) and an arbitary number of enrichment class
\code{>= 1}.}
}}

\examples{
require(GenomicRanges)

#Create a toy instance of type 'enrichR'
fit <- new("NormRFit",
           type="enrichR", n=10L,
           ranges=GRanges("chr1", IRanges(seq(1,100,10), width=10)),
           k=2L, B=1L, map=rep(1:5,2), counts=list(1:5, 1:5),
           amount=rep(2L,5), names=c("chip", "input"), thetastar=.35,
           theta=c(.15,.55), mixtures=c(.9,.1), lnL=seq(-50,-1,10), eps=.001,
           lnposteriors=log(matrix(runif(10), ncol=2)),
           lnenrichment=log(runif(5,0,.2)), lnpvals=log(runif(5)),
           filteredT=2:5, thresholdT=1L, lnqvals=log(runif(5,0,.2)),
           classes=sample(1:2,5,TRUE))

#print some statistics on fits
fit
summary(fit)

## Not run:
#write significant regions to bed
#exportR(fit, filename = "enrich.bed", fdr = 0.1)
#write normalized enrichment to bigWig
#exportR(fit, filename = "enrich.bw")
## End(**Not run**)

###AccessorMethods
#get original counts
getCounts(fit)
#get genomic coordinates for significant ranges as a GenomicRanges instance
getRanges(fit, fdr = .1)
getPosteriors(fit)
getEnrichment(fit)
getPvalues(fit)
getQvalues(fit)
getClasses(fit)
}
\seealso{
\link{normr} for function creating this container
}
\author{
Johannes Helmuth \email{helmuth@molgen.mpg.de}
}
