//
// Original author: Matt Chambers <matt.chambers42 .@. gmail.com>
//
// Copyright 2020 Matt Chambers
//
// Licensed under the Apache License, Version 2.0 (the "License"); 
// you may not use this file except in compliance with the License. 
// You may obtain a copy of the License at 
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software 
// distributed under the License is distributed on an "AS IS" BASIS, 
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
// See the License for the specific language governing permissions and 
// limitations under the License.
//

#define PWIZ_SOURCE

#include "SpectrumList_DiaUmpire.hpp"
#include "pwiz/utility/misc/mru_list.hpp"
#include "pwiz/data/msdata/MSDataFile.hpp"
#include <boost/thread.hpp>

using namespace pwiz::msdata;
using namespace pwiz::util;

namespace pwiz {
namespace analysis {


class SpectrumList_DiaUmpire::Impl
{
    public:
    Impl(const MSData& msd, const SpectrumListPtr& inner, const DiaUmpire::Config& config, const util::IterationListenerRegistry* ilr)
        : diaumpire_(msd, inner, config, ilr)//, spillFiles_(25)
    {
        size_ = diaumpire_.pseudoMsMsKeys().size();
    }

    SpectrumPtr spectrum(size_t index, DetailLevel detailLevel) const
    {
        if (index >= size_)
            throw std::runtime_error("[SpectrumList_DiaUmpire::spectrum()] Bad index: " + lexical_cast<string>(index));

        boost::lock_guard<boost::mutex> lock(readMutex);  // lock_guard will unlock mutex when out of scope or when exception thrown (during destruction)

        auto key = diaumpire_.pseudoMsMsKeys()[index];
        auto insertPair = spillFiles_.insert(make_pair(key.spillFilePtr, MSDataPtr()));
        if (insertPair.second)
            insertPair.first->second.reset(new MSDataFile(key.spillFilePtr->path().string()));

        MSDataPtr currentMSDataPtr = insertPair.first->second;

        SpectrumPtr s = currentMSDataPtr->run.spectrumListPtr->spectrum(key.spillFileIndex, detailLevel);
        s->id = key.id;
        s->index = key.index;
        return s;
    }

    DiaUmpire::DiaUmpire diaumpire_;
    size_t size_;

    typedef map<TemporaryFile*, MSDataPtr> SpillFileCache;

    mutable boost::mutex readMutex;
    mutable SpillFileCache spillFiles_;
};


PWIZ_API_DECL SpectrumList_DiaUmpire::SpectrumList_DiaUmpire(const MSData& msd, const SpectrumListPtr& inner, const DiaUmpire::Config& config, const util::IterationListenerRegistry* ilr)
    : msdata::SpectrumListWrapper(inner), impl_(new Impl(msd, inner, config, ilr))
{

    // add processing methods to the copy of the inner SpectrumList's data processing
    ProcessingMethod method;
    method.order = dp_->processingMethods.size();
    method.userParams.emplace_back(UserParam("Pseudo-spectra generated by DIA-Umpire demultiplexing"));
    for (const auto& kvp : config.instrumentParameters.GetParameterMap())
        method.userParams.emplace_back(kvp.first, kvp.second);

    if (config.diaTargetWindowScheme == DiaUmpire::TargetWindow::Scheme::SWATH_Variable)
    {
        string windowStr;
        for (auto& window : config.diaVariableWindows)
            windowStr += toString(window.mzRange.begin) + "-" + toString(window.mzRange.end) + " ";
        method.userParams.emplace_back("VariableWindows", windowStr);
    }

    if (!dp_->processingMethods.empty())
        method.softwarePtr = dp_->processingMethods[0].softwarePtr;

    dp_->processingMethods.emplace_back(method);
}

PWIZ_API_DECL SpectrumList_DiaUmpire::~SpectrumList_DiaUmpire() {}
PWIZ_API_DECL SpectrumPtr SpectrumList_DiaUmpire::spectrum(size_t index, bool getBinaryData) const { return impl_->spectrum(index, getBinaryData ? DetailLevel_FullData : DetailLevel_FullMetadata); }
PWIZ_API_DECL SpectrumPtr SpectrumList_DiaUmpire::spectrum(size_t index, DetailLevel detailLevel) const { return impl_->spectrum(index, detailLevel); }
PWIZ_API_DECL size_t SpectrumList_DiaUmpire::size() const { return impl_->size_; }
PWIZ_API_DECL const msdata::SpectrumIdentity& SpectrumList_DiaUmpire::spectrumIdentity(size_t index) const { return impl_->diaumpire_.pseudoMsMsKeys()[index]; }

} // namespace analysis
} // namespace pwiz
