checkInteger <- function(N){
    !length(grep("[^[:digit:]]", as.character(N)))
}

#' @importFrom grDevices col2rgb
hex2psrgb<-function(col){
    col<-col2rgb(col)
    col<-col/255
    col<-paste(col,collapse=" ")
    col
}

#' @importFrom grDevices col2rgb rgb
colalpha <- function(col, alpha){
  col<-col2rgb(col)
  rgb(t(col), alpha = round(alpha*255), maxColorValue = 255)
}

#' @importFrom grDevices svg dev.off
#' @importFrom grid convertHeight stringHeight grid.text gpar polygonGrob
#' @importFrom utils install.packages
importSVG <- function(font, color, ch, fontface="bold", envir=.globals){
  run_grImport2 <- TRUE
  if(exists("tmp_motifStack_grImport2", envir = .globals)){
    run_grImport2 <- get("tmp_motifStack_grImport2", envir = .globals)
  }
  if(capabilities("cairo") && !requireNamespace("grImport2", quietly = TRUE) &&
     run_grImport2){
    ## install grImport2
    install.packages("grImport2", repos = "https://cloud.r-project.org",
                     quiet = TRUE)
  }
  if(capabilities("cairo") && requireNamespace("grImport2", quietly = TRUE) &&
     run_grImport2){
    psfilename<-tempfile(fileext = ".svg")
    if(requireNamespace("Cairo")) FUN = Cairo::CairoSVG else FUN = svg
    FUN(psfilename, width = 1, height = 1, bg = NA, pointsize=72, family = font)
    h <- convertHeight(stringHeight(ch), unitTo = "points", valueOnly = TRUE)
    grid.text(ch, gp = gpar(fontsize=floor(1.05*72*(72/h)), 
                            fontfamily=font, col=color, fontface=fontface))
    dev.off()
    x <- tryCatch(grImport2::readPicture(psfilename), error=function(.e){
      message(.e, "Will try grImport")
      assign("tmp_motifStack_grImport2", FALSE, envir=.globals)
      NULL
    })
    unlink(psfilename)
    if(length(x)){
      assign("pictureGrob", grImport2::pictureGrob, envir=envir)
      assign("tmp_motifStack_grImport2", TRUE, envir=.globals)
      return(x)
    }
  }
  checkGhost <- function(){
    # get ghostscript path
    gscmd <- Sys.getenv("R_GSCMD")
    if(is.null(gscmd) || !nzchar(gscmd)) {
      if(.Platform$OS.type=="windows"){
        poss <- Sys.which(c("gswin64c.exe",
                            "gswin32c.exe"))
        poss <- poss[nzchar(poss)]
        return(length(poss)>0)
      }else{
        gscmd <- "gs"
      }
    }
    suppressWarnings(rc <- system(paste(shQuote(gscmd), "-help > /dev/null")))
    return(rc == 0) 
  }
  if(checkGhost() && !requireNamespace("grImport", quietly = TRUE)){
    ## install grImport
    install.packages("grImport", repos = "https://cloud.r-project.org",
                     quiet = TRUE)
  }
  if(checkGhost() && requireNamespace("grImport", quietly = TRUE)){
    motifStack_private_fontsize <- 18
    ps<-paste("%!PS\n/",font,
              " findfont\n",motifStack_private_fontsize,
              " scalefont\n",
              hex2psrgb(color),
              " setrgbcolor\nsetfont\nnewpath\n0 0 moveto\n(",
              ch,") show",sep="")
    psfilename<-tempfile()
    psfilename <- gsub("\\", "/", psfilename, fixed=TRUE)
    # step1 create a ps file
    cat(ps,file=paste(psfilename,".ps",sep=""))
    # step2 convert it by grImport::PostScriptTrace
    grImport::PostScriptTrace(paste(psfilename,".ps",sep=""), 
                              paste(psfilename,".xml",sep=""))
    # step3 read by grImport::readPicture
    x<-grImport::readPicture(paste(psfilename,".xml",sep=""))
    unlink(c(paste(psfilename,".ps",sep=""), 
             paste("capture",basename(psfilename),".ps",sep=""), 
             paste(psfilename,".xml",sep="")))
    assign("pictureGrob", grImport::pictureGrob, envir=envir)
    return(x)
  }
  warning("No import funciton (grImport or grImport2) available.",
          "Use predefined fonts.")
  assign("pictureGrob", function(picture, x = .5, y = .5,
                                 width = 1, height = 1, 
                                 just = "centre", distort = TRUE){
    ## plot
    gl <- gList()
    for(i in seq_along(picture)){
      gl <- gList(gl, polygonGrob(picture[[i]]$x*width+x,
                                  picture[[i]]$y*height+y,
                                  gp = gpar(fill=picture[[i]]$color,
                                            col=picture[[i]]$color)))
    }
    return(gl)
  }, envir=envir)
  letterPolygon <- 
    list(A=list(list(color="black", 
                     x=c(0.95,0.734703,0.6673312,0.3334011,0.2660293,
                         0.05,0.3744101,0.6263222,0.95),
                     y=c(0,0,0.2446256,0.2446256,0,0,1,1,0)),
                list(color="white",  
                     x=c(0.5003662,0.5000439,0.4996631,0.4992238,0.4987258,
                         0.4981692,0.4975541,0.4968804,0.4961481,0.4953572,
                         0.4945077,0.4945077,0.4937535,0.4929552,0.4921131,
                         0.491227,0.490297,0.489323,0.4883051,0.4872433,
                         0.4861375,0.4849878,0.4849878,0.4838015,0.4825858,
                         0.4813409,0.4800667,0.4787632,0.4774304,0.4760683,
                         0.474677,0.4732563,0.4718063,0.4718063,0.4704809,
                         0.4691408,0.467786,0.4664166,0.4650325,0.4636338,
                         0.4622205,0.4607925,0.4593499,0.4578926,0.4578926,
                         0.3788039,0.6219284,0.5428397,0.5413751,0.5399105,
                         0.5384459,0.5369813,0.5355167,0.5340521,0.5325875,
                         0.5311229,0.5296583,0.5281937,0.5281937,0.5268828,
                         0.5255867,0.5243051,0.5230382,0.521786,0.5205484,
                         0.5193255,0.5181172,0.5169235,0.5157445,0.5157445,
                         0.5145875,0.5134597,0.5123613,0.5112921,0.5102522,
                         0.5092417,0.5082604,0.5073084,0.5063857,0.5054923,
                         0.5054923,0.5047819,0.5041155,0.5034931,0.5029146,
                         0.50238,0.5018893,0.5014426,0.5010399,0.500681,
                         0.5003662,0.5003662),
                     y=c(0.8650852,0.8634915,0.8616753,0.8596368,0.8573758,
                         0.8548925,0.8521868,0.8492587,0.8461082,0.8427354,
                         0.8391401,0.8391401,0.8353818,0.8315196,0.8275537,
                         0.8234841,0.8193106,0.8150334,0.8106523,0.8061675,
                         0.8015789,0.7968866,0.7968866,0.7920979,0.7872202,
                         0.7822535,0.7771979,0.7720534,0.7668199,0.7614974,
                         0.756086,0.7505856,0.7449963,0.7449963,0.7395182,
                         0.7340549,0.7286064,0.7231727,0.7177539,0.7123499,
                         0.7069607,0.7015864,0.6962268,0.6908821,0.6908821,
                         0.409192,0.409192,0.6908821,0.6962268,0.7015864,
                         0.7069607,0.7123499,0.7177539,0.7231727,0.7286064,
                         0.7340549,0.7395182,0.7449963,0.7449963,0.7505856,
                         0.756086,0.7614974,0.7668199,0.7720534,0.7771979,
                         0.7822535,0.7872202,0.7920979,0.7968866,0.7968866,
                         0.8015789,0.8061675,0.8106523,0.8150334,0.8193106,
                         0.8234841,0.8275537,0.8315196,0.8353818,0.8391401,
                         0.8391401,0.8427354,0.8461082,0.8492587,0.8521868,
                         0.8548925,0.8573758,0.8596368,0.8616753,0.8634915,
                         0.8650852,0.8650852))
                ),
         B=list(list(color="black", 
                     x=c(0.95,0.949674,0.9486959,0.9470658,0.9447836,
                         0.9418494,0.9382631,0.9340248,0.9291344,0.923592,
                         0.9173975,0.9173975,0.910654,0.9034643,0.8958284,
                         0.8877464,0.8792183,0.870244,0.8608236,0.8509571,
                         0.8406444,0.8298856,0.8298856,0.8187235,0.8072011,
                         0.7953184,0.7830753,0.7704719,0.7575081,0.744184,
                         0.7304995,0.7164547,0.7020496,0.7020496,0.68755,
                         0.6728789,0.6580362,0.6430219,0.627836,0.6124786,
                         0.5969495,0.5812488,0.5653765,0.5493327,0.5493327,
                         0.05,0.05,0.500429,0.5180858,0.5353651,
                         0.5522669,0.5687912,0.584938,0.6007073,0.6160991,
                         0.6311134,0.6457502,0.6600095,0.6600095,0.6738656,
                         0.6872927,0.7002908,0.7128599,0.725,0.7367112,
                         0.7479933,0.7588465,0.7692707,0.779266,0.779266,
                         0.7888151,0.7979009,0.8065234,0.8146826,0.8223785,
                         0.8296111,0.8363804,0.8426864,0.8485291,0.8539085,
                         0.8539085,0.8589619,0.8634833,0.8674728,0.8709304,
                         0.8738561,0.8762498,0.8781115,0.8794414,0.8802393,
                         0.8805052,0.8805052,0.879939,0.8782402,0.875409,
                         0.8714452,0.8663489,0.8601201,0.8527588,0.844265,
                         0.8346387,0.8238799,0.8238799,0.8119457,0.7987931,
                         0.7844223,0.7688332,0.7520257,0.734,0.714756,
                         0.6942936,0.672613,0.649714,0.649714,0.6783527,
                         0.7055844,0.731409,0.7558265,0.778837,0.8004404,
                         0.8206368,0.8394261,0.8568084,0.8727836,0.8727836,
                         0.8874547,0.9005815,0.912164,0.9222021,0.9306959,
                         0.9376454,0.9430505,0.9469113,0.9492278,0.95,
                         0.95),
                     y=c(0.2853966,0.2703484,0.255745,0.2415864,0.2278725,
                         0.2146034,0.2017791,0.1893996,0.1774648,0.1659748,
                         0.1549296,0.1549296,0.1442995,0.1340549,0.1241957,
                         0.114722,0.1056338,0.09693106,0.08861379,0.08068199,
                         0.07313566,0.0659748,0.0659748,0.05933284,0.0530467,
                         0.04711638,0.04154188,0.0363232,0.03146034,0.0269533,
                         0.02280208,0.01900667,0.01556709,0.01556709,0.01260934,
                         0.009962936,0.007627872,0.005604151,0.003891772,
                         0.002490734,0.001401038,0.0006226835,0.0001556709,0,0,
                         0,1,1,0.9998443,0.9993773,0.998599,0.9975093,0.9961082,
                         0.9943958,0.9923721,0.9900371,0.9873907,0.9844329,
                         0.9844329,0.9813047,0.9778503,0.9740697,0.9699629,
                         0.96553,0.9607709,0.9556857,0.9502743,0.9445367,
                         0.9384729,0.9384729,0.932083,0.9253669,0.9183247,
                         0.9109563,0.9032617,0.8952409,0.886894,0.8782209,
                         0.8692216,0.8598962,0.8598962,0.8502446,0.8402669,
                         0.8299629,0.8193328,0.8083766,0.7970941,0.7854855,
                         0.7735508,0.7612898,0.7487027,0.7487027,0.7319199,
                         0.7156709,0.6999555,0.6847739,0.670126,0.6560119,
                         0.6424314,0.6293847,0.6168718,0.6048925,0.6048925,
                         0.5936916,0.5832172,0.5734692,0.5644477,0.5561527,
                         0.5485841,0.541742,0.5356264,0.5302372,0.5255745,
                         0.5255745,0.5224166,0.5182802,0.5131653,0.5070719,0.5,
                         0.4919496,0.4829207,0.4729133,0.4619274,0.4499629,
                         0.4499629,0.437109,0.4234544,0.4089993,0.3937435,
                         0.3776872,0.3608302,0.3431727,0.3247146,0.3054559,
                         0.2853966,0.2853966)),
                list(color="white",  
                     x=c(0.6256911,0.6253051,0.6241468,0.6222164,0.6195138,
                         0.6160391,0.6117922,0.6067731,0.6009819,0.5944185,
                         0.5870829,0.5870829,0.5790867,0.5705415,0.5614471,
                         0.5518036,0.5416111,0.5308694,0.5195786,0.5077388,
                         0.4953499,0.4824118,0.4824118,0.3030982,0.3030982,
                         0.4841277,0.4976921,0.5105872,0.5228132,0.5343699,
                         0.5452574,0.5554757,0.5650248,0.5739047,0.5821153,
                         0.5896568,0.5896568,0.5965033,0.6026292,0.6080343,
                         0.6127188,0.6166826,0.6199256,0.622448,0.6242498,
                         0.6253308,0.6256911,0.6256911,NA,0.6951859,
                         0.69504,0.6946025,0.6938732,0.6928522,0.6915396,
                         0.6899352,0.6880391,0.6858513,0.6833718,0.6806006,
                         0.6806006,0.6775806,0.6743546,0.6709228,0.667285,
                         0.6634414,0.6593918,0.6551363,0.6506749,0.6460076,
                         0.6411344,0.6411344,0.636244,0.6311821,0.6259485,
                         0.6205434,0.6149666,0.6092183,0.6032984,0.5972069,
                         0.5909438,0.5845091,0.5845091,0.5781258,0.571674,
                         0.5651535,0.5585643,0.5519066,0.5451802,0.5383851,
                         0.5315214,0.5245891,0.5175882,0.5175882,0.3030982,
                         0.3030982,0.5235939,0.5304233,0.537184,0.5438761,
                         0.5504995,0.5570543,0.5635405,0.5699581,0.576307,
                         0.5825872,0.5887989,0.5887989,0.595062,0.6011535,
                         0.6070734,0.6128217,0.6183985,0.6238036,0.6290372,
                         0.6340991,0.6389895,0.6437083,0.6437083,0.6484099,
                         0.6529056,0.6571954,0.6612793,0.6651573,0.6688294,
                         0.6722955,0.6755558,0.6786101,0.6814585,0.6814585,
                         0.6840667,0.6864004,0.6884595,0.690244,0.6917541,
                         0.6929895,0.6939504,0.6946368,0.6950486,0.6951859,
                         0.6951859),
                     y=c(0.7190511,0.7315715,0.7433358,0.754344,0.764596,
                         0.7740919,0.7828317,0.7908154,0.798043,0.8045145,
                         0.8102298,0.8102298,0.8154411,0.8201038,0.8242179,
                         0.8277835,0.8308006,0.8332691,0.835189,0.8365604,
                         0.8373832,0.8376575,0.8376575,0.8376575,0.6011861,
                         0.6011861,0.6014826,0.6023721,0.6038547,0.6059303,
                         0.608599,0.6118606,0.6157153,0.6201631,0.6252039,
                         0.6308377,0.6308377,0.6370571,0.6438547,0.6512305,
                         0.6591846,0.6677168,0.6768273,0.6865159,0.6967828,
                         0.7076279,0.7190511,0.7190511,NA,0.3039288,
                         0.3115122,0.3188436,0.3259229,0.3327502,0.3393254,
                         0.3456486,0.3517198,0.3575389,0.363106,0.3684211,
                         0.3684211,0.3735063,0.378384,0.3830541,0.3875167,
                         0.3917717,0.3958191,0.399659,0.4032913,0.4067161,
                         0.4099333,0.4099333,0.4130986,0.4160712,0.418851,
                         0.4214381,0.4238325,0.4260341,0.428043,0.4298592,
                         0.4314826,0.4329133,0.4329133,0.4341809,0.435315,
                         0.4363158,0.4371831,0.437917,0.4385174,0.4389844,
                         0.439318,0.4395182,0.4395849,0.4395849,0.4395849,
                         0.1623425,0.1623425,0.1624092,0.1626093,0.1629429,
                         0.1634099,0.1640104,0.1647443,0.1656116,0.1666123,
                         0.1677465,0.1690141,0.1690141,0.1704374,0.1720385,
                         0.1738176,0.1757746,0.1779096,0.1802224,0.1827131,
                         0.1853818,0.1882283,0.1912528,0.1912528,0.1946256,
                         0.1982209,0.2020385,0.2060786,0.210341,0.2148258,
                         0.219533,0.2244626,0.2296145,0.2349889,0.2349889,
                         0.2406153,0.2465234,0.2527131,0.2591846,0.2659377,
                         0.2729726,0.2802891,0.2878873,0.2957672,0.3039288,
                         0.3039288)),
                list(color="white",  
                     x=c(0.6951859,0.69504,0.6946025,0.6938732,0.6928522,
                         0.6915396,0.6899352,0.6880391,0.6858513,0.6833718,
                         0.6806006,0.6806006,0.6775806,0.6743546,0.6709228,
                         0.667285,0.6634414,0.6593918,0.6551363,0.6506749,
                         0.6460076,0.6411344,0.6411344,0.636244,0.6311821,
                         0.6259485,0.6205434,0.6149666,0.6092183,0.6032984,
                         0.5972069,0.5909438,0.5845091,0.5845091,0.5781258,
                         0.571674,0.5651535,0.5585643,0.5519066,0.5451802,
                         0.5383851,0.5315214,0.5245891,0.5175882,0.5175882,
                         0.3030982,0.3030982,0.5235939,0.5304233,0.537184,
                         0.5438761,0.5504995,0.5570543,0.5635405,0.5699581,
                         0.576307,0.5825872,0.5887989,0.5887989,0.595062,
                         0.6011535,0.6070734,0.6128217,0.6183985,0.6238036,
                         0.6290372,0.6340991,0.6389895,0.6437083,0.6437083,
                         0.6484099,0.6529056,0.6571954,0.6612793,0.6651573,
                         0.6688294,0.6722955,0.6755558,0.6786101,0.6814585,
                         0.6814585,0.6840667,0.6864004,0.6884595,0.690244,
                         0.6917541,0.6929895,0.6939504,0.6946368,0.6950486,
                         0.6951859,0.6951859),
                     y=c(0.3039288,0.3115122,0.3188436,0.3259229,0.3327502,
                         0.3393254,0.3456486,0.3517198,0.3575389,0.363106,
                         0.3684211,0.3684211,0.3735063,0.378384,0.3830541,
                         0.3875167,0.3917717,0.3958191,0.399659,0.4032913,
                         0.4067161,0.4099333,0.4099333,0.4130986,0.4160712,
                         0.418851,0.4214381,0.4238325,0.4260341,0.428043,
                         0.4298592,0.4314826,0.4329133,0.4329133,0.4341809,
                         0.435315,0.4363158,0.4371831,0.437917,0.4385174,
                         0.4389844,0.439318,0.4395182,0.4395849,0.4395849,
                         0.4395849,0.1623425,0.1623425,0.1624092,0.1626093,
                         0.1629429,0.1634099,0.1640104,0.1647443,0.1656116,
                         0.1666123,0.1677465,0.1690141,0.1690141,0.1704374,
                         0.1720385,0.1738176,0.1757746,0.1779096,0.1802224,
                         0.1827131,0.1853818,0.1882283,0.1912528,0.1912528,
                         0.1946256,0.1982209,0.2020385,0.2060786,0.210341,
                         0.2148258,0.219533,0.2244626,0.2296145,0.2349889,
                         0.2349889,0.2406153,0.2465234,0.2527131,0.2591846,
                         0.2659377,0.2729726,0.2802891,0.2878873,0.2957672,
                         0.3039288,0.3039288))
                ),
         C=list(list(color="black", 
                     x=c(0.2972527,0.2978297,0.2995604,0.3024451,0.3064835,
                         0.3116758,0.318022,0.325522,0.3341758,0.3439835,
                         0.3549451,0.3549451,0.3670522,0.3802967,0.3946786,
                         0.4101978,0.4268544,0.4446484,0.4635797,0.4836484,
                         0.5048544,0.5271978,0.5271978,0.5345495,0.5417692,
                         0.5488571,0.5558132,0.5626374,0.5693297,0.5758901,
                         0.5823187,0.5886154,0.5947802,0.5947802,0.6008132,
                         0.6067143,0.6124835,0.6181209,0.6236264,0.629,
                         0.6342418,0.6393516,0.6443297,0.6491758,0.6491758,
                         0.6540549,0.6588022,0.6634176,0.6679011,0.6722527,
                         0.6764725,0.6805604,0.6845165,0.6883407,0.692033,
                         0.692033,0.6957747,0.6994176,0.7029615,0.7064066,
                         0.7097527,0.713,0.7161484,0.7191978,0.7221484,
                         0.725,0.725,0.95,0.9447582,0.9392527,
                         0.9334835,0.9274505,0.9211538,0.9145934,0.9077692,
                         0.9006813,0.8933297,0.8857143,0.8857143,0.8779835,
                         0.869956,0.8616319,0.853011,0.8440934,0.8348791,
                         0.8253681,0.8155604,0.805456,0.7950549,0.7950549,
                         0.7845137,0.7736593,0.7624918,0.751011,0.739217,
                         0.7271099,0.7146896,0.701956,0.6889093,0.6755495,
                         0.6755495,0.6620247,0.6481538,0.6339368,0.6193736,
                         0.6044643,0.5892088,0.5736071,0.5576593,0.5413654,
                         0.5247253,0.5247253,0.5000247,0.476033,0.45275,
                         0.4301758,0.4083104,0.3871538,0.366706,0.346967,
                         0.3279368,0.3096154,0.3096154,0.2919698,0.274967,
                         0.2586071,0.2428901,0.2278159,0.2133846,0.1995962,
                         0.1864505,0.1739478,0.1620879,0.1620879,0.1508544,
                         0.1402308,0.130217,0.1208132,0.1120192,0.1038352,
                         0.096261,0.0892967,0.0829423,0.0771978,0.0771978,
                         0.0720302,0.0674066,0.0633269,0.0597912,0.0567995,
                         0.0543516,0.0524478,0.0510879,0.050272,0.05,
                         0.05,0.0502802,0.0511209,0.052522,0.0544835,
                         0.0570055,0.0600879,0.0637308,0.0679341,0.0726978,
                         0.078022,0.078022,0.0839231,0.0904176,0.0975055,
                         0.1051868,0.1134615,0.1223297,0.1317912,0.1418462,
                         0.1524945,0.1637363,0.1637363,0.1757363,0.1883297,
                         0.2015165,0.2152967,0.2296703,0.2446374,0.2601978,
                         0.2763516,0.2930989,0.3104396,0.3104396,0.3283901,
                         0.346967,0.3661703,0.386,0.406456,0.4275385,
                         0.4492473,0.4715824,0.494544,0.5181319,0.5181319,
                         0.5350769,0.5516264,0.5677802,0.5835385,0.5989011,
                         0.6138681,0.6284396,0.6426154,0.6563956,0.6697802,
                         0.6697802,0.6829588,0.6957912,0.7082775,0.7204176,
                         0.7322115,0.7436593,0.754761,0.7655165,0.7759258,
                         0.785989,0.785989,0.7958874,0.8054725,0.8147445,
                         0.8237033,0.8323489,0.8406813,0.8487005,0.8564066,
                         0.8637995,0.8708791,0.8708791,0.8776703,0.8841978,
                         0.8904615,0.8964615,0.9021978,0.9076703,0.9128791,
                         0.9178242,0.9225055,0.9269231,0.9269231,0.7027473,
                         0.7005385,0.6981978,0.6957253,0.6931209,0.6903846,
                         0.6875165,0.6845165,0.6813846,0.6781209,0.6747253,
                         0.6747253,0.6713626,0.6678681,0.6642418,0.6604835,
                         0.6565934,0.6525714,0.6484176,0.6441319,0.6397143,
                         0.6351648,0.6351648,0.6304918,0.6257033,0.6207995,
                         0.6157802,0.6106456,0.6053956,0.6000302,0.5945495,
                         0.5889533,0.5832418,0.5832418,0.5775797,0.5718022,
                         0.5659093,0.5599011,0.5537775,0.5475385,0.5411841,
                         0.5347143,0.5281291,0.5214286,0.5214286,0.4995714,
                         0.4788352,0.4592198,0.4407253,0.4233516,0.4070989,
                         0.391967,0.377956,0.3650659,0.3532967,0.3532967,
                         0.3426484,0.3331209,0.3247143,0.3174286,0.3112637,
                         0.3062198,0.3022967,0.2994945,0.2978132,0.2972527,
                         0.2972527),
                     y=c(0.5043165,0.4726835,0.4425324,0.4138633,0.3866763,
                         0.3609712,0.3367482,0.3140072,0.2927482,0.2729712,
                         0.2546763,0.2546763,0.238,0.2230791,0.2099137,
                         0.1985036,0.1888489,0.1809496,0.1748058,0.1704173,
                         0.1677842,0.1669065,0.1669065,0.167036,0.1674245,
                         0.1680719,0.1689784,0.1701439,0.1715683,0.1732518,
                         0.1751942,0.1773957,0.1798561,0.1798561,0.1825468,
                         0.1854388,0.1885324,0.1918273,0.1953237,0.1990216,
                         0.2029209,0.2070216,0.2113237,0.2158273,0.2158273,
                         0.2206619,0.2256691,0.2308489,0.2362014,0.2417266,
                         0.2474245,0.253295,0.2593381,0.265554,0.2719424,
                         0.2719424,0.2784748,0.2851223,0.2918849,0.2987626,
                         0.3057554,0.3128633,0.3200863,0.3274245,0.3348777,
                         0.342446,0.342446,0.305036,0.2909353,0.2771223,
                         0.2635971,0.2503597,0.2374101,0.2247482,0.2123741,
                         0.2002878,0.1884892,0.1769784,0.1769784,0.1659209,
                         0.1551942,0.1447986,0.1347338,0.125,0.1155971,
                         0.1065252,0.0977842,0.0893741,0.081295,0.081295,
                         0.0736978,0.066446,0.0595396,0.0529784,0.0467626,
                         0.0408921,0.0353669,0.0301871,0.0253525,0.0208633,
                         0.0208633,0.0168993,0.0133525,0.010223,0.0075108,
                         0.0052158,0.0033381,0.0018777,0.0008345,0.0002086,
                         0,0,0.0003597,0.0014388,0.0032374,
                         0.0057554,0.0089928,0.0129496,0.0176259,0.0230216,
                         0.0291367,0.0359712,0.0359712,0.0436043,0.0518273,
                         0.0606403,0.0700432,0.080036,0.0906187,0.1017914,
                         0.113554,0.1259065,0.1388489,0.1388489,0.1523525,
                         0.1663885,0.1809568,0.1960576,0.2116906,0.2278561,
                         0.244554,0.2617842,0.2795468,0.2978417,0.2978417,
                         0.3166115,0.3357986,0.3554029,0.3754245,0.3958633,
                         0.4167194,0.4379928,0.4596835,0.4817914,0.5043165,
                         0.5043165,0.5280647,0.5512518,0.5738777,0.5959424,
                         0.617446,0.6383885,0.6587698,0.6785899,0.6978489,
                         0.7165468,0.7165468,0.7346763,0.7522302,0.7692086,
                         0.7856115,0.8014388,0.8166906,0.8313669,0.8454676,
                         0.8589928,0.8719424,0.8719424,0.884295,0.8960288,
                         0.9071439,0.9176403,0.927518,0.936777,0.9454173,
                         0.9534388,0.9608417,0.9676259,0.9676259,0.973777,
                         0.9792806,0.9841367,0.9883453,0.9919065,0.9948201,
                         0.9970863,0.998705,0.9996763,1,1,
                         0.9997914,0.9991655,0.9981223,0.9966619,0.9947842,
                         0.9924892,0.989777,0.9866475,0.9831007,0.9791367,
                         0.9791367,0.9747986,0.9701295,0.9651295,0.9597986,
                         0.9541367,0.9481439,0.9418201,0.9351655,0.9281799,
                         0.9208633,0.9208633,0.9133741,0.9055827,0.8974892,
                         0.8890935,0.8803957,0.8713957,0.8620935,0.8524892,
                         0.8425827,0.8323741,0.8323741,0.8220576,0.8115396,
                         0.8008201,0.7898993,0.778777,0.7674532,0.7559281,
                         0.7442014,0.7322734,0.7201439,0.7201439,0.6726619,
                         0.679518,0.6862734,0.6929281,0.699482,0.7059353,
                         0.7122878,0.7185396,0.7246906,0.730741,0.7366906,
                         0.7366906,0.7425108,0.7481727,0.7536763,0.7590216,
                         0.7642086,0.7692374,0.7741079,0.7788201,0.7833741,
                         0.7877698,0.7877698,0.7919856,0.796,0.7998129,
                         0.8034245,0.8068345,0.8100432,0.8130504,0.8158561,
                         0.8184604,0.8208633,0.8208633,0.8231871,0.8252662,
                         0.8271007,0.8286906,0.830036,0.8311367,0.8319928,
                         0.8326043,0.8329712,0.8330935,0.8330935,0.832259,
                         0.8297554,0.8255827,0.819741,0.8122302,0.8030504,
                         0.7922014,0.7796835,0.7654964,0.7496403,0.7496403,
                         0.732295,0.7133525,0.6928129,0.6706763,0.6469424,
                         0.6216115,0.5946835,0.5661583,0.536036,0.5043165,
                         0.5043165))
                ),
         D=list(list(color="black", 
                     x=c(0.95,0.949645,0.9485799,0.9468047,0.9443195,
                         0.9411243,0.9372189,0.9326036,0.9272781,0.9212426,
                         0.914497,0.914497,0.9070769,0.8990178,0.8903195,
                         0.8809822,0.8710059,0.8603905,0.8491361,0.8372426,
                         0.8247101,0.8115385,0.8115385,0.7977367,0.7833136,
                         0.7682692,0.7526036,0.7363166,0.7194083,0.7018787,
                         0.6837278,0.6649556,0.6455621,0.6455621,0.6257692,
                         0.6054438,0.5845858,0.5631953,0.5412722,0.5188166,
                         0.4958284,0.4723077,0.4482544,0.4236686,0.4236686,
                         0.05,0.05,0.3704142,0.398142,0.4252308,
                         0.4516805,0.4774911,0.5026627,0.5271953,0.5510888,
                         0.5743432,0.5969586,0.6189349,0.6189349,0.6402456,
                         0.6608639,0.6807899,0.7000237,0.7185651,0.7364142,
                         0.753571,0.7700355,0.7858077,0.8008876,0.8008876,
                         0.8152574,0.8288994,0.8418136,0.854,0.8654586,
                         0.8761893,0.8861923,0.8954675,0.9040148,0.9118343,
                         0.9118343,0.9190858,0.925574,0.9312988,0.9362604,
                         0.9404586,0.9438935,0.9465651,0.9484734,0.9496183,
                         0.95,0.95),
                     y=c(0.5077835,0.4831727,0.4591549,0.4357302,0.4128984,
                         0.3906597,0.3690141,0.3479615,0.3275019,0.3076353,
                         0.2883617,0.2883617,0.2696887,0.2516234,0.234166,
                         0.2173165,0.2010749,0.1854411,0.1704151,0.155997,
                         0.1421868,0.1289844,0.1289844,0.1165456,0.1047294,
                         0.093536,0.0829652,0.073017,0.0636916,0.0549889,
                         0.0469088,0.0394514,0.0326168,0.0326168,0.0264196,
                         0.0208747,0.0159822,0.011742,0.0081542,0.0052187,
                         0.0029355,0.0013047,0.0003262,0,0,
                         0,1,1,0.9997109,0.9988436,
                         0.9973981,0.9953744,0.9927724,0.9895923,0.985834,
                         0.9814974,0.9765827,0.9710897,0.9710897,0.9649963,
                         0.9582802,0.9509414,0.94298,0.9343958,0.925189,
                         0.9153595,0.9049073,0.8938325,0.8821349,0.8821349,
                         0.8697924,0.8567828,0.843106,0.828762,0.8137509,
                         0.7980726,0.7817272,0.7647146,0.7470348,0.7286879,
                         0.7286879,0.7097999,0.6902001,0.6698888,0.6488658,
                         0.6271312,0.604685,0.5815271,0.5576575,0.5330764,
                         0.5077835,0.5077835)),
                list(color="white",  
                     x=c(0.6863905,0.6861953,0.6856095,0.6846331,0.6832663,
                         0.6815089,0.6793609,0.6768225,0.6738935,0.670574,
                         0.6668639,0.6668639,0.6627544,0.6582367,0.6533107,
                         0.6479763,0.6422337,0.6360828,0.6295237,0.6225562,
                         0.6151805,0.6073964,0.6073964,0.5993728,0.5909231,
                         0.5820473,0.5727456,0.5630178,0.5528639,0.542284,
                         0.5312781,0.5198462,0.5079882,0.5079882,0.4957041,
                         0.4829941,0.469858,0.4562959,0.4423077,0.4278935,
                         0.4130533,0.397787,0.3820947,0.3659763,0.3659763,
                         0.3118343,0.3118343,0.4023669,0.4303876,0.4569349,
                         0.4820089,0.5056095,0.5277367,0.5483905,0.567571,
                         0.5852781,0.6015118,0.6162722,0.6162722,0.6295947,
                         0.6415148,0.6520325,0.6611479,0.6688609,0.6751716,
                         0.6800799,0.6835858,0.6856893,0.6863905,0.6863905),
                     y=c(0.5077835,0.5248703,0.5414381,0.557487,0.573017,
                         0.5880282,0.6025204,0.6164937,0.6299481,0.6428836,
                         0.6553002,0.6553002,0.6673758,0.6789918,0.6901483,
                         0.7008451,0.7110823,0.7208599,0.7301779,0.7390363,
                         0.7474351,0.7553744,0.7553744,0.763017,0.7702298,
                         0.7770126,0.7833655,0.7892884,0.7947813,0.7998443,
                         0.8044774,0.8086805,0.8124537,0.8124537,0.8159748,
                         0.8191253,0.8219051,0.8243143,0.8263529,0.8280208,
                         0.829318,0.8302446,0.8308006,0.8309859,0.8309859,
                         0.8309859,0.1690141,0.1690141,0.1698592,0.1723944,
                         0.1766197,0.1825352,0.1901408,0.1994366,0.2104225,
                         0.2230986,0.2374648,0.2535211,0.2535211,0.2714085,
                         0.2909711,0.312209,0.3351223,0.3597109,0.3859748,
                         0.413914,0.4435285,0.4748184,0.5077835,0.5077835))
                ),
         E=list(list(color="black", 
                     x=c(0.05,0.05,0.9134291,0.9134291,0.3131318,
                         0.3131318,0.8590188,0.8590188,0.3131318,0.3131318,
                         0.95,0.95,0.05),
                     y=c(0,1,1,0.8309859,0.8309859,
                         0.590808,0.590808,0.4217939,0.4217939,0.1690141,
                         0.1690141,0,0))
                ),
         F=list(list(color="black", 
                     x=c(0.3327476,0.3327476,0.9279553,0.9279553,0.3327476,
                         0.3327476,0.05,0.05,0.95,0.95,
                         0.3327476),
                     y=c(0.8309859,0.5463306,0.5463306,0.3773165,0.3773165,
                         0,0,1,1,0.8309859,
                         0.8309859))
                ),
         G=list(list(color="black", 
                     x=c(0.95,0.9433616,0.9365,0.9294151,0.9221069,
                         0.9145754,0.9068206,0.8988426,0.8906412,0.8822166,
                         0.8735687,0.8735687,0.8648693,0.8559466,0.8468006,
                         0.8374313,0.8278387,0.8180229,0.8079838,0.7977214,
                         0.7872357,0.7765267,0.7765267,0.7657834,0.7548511,
                         0.74373,0.7324198,0.7209208,0.7092328,0.6973559,
                         0.6852901,0.6730353,0.6605916,0.6605916,0.6479676,
                         0.6351718,0.6222042,0.6090649,0.5957538,0.582271,
                         0.5686164,0.5547901,0.540792,0.5266221,0.5266221,
                         0.5027137,0.4794237,0.4567519,0.4346985,0.4132634,
                         0.3924466,0.3722481,0.3526679,0.3337061,0.3153626,
                         0.3153626,0.2978006,0.2808397,0.26448,0.2487214,
                         0.2335639,0.2190076,0.2050525,0.1916985,0.1789456,
                         0.1667939,0.1667939,0.1552433,0.1442939,0.1339456,
                         0.1241985,0.1150525,0.1065076,0.0985639,0.0912214,
                         0.08448,0.0783397,0.0783397,0.0729552,0.0681374,
                         0.0638865,0.0602023,0.0570849,0.0545344,0.0525506,
                         0.0511336,0.0502834,0.05,0.05,0.0502834,
                         0.0511336,0.0525506,0.0545344,0.0570849,0.0602023,
                         0.0638865,0.0681374,0.0729552,0.0783397,0.0783397,
                         0.0843082,0.0908779,0.0980487,0.1058206,0.1141937,
                         0.1231679,0.1327433,0.1429198,0.1536975,0.1650763,
                         0.1650763,0.1772366,0.1900153,0.2034122,0.2174275,
                         0.2320611,0.247313,0.2631832,0.2796718,0.2967786,
                         0.3145038,0.3145038,0.3328817,0.3519466,0.3716985,
                         0.3921374,0.4132634,0.4350763,0.4575763,0.4807634,
                         0.5046374,0.5291985,0.5291985,0.5461594,0.5626908,
                         0.5787929,0.5944656,0.609709,0.6245229,0.6389074,
                         0.6528626,0.6663884,0.6794847,0.6794847,0.6923578,
                         0.7048702,0.7170219,0.728813,0.7402433,0.751313,
                         0.7620219,0.7723702,0.7823578,0.7919847,0.7919847,
                         0.8014485,0.8106031,0.8194485,0.8279847,0.8362118,
                         0.8441298,0.8517385,0.8590382,0.8660286,0.8727099,
                         0.8727099,0.8791164,0.8852824,0.891208,0.8968931,
                         0.9023378,0.907542,0.9125057,0.917229,0.9217118,
                         0.9259542,0.9259542,0.6906489,0.6858397,0.680687,
                         0.6751908,0.6693511,0.6631679,0.6566412,0.649771,
                         0.6425573,0.635,0.6270992,0.6270992,0.6188807,
                         0.6103702,0.6015677,0.5924733,0.5830868,0.5734084,
                         0.563438,0.5531756,0.5426212,0.5317748,0.5317748,
                         0.5195973,0.5077977,0.496376,0.4853321,0.474666,
                         0.4643779,0.4544676,0.4449351,0.4357805,0.4270038,
                         0.4270038,0.4185792,0.4104809,0.402709,0.3952634,
                         0.3881441,0.3813511,0.3748845,0.3687443,0.3629303,
                         0.3574427,0.3574427,0.3522643,0.3473779,0.3427834,
                         0.3384809,0.3344704,0.3307519,0.3273254,0.3241908,
                         0.3213483,0.3187977,0.3187977,0.3165134,0.3144695,
                         0.312666,0.3111031,0.3097805,0.3086985,0.3078569,
                         0.3072557,0.306895,0.3067748,0.3067748,0.3073502,
                         0.3090763,0.3119532,0.3159809,0.3211594,0.3274885,
                         0.3349685,0.3435992,0.3533807,0.364313,0.364313,
                         0.3764218,0.3897328,0.4042462,0.4199618,0.4368798,
                         0.455,0.4743225,0.4948473,0.5165744,0.5395038,
                         0.5395038,0.5459876,0.5523855,0.5586975,0.5649237,
                         0.5710639,0.5771183,0.5830868,0.5889695,0.5947662,
                         0.6004771,0.6004771,0.6062481,0.6118817,0.6173779,
                         0.6227366,0.627958,0.633042,0.6379885,0.6427977,
                         0.6474695,0.6520038,0.6520038,0.6565725,0.6610038,
                         0.6652977,0.6694542,0.6734733,0.677355,0.6810992,
                         0.6847061,0.6881756,0.6915076,0.6915076,0.6948655,
                         0.6980687,0.7011174,0.7040115,0.706751,0.7093359,
                         0.7117662,0.714042,0.7161632,0.7181298,0.7181298,
                         0.7181298,0.4828244,0.4828244,0.95,0.95),
                     y=c(0.147482,0.1428489,0.1381583,0.1334101,0.1286043,
                         0.123741,0.1188201,0.1138417,0.1088058,0.1037122,
                         0.0985612,0.0985612,0.0934173,0.0883453,0.0833453,
                         0.0784173,0.0735612,0.068777,0.0640647,0.0594245,
                         0.0548561,0.0503597,0.0503597,0.0461151,0.0420144,
                         0.0380576,0.0342446,0.0305755,0.0270504,0.0236691,
                         0.0204317,0.0173381,0.0143885,0.0143885,0.0116547,
                         0.0092086,0.0070504,0.0051799,0.0035971,0.0023022,
                         0.001295,0.0005755,0.0001439,0,0,
                         0.0003741,0.0014964,0.0033669,0.0059856,0.0093525,
                         0.0134676,0.0183309,0.0239424,0.0303022,0.0374101,
                         0.0374101,0.0453237,0.0538129,0.0628777,0.072518,
                         0.0827338,0.0935252,0.1048921,0.1168345,0.1293525,
                         0.142446,0.142446,0.1560791,0.1702158,0.1848561,
                         0.2,0.2156475,0.2317986,0.2484532,0.2656115,
                         0.2832734,0.3014388,0.3014388,0.3201727,0.3392518,
                         0.3586763,0.378446,0.3985612,0.4190216,0.4398273,
                         0.4609784,0.4824748,0.5043165,0.5043165,0.5280647,
                         0.5512518,0.5738777,0.5959424,0.617446,0.6383885,
                         0.6587698,0.6785899,0.6978489,0.7165468,0.7165468,
                         0.7348058,0.7524604,0.7695108,0.7859568,0.8017986,
                         0.817036,0.8316691,0.8456978,0.8591223,0.8719424,
                         0.8719424,0.884295,0.8960288,0.9071439,0.9176403,
                         0.927518,0.936777,0.9454173,0.9534388,0.9608417,
                         0.9676259,0.9676259,0.973777,0.9792806,0.9841367,
                         0.9883453,0.9919065,0.9948201,0.9970863,0.998705,
                         0.9996763,1,1,0.9998058,0.999223,
                         0.9982518,0.9968921,0.9951439,0.9930072,0.990482,
                         0.9875683,0.9842662,0.9805755,0.9805755,0.9765252,
                         0.9721439,0.9674317,0.9623885,0.9570144,0.9513094,
                         0.9452734,0.9389065,0.9322086,0.9251799,0.9251799,
                         0.9179856,0.9105036,0.9027338,0.8946763,0.8863309,
                         0.8776978,0.868777,0.8595683,0.8500719,0.8402878,
                         0.8402878,0.8302518,0.82,0.8095324,0.7988489,
                         0.7879496,0.7768345,0.7655036,0.7539568,0.7421942,
                         0.7302158,0.7302158,0.6913669,0.7053813,0.7186475,
                         0.7311655,0.7429353,0.7539568,0.7642302,0.7737554,
                         0.7825324,0.7905612,0.7978417,0.7978417,0.8045396,
                         0.8105324,0.8158201,0.8204029,0.8242806,0.8274532,
                         0.8299209,0.8316835,0.832741,0.8330935,0.8330935,
                         0.8328777,0.8322302,0.8311511,0.8296403,0.8276978,
                         0.8253237,0.822518,0.8192806,0.8156115,0.8115108,
                         0.8115108,0.8071295,0.8023309,0.7971151,0.791482,
                         0.7854317,0.778964,0.7720791,0.764777,0.7570576,
                         0.7489209,0.7489209,0.7405108,0.7316835,0.7224388,
                         0.712777,0.7026978,0.6922014,0.6812878,0.6699568,
                         0.6582086,0.6460432,0.6460432,0.6336187,0.6208058,
                         0.6076043,0.5940144,0.580036,0.5656691,0.5509137,
                         0.5357698,0.5202374,0.5043165,0.5043165,0.472964,
                         0.4430791,0.4146619,0.3877122,0.3622302,0.3382158,
                         0.3156691,0.2945899,0.2749784,0.2568345,0.2568345,
                         0.240295,0.2254964,0.2124388,0.2011223,0.1915468,
                         0.1837122,0.1776187,0.1732662,0.1706547,0.1697842,
                         0.1697842,0.1698417,0.1700144,0.1703022,0.170705,
                         0.171223,0.1718561,0.1726043,0.1734676,0.174446,
                         0.1755396,0.1755396,0.1768489,0.1781871,0.179554,
                         0.1809496,0.1823741,0.1838273,0.1853094,0.1868201,
                         0.1883597,0.1899281,0.1899281,0.1916547,0.1933813,
                         0.1951079,0.1968345,0.1985612,0.2002878,0.2020144,
                         0.203741,0.2054676,0.2071942,0.2071942,0.209036,
                         0.2108201,0.2125468,0.2142158,0.2158273,0.2173813,
                         0.2188777,0.2203165,0.2216978,0.2230216,0.2230216,
                         0.3920863,0.3920863,0.5489209,0.5489209,0.147482))
                ),
         H=list(list(color="black", 
                     x=c(0.6714061,0.6714061,0.3285939,0.3285939,0.05,
                         0.05,0.3285939,0.3285939,0.6714061,0.6714061,
                         0.95,0.95,0.6714061),
                     y=c(0,0.4255004,0.4255004,0,0,
                         1,1,0.6063751,0.6063751,1,
                         1,0,0))
                ),
         I=list(list(color="black", 
                     x=c(0.164878,0.835122,0.835122,0.607561,0.607561,
                         0.835122,0.835122,0.164878,0.164878,0.392439,
                         0.392439,0.164878,0.164878),
                     y=c(1,1,0.8309859,0.8309859,0.1690141,
                         0.1690141,0,0,0.1690141,0.1690141,
                         0.8309859,0.8309859,1))
                ),
         J=list(list(color="black", 
                     x=c(0.5034351,0.4841985,0.4653544,0.4469029,0.4288441,
                         0.4111778,0.393904,0.3770229,0.3605344,0.3444384,
                         0.328735,0.328735,0.313434,0.2985453,0.2840687,
                         0.2700044,0.2563522,0.2431123,0.2302846,0.2178691,
                         0.2058659,0.1942748,0.1942748,0.183096,0.1723293,
                         0.1619749,0.1520327,0.1425027,0.133385,0.1246794,
                         0.116386,0.1085049,0.101036,0.101036,0.0939891,
                         0.087374,0.0811908,0.0754395,0.07012,0.0652323,
                         0.0607764,0.0567525,0.0531603,0.05,0.05,
                         0.3356052,0.3397764,0.3444384,0.3495911,0.3552345,
                         0.3613686,0.3679935,0.3751091,0.3827154,0.3908124,
                         0.3994002,0.3994002,0.40841,0.4177732,0.4274896,
                         0.4375594,0.4479826,0.458759,0.4698888,0.4813719,
                         0.4932083,0.505398,0.505398,0.5194329,0.532879,
                         0.5457361,0.5580044,0.5696838,0.5807743,0.5912759,
                         0.6011887,0.6105125,0.6192475,0.6192475,0.6272661,
                         0.6344406,0.640771,0.6462574,0.6508997,0.6546979,
                         0.6576521,0.6597623,0.6610284,0.6614504,0.6614504,
                         0.6614504,0.3660305,0.3660305,0.95,0.95,
                         0.9496957,0.948783,0.9472617,0.945132,0.9423937,
                         0.9390469,0.9350916,0.9305278,0.9253555,0.9195747,
                         0.9195747,0.9134013,0.9066587,0.8993468,0.8914656,
                         0.8830153,0.8739956,0.8644068,0.8542486,0.8435213,
                         0.8322246,0.8322246,0.8205747,0.8083948,0.7956848,
                         0.7824449,0.768675,0.7543751,0.7395453,0.7241854,
                         0.7082955,0.6918757,0.6918757,0.6751516,0.6579564,
                         0.6402901,0.6221527,0.6035442,0.5844646,0.5649138,
                         0.544892,0.5243991,0.5034351,0.5034351),
                     y=c(0,0.0002045,0.0008181,0.0018408,0.0032725,
                         0.0051132,0.007363,0.0100219,0.0130898,0.0165668,
                         0.0204529,0.0204529,0.0247334,0.0293937,0.0344339,
                         0.0398539,0.0456538,0.0518335,0.058393,0.0653324,
                         0.0726516,0.0803506,0.0803506,0.0884149,0.0968298,
                         0.1055953,0.1147115,0.1241782,0.1339956,0.1441636,
                         0.1546822,0.1655515,0.1767714,0.1767714,0.1884733,
                         0.2004967,0.2128415,0.2255077,0.2384953,0.2518042,
                         0.2654346,0.2793864,0.2936596,0.3082542,0.3082542,
                         0.339664,0.3228634,0.3069394,0.2918919,0.277721,
                         0.2644266,0.2520088,0.2404675,0.2298028,0.2200146,
                         0.211103,0.211103,0.2030533,0.195851,0.189496,
                         0.1839883,0.179328,0.175515,0.1725493,0.170431,
                         0.16916,0.1687363,0.1687363,0.1691673,0.1704602,
                         0.172615,0.1756318,0.1795106,0.1842513,0.1898539,
                         0.1963185,0.203645,0.2118335,0.2118335,0.2208254,
                         0.2305625,0.2410446,0.2522717,0.264244,0.2769613,
                         0.2904237,0.3046311,0.3195836,0.3352812,0.3352812,
                         0.8312637,0.8312637,1,1,0.3403944,
                         0.3249087,0.3097151,0.2948137,0.2802045,0.2658875,
                         0.2518627,0.23813,0.2246896,0.2115413,0.1986852,
                         0.1986852,0.1863039,0.1742878,0.162637,0.1513514,
                         0.140431,0.1298758,0.1196859,0.1098612,0.1004018,
                         0.0913075,0.0913075,0.0827392,0.0745654,0.066786,
                         0.059401,0.0524105,0.0458145,0.0396129,0.0338057,
                         0.028393,0.0233747,0.0233747,0.0189335,0.0149598,
                         0.0114536,0.0084149,0.0058437,0.00374,0.0021037,
                         0.000935,0.0002337,0,0))
                ),
         K=list(list(color="black", 
                     x=c(0.6634492,0.397484,0.2929094,0.2929094,0.05,
                         0.05,0.2929094,0.2929094,0.6519213,0.934355,
                         0.555581,0.95,0.6634492),
                     y=c(0,0.4128984,0.3076353,0,0,
                         1,1,0.5559674,1,1,
                         0.5707932,0,0))
                ),
         L=list(list(color="black", 
                     x=c(0.05,0.05,0.341438,0.341438,0.95,
                         0.95,0.05),
                     y=c(0,1,1,0.1690141,0.1690141,
                         0,0))
                ),
         M=list(list(color="black", 
                     x=c(0.7562076,0.7562076,0.756225,0.7562774,0.7563647,
                         0.7564869,0.756644,0.7568361,0.757063,0.7573249,
                         0.7576217,0.7579534,0.7579534,0.7584772,0.759001,
                         0.7595247,0.7600485,0.7605723,0.761096,0.7616198,
                         0.7621435,0.7626673,0.7631911,0.7631911,0.7638894,
                         0.7645878,0.7652861,0.7659845,0.7666828,0.7673812,
                         0.7680795,0.7687779,0.7694762,0.7701746,0.7701746,
                         0.7677391,0.765321,0.7629205,0.7605373,0.7581717,
                         0.7558235,0.7534927,0.7511794,0.7488836,0.7466052,
                         0.7466052,0.7455577,0.7445102,0.7434627,0.7424151,
                         0.7413676,0.7403201,0.7392726,0.738225,0.7371775,
                         0.73613,0.73613,0.7350824,0.7340349,0.7329874,
                         0.7319399,0.7308923,0.7298448,0.7287973,0.7277498,
                         0.7267022,0.7256547,0.7256547,0.7246072,0.7235597,
                         0.7225121,0.7214646,0.7204171,0.7193695,0.718322,
                         0.7172745,0.716227,0.7151794,0.7151794,0.7141319,
                         0.7130844,0.7120369,0.7109893,0.7099418,0.7088943,
                         0.7078468,0.7067992,0.7057517,0.7047042,0.7047042,
                         0.5964597,0.4061591,0.2961688,0.295453,0.2947022,
                         0.2939166,0.293096,0.2922405,0.2913501,0.2904248,
                         0.2894646,0.2884694,0.2874394,0.2874394,0.2863919,
                         0.2853443,0.2842968,0.2832493,0.2822017,0.2811542,
                         0.2801067,0.2790592,0.2780116,0.2769641,0.2769641,
                         0.2759079,0.2748341,0.273743,0.2726343,0.2715082,
                         0.2703647,0.2692037,0.2680252,0.2668293,0.2656159,
                         0.2656159,0.2645684,0.2635209,0.2624733,0.2614258,
                         0.2603783,0.2593307,0.2582832,0.2572357,0.2561882,
                         0.2551406,0.2551406,0.2526877,0.2502173,0.2477294,
                         0.2452241,0.2427013,0.240161,0.2376033,0.2350281,
                         0.2324355,0.2298254,0.2298254,0.2303579,0.2309079,
                         0.2314753,0.2320601,0.2326625,0.2332823,0.2339195,
                         0.2345742,0.2352464,0.235936,0.235936,0.236451,
                         0.2369486,0.2374287,0.2378914,0.2383366,0.2387643,
                         0.2391746,0.2395674,0.2399428,0.2403007,0.2403007,
                         0.2406324,0.2409292,0.2411911,0.241418,0.2416101,
                         0.2417672,0.2418894,0.2419767,0.2420291,0.2420466,
                         0.2420466,0.2420466,0.05,0.05,0.3529098,
                         0.4707565,0.4721358,0.4734801,0.4747895,0.476064,
                         0.4773036,0.4785082,0.479678,0.4808128,0.4819127,
                         0.4829777,0.4829777,0.4841823,0.4853521,0.4864869,
                         0.4875868,0.4886518,0.4896819,0.490677,0.4916372,
                         0.4925626,0.493453,0.493453,0.4945005,0.495548,
                         0.4965955,0.4976431,0.4986906,0.4997381,0.5007856,
                         0.5018332,0.5028807,0.5039282,0.5039282,0.5049845,
                         0.5060582,0.5071494,0.508258,0.5093841,0.5105276,
                         0.5116887,0.5128671,0.514063,0.5152764,0.5152764,
                         0.5163327,0.5174064,0.5184976,0.5196062,0.5207323,
                         0.5218758,0.5230369,0.5242153,0.5254113,0.5266246,
                         0.5266246,0.5278468,0.5290689,0.530291,0.5315131,
                         0.5327352,0.5339573,0.5351794,0.5364016,0.5376237,
                         0.5388458,0.5388458,0.6575655,0.95,0.95,
                         0.7562076),
                     y=c(0,0.6434396,0.6510156,0.6586212,0.6662565,
                         0.6739214,0.681616,0.6893403,0.6970941,0.7048777,
                         0.7126909,0.7205337,0.7205337,0.7283247,0.7359822,
                         0.7435063,0.750897,0.7581542,0.765278,0.7722683,
                         0.7791253,0.7858488,0.7924388,0.7924388,0.7999852,
                         0.8075019,0.8149889,0.8224463,0.829874,0.8372721,
                         0.8446405,0.8519792,0.8592884,0.8665678,0.8665678,
                         0.8563751,0.8462565,0.836212,0.8262417,0.8163454,
                         0.8065234,0.7967754,0.7871016,0.7775019,0.7679763,
                         0.7679763,0.7639585,0.759911,0.755834,0.7517272,
                         0.7475908,0.7434248,0.7392291,0.7350037,0.7307487,
                         0.726464,0.726464,0.722172,0.7178947,0.7136323,
                         0.7093847,0.705152,0.700934,0.6967309,0.6925426,
                         0.6883692,0.6842105,0.6842105,0.6802224,0.6762639,
                         0.6723351,0.6684359,0.6645663,0.6607265,0.6569162,
                         0.6531357,0.6493847,0.6456635,0.6456635,0.6421423,
                         0.6386953,0.6353225,0.6320237,0.6287991,0.6256486,
                         0.6225723,0.6195701,0.616642,0.613788,0.613788,
                         0.2668643,0.2668643,0.613788,0.6160712,0.6184729,
                         0.6209933,0.6236323,0.6263899,0.6292661,0.6322609,
                         0.6353744,0.6386064,0.641957,0.641957,0.6453966,
                         0.6488955,0.6524537,0.6560712,0.659748,0.6634841,
                         0.6672795,0.6711342,0.6750482,0.6790215,0.6790215,
                         0.6830393,0.6870867,0.6911638,0.6952706,0.699407,
                         0.703573,0.7077687,0.7119941,0.7162491,0.7205337,
                         0.7205337,0.7248258,0.729103,0.7333655,0.737613,
                         0.7418458,0.7460638,0.7502669,0.7544552,0.7586286,
                         0.7627872,0.7627872,0.7726316,0.7825945,0.7926761,
                         0.8028762,0.813195,0.8236323,0.8341883,0.8448629,
                         0.855656,0.8665678,0.8665678,0.8594218,0.8522165,
                         0.8449518,0.8376279,0.8302446,0.8228021,0.8153002,
                         0.8077391,0.8001186,0.7924388,0.7924388,0.7857079,
                         0.7788584,0.7718903,0.7648036,0.7575982,0.7502743,
                         0.7428317,0.7352706,0.7275908,0.7197924,0.7197924,
                         0.7120904,0.7044033,0.6967309,0.6890734,0.6814307,
                         0.6738028,0.6661898,0.6585915,0.6510082,0.6434396,
                         0.6434396,0,0,1,1,
                         0.6300964,0.6261453,0.6220015,0.6176649,0.6131357,
                         0.6084136,0.6034989,0.5983914,0.5930912,0.5875982,
                         0.5819125,0.5819125,0.5761453,0.5704077,0.5646998,
                         0.5590215,0.5533729,0.5477539,0.5421646,0.5366049,
                         0.5310749,0.5255745,0.5255745,0.5191698,0.5127057,
                         0.5061824,0.4995997,0.4929577,0.4862565,0.4794959,
                         0.4726761,0.4657969,0.4588584,0.4588584,0.4657969,
                         0.4726761,0.4794959,0.4862565,0.4929577,0.4995997,
                         0.5061824,0.5127057,0.5191698,0.5255745,0.5255745,
                         0.5310675,0.5365752,0.5420979,0.5476353,0.5531875,
                         0.5587546,0.5643365,0.5699333,0.5755448,0.5811712,
                         0.5811712,0.5868569,0.5923499,0.5976501,0.6027576,
                         0.6076723,0.6123944,0.6169236,0.6212602,0.625404,
                         0.6293551,0.6293551,1,1,0,
                         0))
                ),
         N=list(list(color="black", 
                     x=c(0.6252868,0.2649114,0.2663942,0.2678394,0.2692471,
                         0.2706173,0.2719499,0.273245,0.2745026,0.2757226,
                         0.2769051,0.2780501,0.2780501,0.2786038,0.2791387,
                         0.2796548,0.2801522,0.2806309,0.2810907,0.2815318,
                         0.2819541,0.2823577,0.2827424,0.2827424,0.2832961,
                         0.2838311,0.2843472,0.2848446,0.2853233,0.2857831,
                         0.2862242,0.2866465,0.2870501,0.2874348,0.2874348,
                         0.2878008,0.2881481,0.2884765,0.2887862,0.2890772,
                         0.2893493,0.2896027,0.2898373,0.2900532,0.2902503,
                         0.2902503,0.2904286,0.2905881,0.2907289,0.2908509,
                         0.2909541,0.2910386,0.2911043,0.2911512,0.2911794,
                         0.2911887,0.2911887,0.2911887,0.05,0.05,
                         0.3662669,0.7350886,0.7336058,0.7321606,0.7307529,
                         0.7293827,0.7280501,0.726755,0.7254974,0.7242774,
                         0.7230949,0.7219499,0.7219499,0.7210115,0.720073,
                         0.7191345,0.718196,0.7172576,0.7163191,0.7153806,
                         0.7144421,0.7135036,0.7125652,0.7125652,0.7118519,
                         0.7112138,0.7106507,0.7101627,0.7097497,0.7094119,
                         0.7091491,0.7089614,0.7088488,0.7088113,0.7088113,
                         0.7088113,0.95,0.95,0.6252868),
                     y=c(0,0.7487027,0.7380726,0.7275315,0.7170793,
                         0.7067161,0.6964418,0.6862565,0.6761601,0.6661527,
                         0.6562342,0.6464047,0.6464047,0.6422387,0.638043,
                         0.6338176,0.6295626,0.625278,0.6209637,0.6166197,
                         0.6122461,0.6078428,0.6034099,0.6034099,0.5989696,
                         0.5945441,0.5901334,0.5857376,0.5813566,0.5769904,
                         0.572639,0.5683024,0.5639807,0.5596738,0.5596738,
                         0.5555374,0.5514307,0.5473536,0.5433062,0.5392884,
                         0.5353002,0.5313417,0.5274129,0.5235137,0.5196442,
                         0.5196442,0.5159748,0.5123795,0.5088584,0.5054114,
                         0.5020385,0.4987398,0.4955152,0.4923647,0.4892884,
                         0.4862861,0.4862861,0,0,1,
                         1,0.2312824,0.2412083,0.2511193,0.2610156,
                         0.270897,0.2807635,0.2906153,0.3004522,0.3102743,
                         0.3200815,0.329874,0.329874,0.3383617,0.3469236,
                         0.3555597,0.3642698,0.3730541,0.3819125,0.3908451,
                         0.3998517,0.4089325,0.4180875,0.4180875,0.4273314,
                         0.4363825,0.4452409,0.4539066,0.4623795,0.4706597,
                         0.4787472,0.486642,0.494344,0.5018532,0.5018532,
                         1,1,0,0))
                ),
         O=list(list(color="black", 
                     x=c(0.95,0.9488387,0.9453548,0.9395484,0.9314194,
                         0.9209677,0.9081935,0.8930968,0.8756774,0.8559355,
                         0.833871,0.833871,0.8096332,0.7833714,0.7550857,
                         0.724776,0.6924424,0.6580848,0.6217032,0.5832977,
                         0.5428682,0.5004147,0.5004147,0.457953,0.4174986,
                         0.3790516,0.342612,0.3081797,0.2757548,0.2453373,
                         0.2169272,0.1905244,0.166129,0.166129,0.1440645,
                         0.1243226,0.1069032,0.0918065,0.0790323,0.0685806,
                         0.0604516,0.0546452,0.0511613,0.05,0.05,
                         0.0511613,0.0546452,0.0604516,0.0685806,0.0790323,
                         0.0918065,0.1069032,0.1243226,0.1440645,0.166129,
                         0.166129,0.1905244,0.2169272,0.2453373,0.2757548,
                         0.3081797,0.342612,0.3790516,0.4174986,0.457953,
                         0.5004147,0.5004147,0.5428682,0.5832977,0.6217032,
                         0.6580848,0.6924424,0.724776,0.7550857,0.7833714,
                         0.8096332,0.833871,0.833871,0.8559355,0.8756774,
                         0.8930968,0.9081935,0.9209677,0.9314194,0.9395484,
                         0.9453548,0.9488387,0.95,0.95),
                     y=c(0.5028777,0.454446,0.4084317,0.3648345,0.3236547,
                         0.2848921,0.2485468,0.2146187,0.1831079,0.1540144,
                         0.1273381,0.1273381,0.1031439,0.0814964,0.0623957,
                         0.0458417,0.0318345,0.0203741,0.0114604,0.0050935,
                         0.0012734,0,0,0.0012734,0.0050935,
                         0.0114604,0.0203741,0.0318345,0.0458417,0.0623957,
                         0.0814964,0.1031439,0.1273381,0.1273381,0.1540144,
                         0.1831079,0.2146187,0.2485468,0.2848921,0.3236547,
                         0.3648345,0.4084317,0.454446,0.5028777,0.5028777,
                         0.5512806,0.5972086,0.6406619,0.6816403,0.7201439,
                         0.7561727,0.7897266,0.8208058,0.8494101,0.8755396,
                         0.8755396,0.8991871,0.9203453,0.9390144,0.9551942,
                         0.9688849,0.9800863,0.9887986,0.9950216,0.9987554,
                         1,1,0.9987554,0.9950216,0.9887986,
                         0.9800863,0.9688849,0.9551942,0.9390144,0.9203453,
                         0.8991871,0.8755396,0.8755396,0.8494101,0.8208058,
                         0.7897266,0.7561727,0.7201439,0.6816403,0.6406619,
                         0.5972086,0.5512806,0.5028777,0.5028777)),
                list(color="white",  
                     x=c(0.7019816,0.7014756,0.6999576,0.6974276,0.6938857,
                         0.6893318,0.6837659,0.677188,0.6695982,0.6609963,
                         0.6513825,0.6513825,0.640765,0.6291521,0.6165438,
                         0.6029401,0.588341,0.5727465,0.5561567,0.5385714,
                         0.5199908,0.5004147,0.5004147,0.4808304,0.4622249,
                         0.4445982,0.4279502,0.4122811,0.3975908,0.3838793,
                         0.3711465,0.3593926,0.3486175,0.3486175,0.3390037,
                         0.3304018,0.322812,0.3162341,0.3106682,0.3061143,
                         0.3025724,0.3000424,0.2985244,0.2980184,0.2980184,
                         0.2985244,0.3000424,0.3025724,0.3061143,0.3106682,
                         0.3162341,0.322812,0.3304018,0.3390037,0.3486175,
                         0.3486175,0.3593926,0.3711465,0.3838793,0.3975908,
                         0.4122811,0.4279502,0.4445982,0.4622249,0.4808304,
                         0.5004147,0.5004147,0.5199908,0.5385714,0.5561567,
                         0.5727465,0.588341,0.6029401,0.6165438,0.6291521,
                         0.640765,0.6513825,0.6513825,0.6609963,0.6695982,
                         0.677188,0.6837659,0.6893318,0.6938857,0.6974276,
                         0.6999576,0.7014756,0.7019816,0.7019816),
                     y=c(0.5028777,0.5355396,0.5664748,0.5956835,0.6231655,
                         0.6489209,0.6729496,0.6952518,0.7158273,0.7346763,
                         0.7517986,0.7517986,0.7672446,0.7810647,0.793259,
                         0.8038273,0.8127698,0.8200863,0.825777,0.8298417,
                         0.8322806,0.8330935,0.8330935,0.8322662,0.8297842,
                         0.8256475,0.8198561,0.8124101,0.8033094,0.792554,
                         0.7801439,0.7660791,0.7503597,0.7503597,0.7331223,
                         0.7142158,0.6936403,0.6713957,0.647482,0.6218993,
                         0.5946475,0.5657266,0.5351367,0.5028777,0.5028777,
                         0.4705827,0.4398849,0.4107842,0.3832806,0.3573741,
                         0.3330647,0.3103525,0.2892374,0.2697194,0.2517986,
                         0.2517986,0.2356691,0.2212374,0.2085036,0.1974676,
                         0.1881295,0.1804892,0.1745468,0.1703022,0.1677554,
                         0.1669065,0.1669065,0.1677554,0.1703022,0.1745468,
                         0.1804892,0.1881295,0.1974676,0.2085036,0.2212374,
                         0.2356691,0.2517986,0.2517986,0.2697194,0.2892374,
                         0.3103525,0.3330647,0.3573741,0.3832806,0.4107842,
                         0.4398849,0.4705827,0.5028777,0.5028777))
                ),
         P=list(list(color="black", 
                     x=c(0.95,0.9497227,0.9488907,0.947504,0.9455626,
                         0.9430666,0.9400159,0.9364105,0.9322505,0.9275358,
                         0.9222664,0.9222664,0.9166034,0.9103499,0.903506,
                         0.8960716,0.8880467,0.8794314,0.8702256,0.8604294,
                         0.8500427,0.8390656,0.8390656,0.8274891,0.8153042,
                         0.8025109,0.7891093,0.7750994,0.7604811,0.7452545,
                         0.7294195,0.7129761,0.6959245,0.6959245,0.6782465,
                         0.6599245,0.6409583,0.6213479,0.6010934,0.5801948,
                         0.5586521,0.5364652,0.5136342,0.490159,0.490159,
                         0.3139165,0.3139165,0.05,0.05,0.4794235,
                         0.5032565,0.5264453,0.5489901,0.5708907,0.5921471,
                         0.6127594,0.6327276,0.6520517,0.6707316,0.6887674,
                         0.6887674,0.7061769,0.7229781,0.739171,0.7547555,
                         0.7697316,0.7840994,0.7978588,0.8110099,0.8235527,
                         0.8354871,0.8354871,0.8468131,0.8575308,0.8676402,
                         0.8771412,0.8860338,0.8943181,0.901994,0.9090616,
                         0.9155209,0.9213718,0.9213718,0.9268111,0.9316779,
                         0.9359722,0.9396938,0.9428429,0.9454195,0.9474235,
                         0.9488549,0.9497137,0.95,0.95),
                     y=c(0.6842105,0.6707858,0.6574944,0.6443365,0.6313121,
                         0.6184211,0.6056635,0.5930393,0.5805486,0.5681913,
                         0.5559674,0.5559674,0.5441067,0.5325426,0.521275,
                         0.5103039,0.4996294,0.4892513,0.4791698,0.4693847,
                         0.4598962,0.4507042,0.4507042,0.4418458,0.433358,
                         0.4252409,0.4174944,0.4101186,0.4031134,0.3964789,
                         0.390215,0.3843217,0.3787991,0.3787991,0.3738695,
                         0.3694589,0.3655671,0.3621942,0.3593403,0.3570052,
                         0.355189,0.3538918,0.3531134,0.352854,0.352854,
                         0.352854,0,0,1,1,
                         0.9997776,0.9991105,0.9979985,0.9964418,0.9944403,
                         0.9919941,0.989103,0.9857672,0.9819867,0.9777613,
                         0.9777613,0.9732691,0.9683914,0.9631282,0.9574796,
                         0.9514455,0.9450259,0.9382209,0.9310304,0.9234544,
                         0.915493,0.915493,0.9071609,0.8984729,0.8894292,
                         0.8800297,0.8702743,0.8601631,0.8496961,0.8388732,
                         0.8276946,0.8161601,0.8161601,0.8042995,0.7921423,
                         0.7796887,0.7669385,0.7538918,0.7405486,0.7269088,
                         0.7129726,0.6987398,0.6842105,0.6842105)),
                list(color="white",  
                     x=c(0.6842942,0.6837038,0.6819324,0.6789801,0.6748469,
                         0.6695328,0.6630378,0.6553618,0.646505,0.6364672,
                         0.6252485,0.6252485,0.6130278,0.5996262,0.5850437,
                         0.5692803,0.552336,0.5342107,0.5149046,0.4944175,
                         0.4727495,0.4499006,0.4499006,0.3139165,0.3139165,
                         0.4570577,0.4800408,0.5017535,0.5221958,0.5413678,
                         0.5592694,0.5759006,0.5912614,0.6053519,0.618172,
                         0.6297217,0.6297217,0.6400905,0.6493678,0.6575537,
                         0.6646481,0.6706511,0.6755626,0.6793827,0.6821113,
                         0.6837485,0.6842942,0.6842942),
                     y=c(0.6805041,0.6961972,0.7109563,0.7247813,0.7376723,
                         0.7496294,0.7606523,0.7707413,0.7798962,0.7881171,
                         0.795404,0.795404,0.8020237,0.8079466,0.8131727,
                         0.817702,0.8215345,0.8246701,0.827109,0.828851,
                         0.8298962,0.8302446,0.8302446,0.8302446,0.5211268,
                         0.5211268,0.5215271,0.5227279,0.5247294,0.5275315,
                         0.5311342,0.5355374,0.5407413,0.5467457,0.5535508,
                         0.5611564,0.5611564,0.5696887,0.578977,0.5890215,
                         0.5998221,0.6113788,0.6236916,0.6367606,0.6505856,
                         0.6651668,0.6805041,0.6805041))
                ),
         Q=list(list(color="black", 
                     x=c(0.9409672,0.9401706,0.937781,0.9337984,0.9282226,
                         0.9210538,0.912292,0.901937,0.8899891,0.876448,
                         0.8613139,0.8613139,0.8448166,0.8268577,0.807437,
                         0.7865547,0.7642108,0.7404051,0.7151378,0.6884088,
                         0.6602181,0.6305657,0.6305657,0.6339078,0.637365,
                         0.640937,0.6446241,0.6484261,0.6523431,0.656375,
                         0.6605219,0.6647838,0.6691606,0.6691606,0.6736442,
                         0.6782263,0.6829069,0.6876861,0.6925639,0.6975401,
                         0.702615,0.7077883,0.7130602,0.7184307,0.7184307,
                         0.7240639,0.7297956,0.7356259,0.7415547,0.7475821,
                         0.753708,0.7599325,0.7662555,0.772677,0.7791971,
                         0.7791971,0.7858075,0.7925,0.7992746,0.8061314,
                         0.8130703,0.8200912,0.8271943,0.8343796,0.8416469,
                         0.8489964,0.8489964,0.8537838,0.8586204,0.8635064,
                         0.8684416,0.8734261,0.8784599,0.8835429,0.8886752,
                         0.8938568,0.8990876,0.8990876,0.9043266,0.9095328,
                         0.9147062,0.9198467,0.9249544,0.9300292,0.9350712,
                         0.9400803,0.9450566,0.95,0.95,0.95,
                         0.9410903,0.9320985,0.9230246,0.9138686,0.9046305,
                         0.8953102,0.8859078,0.8764234,0.8668568,0.857208,
                         0.857208,0.8474279,0.8374672,0.8273257,0.8170036,
                         0.8065009,0.7958175,0.7849535,0.7739088,0.7626834,
                         0.7512774,0.7512774,0.7358641,0.7208285,0.7061706,
                         0.6918905,0.6779881,0.6644635,0.6513166,0.6385474,
                         0.626156,0.6141423,0.6141423,0.6026296,0.5914124,
                         0.5804909,0.569865,0.5595347,0.5495,0.5397609,
                         0.5303175,0.5211697,0.5123175,0.5123175,0.5039005,
                         0.4957299,0.4878057,0.4801277,0.4726962,0.4655109,
                         0.4585721,0.4518796,0.4454334,0.4392336,0.4392336,
                         0.4332391,0.4274088,0.4217427,0.4162409,0.4109033,
                         0.4057299,0.4007208,0.3958759,0.3911953,0.3866788,
                         0.3866788,0.3546122,0.3241058,0.2951597,0.2677737,
                         0.241948,0.2176825,0.1949772,0.1738321,0.1542473,
                         0.1362226,0.1362226,0.1198403,0.1051825,0.0922491,
                         0.0810401,0.0715557,0.0637956,0.05776,0.0534489,
                         0.0508622,0.05,0.05,0.0511496,0.0545985,
                         0.0603467,0.0683942,0.0787409,0.0913869,0.1063321,
                         0.1235766,0.1431204,0.1649635,0.1649635,0.1891141,
                         0.2152518,0.2433768,0.2734891,0.3055885,0.3396752,
                         0.3757491,0.4138102,0.4538586,0.4958942,0.4958942,
                         0.5379215,0.5779453,0.6159653,0.6519818,0.6859945,
                         0.7180036,0.7480091,0.7760109,0.8020091,0.8260036,
                         0.8260036,0.8478467,0.8673905,0.884635,0.8995803,
                         0.9122263,0.922573,0.9306204,0.9363686,0.9398175,
                         0.9409672,0.9409672),
                     y=c(0.6102651,0.5784433,0.5479526,0.518793,0.4909645,
                         0.464467,0.4393006,0.4154653,0.3929611,0.3717879,
                         0.3519459,0.3519459,0.3334405,0.3162775,0.3004569,
                         0.2859786,0.2728426,0.2610491,0.2505979,0.241489,
                         0.2337225,0.2272984,0.2272984,0.2212014,0.2153187,
                         0.2096503,0.2041963,0.1989566,0.1939312,0.1891201,
                         0.1845234,0.180141,0.1759729,0.1759729,0.1719966,
                         0.1681895,0.1645516,0.1610829,0.1577834,0.1546531,
                         0.151692,0.1489002,0.1462775,0.143824,0.143824,
                         0.1415285,0.1393796,0.1373773,0.1355217,0.1338127,
                         0.1322504,0.1308347,0.1295657,0.1284433,0.1274676,
                         0.1274676,0.1265031,0.1256402,0.1248787,0.1242188,
                         0.1236605,0.1232036,0.1228483,0.1225945,0.1224422,
                         0.1223914,0.1223914,0.122414,0.1224817,0.1225945,
                         0.1227524,0.1229554,0.1232036,0.1234969,0.1238353,
                         0.1242188,0.1246475,0.1246475,0.1251043,0.1255725,
                         0.1260519,0.1265426,0.1270446,0.1275578,0.1280823,
                         0.1286182,0.1291653,0.1297236,0.1297236,0.0157924,
                         0.0144614,0.0131754,0.0119346,0.0107389,0.0095883,
                         0.0084828,0.0074224,0.0064072,0.0054371,0.0045121,
                         0.0045121,0.0036548,0.0028878,0.0022109,0.0016244,
                         0.001128,0.0007219,0.0004061,0.0001805,4.51e-05,
                         0,0,0.0001636,0.0006543,0.0014721,
                         0.002617,0.0040891,0.0058883,0.0080147,0.0104681,
                         0.0132487,0.0163565,0.0163565,0.0197631,0.0234405,
                         0.0273886,0.0316074,0.036097,0.0408573,0.0458883,
                         0.0511901,0.0567625,0.0626058,0.0626058,0.06859,
                         0.0748111,0.081269,0.0879639,0.0948957,0.1020643,
                         0.1094698,0.1171122,0.1249915,0.1331077,0.1331077,
                         0.1414213,0.1498928,0.1585223,0.1673096,0.1762549,
                         0.1853582,0.1946193,0.2040384,0.2136153,0.2233503,
                         0.2233503,0.2288212,0.2357586,0.2441624,0.2540327,
                         0.2653694,0.2781726,0.2924422,0.3081782,0.3253807,
                         0.3440496,0.3440496,0.3641737,0.3857417,0.4087535,
                         0.4332092,0.4591089,0.4864523,0.5152397,0.545471,
                         0.5771461,0.6102651,0.6102651,0.6482121,0.6842188,
                         0.7182854,0.7504117,0.7805979,0.8088438,0.8351495,
                         0.8595149,0.8819402,0.9024253,0.9024253,0.9209645,
                         0.9375522,0.9521884,0.9648731,0.9756063,0.984388,
                         0.9912183,0.996097,0.9990243,1,1,
                         0.9990243,0.996097,0.9912183,0.984388,0.9756063,
                         0.9648731,0.9521884,0.9375522,0.9209645,0.9024253,
                         0.9024253,0.8819402,0.8595149,0.8351495,0.8088438,
                         0.7805979,0.7504117,0.7182854,0.6842188,0.6482121,
                         0.6102651,0.6102651)),
                list(color="white",  
                     x=c(0.695438,0.694937,0.6934343,0.6909297,0.6874234,
                         0.6829151,0.6774051,0.6708932,0.6633796,0.6548641,
                         0.6453467,0.6453467,0.6348358,0.6233394,0.6108577,
                         0.5973905,0.582938,0.5675,0.5510766,0.5336679,
                         0.5152737,0.4958942,0.4958942,0.4765064,0.4580876,
                         0.4406378,0.4241569,0.4086451,0.3941022,0.3805283,
                         0.3679234,0.3562874,0.3456204,0.3456204,0.3361031,
                         0.3275876,0.3200739,0.313562,0.308052,0.3035438,
                         0.3000374,0.2975328,0.2960301,0.2955292,0.2955292,
                         0.2960301,0.2975328,0.3000374,0.3035438,0.308052,
                         0.313562,0.3200739,0.3275876,0.3361031,0.3456204,
                         0.3456204,0.3562874,0.3679234,0.3805283,0.3941022,
                         0.4086451,0.4241569,0.4406378,0.4580876,0.4765064,
                         0.4958942,0.4958942,0.5152737,0.5336679,0.5510766,
                         0.5675,0.582938,0.5973905,0.6108577,0.6233394,
                         0.6348358,0.6453467,0.6453467,0.6548641,0.6633796,
                         0.6708932,0.6774051,0.6829151,0.6874234,0.6909297,
                         0.6934343,0.694937,0.695438,0.695438),
                     y=c(0.6102651,0.6358714,0.6601241,0.6830231,0.7045685,
                         0.7247603,0.7435984,0.7610829,0.7772138,0.791991,
                         0.8054146,0.8054146,0.817524,0.8283587,0.8379188,
                         0.8462042,0.8532149,0.8589509,0.8634123,0.866599,
                         0.868511,0.8691483,0.8691483,0.8684997,0.8665539,
                         0.8633108,0.8587704,0.8529329,0.8457981,0.837366,
                         0.8276368,0.8166103,0.8042865,0.8042865,0.7907727,
                         0.7759504,0.7598195,0.7423801,0.7236323,0.7035759,
                         0.6822109,0.6595375,0.6355556,0.6102651,0.6102651,
                         0.5849464,0.5608799,0.5380654,0.5165031,0.4961929,
                         0.4771348,0.4593288,0.442775,0.4274732,0.4134236,
                         0.4134236,0.4007783,0.3894642,0.3794811,0.3708291,
                         0.3635082,0.3575183,0.3528596,0.3495319,0.3475353,
                         0.3468697,0.3468697,0.3475353,0.3495319,0.3528596,
                         0.3575183,0.3635082,0.3708291,0.3794811,0.3894642,
                         0.4007783,0.4134236,0.4134236,0.4274732,0.442775,
                         0.4593288,0.4771348,0.4961929,0.5165031,0.5380654,
                         0.5608799,0.5849464,0.6102651,0.6102651))
                ),
         R=list(list(color="black", 
                     x=c(0.6758716,0.436422,0.293578,0.293578,0.05,
                         0.05,0.4578899,0.5003881,0.5406899,0.5787954,
                         0.6147046,0.6484174,0.6799339,0.7092541,0.736378,
                         0.7613055,0.7840367,0.7840367,0.8045881,0.8229761,
                         0.8392009,0.8532624,0.8651606,0.8748954,0.882467,
                         0.8878752,0.8911202,0.8922018,0.8922018,0.8920284,
                         0.8915083,0.8906413,0.8894275,0.887867,0.8859596,
                         0.8837055,0.8811046,0.8781569,0.8748624,0.8748624,
                         0.8712624,0.8673982,0.8632697,0.8588771,0.8542202,
                         0.8492991,0.8441138,0.8386642,0.8329505,0.8269725,
                         0.8269725,0.8207468,0.8142899,0.8076018,0.8006826,
                         0.7935321,0.7861505,0.7785376,0.7706936,0.7626183,
                         0.7543119,0.7543119,0.7458073,0.7371376,0.7283028,
                         0.7193028,0.7101376,0.7008073,0.6913119,0.6816514,
                         0.6718257,0.6618349,0.6618349,0.95,0.6758716),
                     y=c(0,0.3817643,0.3817643,0,0,
                         1,1,0.9992365,0.9969459,0.9931282,
                         0.9877835,0.9809118,0.972513,0.9625871,0.9511342,
                         0.9381542,0.9236471,0.9236471,0.907702,0.8904077,
                         0.8717643,0.8517717,0.8304299,0.8077391,0.783699,
                         0.7583099,0.7315715,0.7034841,0.7034841,0.6926168,
                         0.681957,0.6715048,0.6612602,0.6512231,0.6413936,
                         0.6317717,0.6223573,0.6131505,0.6041512,0.6041512,
                         0.5953595,0.5867754,0.5783988,0.5702298,0.5622683,
                         0.5545145,0.5469681,0.5396294,0.5324981,0.5255745,
                         0.5255745,0.5189918,0.5125871,0.5063603,0.5003113,
                         0.4944403,0.4887472,0.483232,0.4778947,0.4727354,
                         0.4677539,0.4677539,0.4631134,0.4586805,0.4544552,
                         0.4504374,0.4466271,0.4430245,0.4396294,0.4364418,
                         0.4334618,0.4306894,0.4306894,0,0)),
                list(color="white",  
                     x=c(0.6469725,0.6468239,0.646378,0.6456349,0.6445945,
                         0.6432569,0.641622,0.6396899,0.6374606,0.6349339,
                         0.6321101,0.6321101,0.6291706,0.625967,0.6224991,
                         0.618767,0.6147706,0.6105101,0.6059853,0.6011963,
                         0.5961431,0.5908257,0.5908257,0.585244,0.5793982,
                         0.5732881,0.5669138,0.5602752,0.5533725,0.5462055,
                         0.5387743,0.5310789,0.5231193,0.5231193,0.5150771,
                         0.5068037,0.4982991,0.4895633,0.4805963,0.4713982,
                         0.4619688,0.4523083,0.4424165,0.4322936,0.4322936,
                         0.293578,0.293578,0.4388991,0.4601028,0.4801174,
                         0.4989431,0.5165798,0.5330275,0.5482862,0.562356,
                         0.5752367,0.5869284,0.5974312,0.5974312,0.606844,
                         0.6152661,0.6226972,0.6291376,0.6345872,0.6390459,
                         0.6425138,0.6449908,0.6464771,0.6469725,0.6469725),
                     y=c(0.6945886,0.7020311,0.7092365,0.7162046,0.7229355,
                         0.7294292,0.7356857,0.741705,0.747487,0.7530319,
                         0.7583395,0.7583395,0.7634248,0.7683024,0.7729726,
                         0.7774351,0.7816901,0.7857376,0.7895775,0.7932098,
                         0.7966345,0.7998517,0.7998517,0.803017,0.8059896,
                         0.8087695,0.8113566,0.8137509,0.8159526,0.8179615,
                         0.8197776,0.821401,0.8228317,0.8228317,0.8242402,
                         0.8255004,0.8266123,0.827576,0.8283914,0.8290586,
                         0.8295775,0.8299481,0.8301705,0.8302446,0.8302446,
                         0.8302446,0.5515196,0.5515196,0.5518829,0.5529726,
                         0.5547887,0.5573314,0.5606004,0.564596,0.569318,
                         0.5747665,0.5809414,0.5878428,0.5878428,0.5955819,
                         0.6039733,0.613017,0.6227131,0.6330615,0.6440623,
                         0.6557153,0.6680208,0.6809785,0.6945886,0.6945886))
                ),
         S=list(list(color="black", 
                     x=c(0.95,0.9489139,0.9456557,0.9402254,0.9326229,
                         0.9228483,0.9108696,0.8966548,0.8802041,0.8615173,
                         0.8405945,0.8174357,0.7920408,0.7644099,0.734543,
                         0.7024401,0.6680692,0.6313984,0.5924277,0.5511571,
                         0.5075865,0.435299,0.3685697,0.3073984,0.2517853,
                         0.2017303,0.1578403,0.1207223,0.0903762,0.0668021,
                         0.05,0.2775954,0.2865395,0.2986779,0.3140106,
                         0.3325377,0.3542591,0.3793026,0.4077959,0.4397391,
                         0.4751322,0.5139752,0.5327258,0.550646,0.5677356,
                         0.5839947,0.5994232,0.6143088,0.6282999,0.6413966,
                         0.6535989,0.6649068,0.6752564,0.6845839,0.6928891,
                         0.7001721,0.706433,0.7116078,0.7156327,0.7185075,
                         0.7202325,0.7208075,0.7197853,0.7167187,0.7116078,
                         0.7044525,0.6952529,0.6847755,0.6731482,0.6603709,
                         0.6464437,0.6313665,0.6156823,0.5992955,0.5822059,
                         0.5644135,0.5459184,0.5271996,0.5087365,0.4905288,
                         0.4725768,0.4548802,0.4397072,0.4242147,0.4084028,
                         0.3922715,0.3758208,0.3595936,0.3434942,0.3275226,
                         0.3116788,0.2959627,0.2804703,0.2652972,0.2504437,
                         0.2359095,0.2216948,0.2079911,0.1949902,0.1826921,
                         0.1710967,0.1602041,0.1500781,0.1407826,0.1323177,
                         0.1246832,0.1178793,0.112417,0.1081686,0.105134,
                         0.1033132,0.1027063,0.1040479,0.1080728,0.1147808,
                         0.1241721,0.1362467,0.1508767,0.1672955,0.1855031,
                         0.2054996,0.2272848,0.2506353,0.2753274,0.3013611,
                         0.3287365,0.3574534,0.3870967,0.4172511,0.4479166,
                         0.4790932,0.5107808,0.5792032,0.6413647,0.6972653,
                         0.7469051,0.7902839,0.8276894,0.8594091,0.8854428,
                         0.9057906,0.9204525,0.6920586,0.683913,0.6735315,
                         0.6609139,0.6460603,0.6289707,0.6094854,0.5874445,
                         0.5628483,0.5356965,0.5059894,0.4865359,0.4684241,
                         0.4516539,0.4362254,0.4221384,0.4095209,0.3978616,
                         0.3871606,0.3774179,0.3686335,0.3610949,0.3544508,
                         0.348701,0.3438456,0.3398846,0.3370098,0.3347737,
                         0.3331766,0.3322183,0.3318988,0.3326974,0.3350932,
                         0.3390861,0.3446761,0.3518634,0.3603283,0.3697516,
                         0.3801331,0.3914729,0.4037711,0.4172192,0.43137,
                         0.4462236,0.4617799,0.478039,0.4950009,0.5120266,
                         0.5291162,0.5462697,0.5634871,0.5805129,0.5977303,
                         0.6151393,0.63274,0.6505324,0.6683567,0.6860532,
                         0.703622,0.721063,0.7383762,0.7554339,0.7721083,
                         0.7883993,0.804307,0.8198314,0.8347808,0.8489636,
                         0.8623798,0.8750293,0.8869122,0.8979326,0.9079947,
                         0.9170985,0.925244,0.9324312,0.938756,0.9436752,
                         0.947189,0.9492972,0.95),
                     y=c(0.2956835,0.270705,0.2464173,0.2228201,0.1999137,
                         0.1776978,0.1566906,0.1368345,0.1181295,0.1005755,
                         0.0841727,0.069036,0.0552806,0.0429065,0.0319137,
                         0.0223022,0.0142734,0.0080288,0.0035683,0.0008921,
                         0,0.0028489,0.0113957,0.0256403,0.0455827,
                         0.071223,0.1023022,0.1385612,0.18,0.2266187,
                         0.2784173,0.305036,0.2791079,0.2554245,0.2339856,
                         0.2147914,0.1978417,0.1838561,0.1729784,0.1652086,
                         0.1605468,0.1589928,0.1592518,0.1600288,0.1613237,
                         0.1631367,0.1654676,0.1683741,0.1719137,0.1760863,
                         0.1808921,0.1863309,0.1924604,0.1993381,0.206964,
                         0.2153381,0.2244604,0.2346475,0.2456403,0.2574388,
                         0.2700432,0.2834532,0.2984173,0.3122302,0.3248921,
                         0.3364029,0.3467626,0.3564604,0.3654101,0.3736115,
                         0.3810647,0.3877698,0.3941295,0.3999712,0.405295,
                         0.4101007,0.4143885,0.418705,0.4230216,0.4273381,
                         0.4316547,0.4359712,0.4397698,0.4436835,0.4477122,
                         0.4518561,0.4561151,0.4608921,0.4660144,0.471482,
                         0.477295,0.4834532,0.4899856,0.4969209,0.504259,
                         0.512,0.5201439,0.5290935,0.5386763,0.5488921,
                         0.559741,0.571223,0.5833669,0.5962014,0.6097266,
                         0.6239424,0.6388489,0.6548201,0.6716547,0.6893525,
                         0.7079137,0.7273381,0.7547626,0.7806331,0.8049496,
                         0.8277122,0.8489209,0.8688921,0.8873669,0.9043453,
                         0.9198273,0.9338129,0.9466187,0.9579856,0.9679137,
                         0.9764029,0.9834532,0.9894101,0.9940432,0.9973525,
                         0.9993381,1,0.9974388,0.9897554,0.9769496,
                         0.9590216,0.9359712,0.9078273,0.8740432,0.8346187,
                         0.789554,0.7388489,0.7179856,0.7428777,0.7651799,
                         0.7848921,0.8020144,0.8165468,0.8284604,0.8377266,
                         0.8443453,0.8483165,0.8496403,0.8492662,0.8481439,
                         0.8462734,0.8436547,0.8402878,0.8365468,0.8322302,
                         0.8273381,0.8218705,0.8158273,0.8095827,0.8029353,
                         0.7958849,0.7884317,0.7805755,0.7727194,0.7646906,
                         0.7564892,0.7481151,0.7395683,0.7262158,0.7137842,
                         0.7022734,0.6916835,0.6820144,0.6733813,0.6653237,
                         0.6578417,0.6509353,0.6446043,0.6387338,0.6332086,
                         0.6280288,0.6231942,0.618705,0.6146763,0.6106475,
                         0.6066187,0.6025899,0.5985612,0.594705,0.5906187,
                         0.5863022,0.5817554,0.5769784,0.5721727,0.566964,
                         0.5613525,0.5553381,0.5489209,0.5423309,0.535223,
                         0.5275971,0.5194532,0.5107914,0.5018129,0.4921439,
                         0.4817842,0.4707338,0.4589928,0.4465036,0.4332086,
                         0.4191079,0.4042014,0.3884892,0.3718849,0.3543022,
                         0.335741,0.3162014,0.2956835))
                ),
         T=list(list(color="black", 
                     x=c(0.6199187,0.6199187,0.3800813,0.3800813,0.05,
                         0.05,0.95,0.95,0.6199187),
                     y=c(0.8309859,0,0,0.8309859,0.8309859,
                         1,1,0.8309859,0.8309859))
                ),
         U=list(list(color="black", 
                     x=c(0.4856334,0.465622,0.446,0.4267673,0.4079238,
                         0.3894696,0.3714047,0.3537291,0.3364428,0.3195458,
                         0.3030381,0.3030381,0.2871514,0.2717467,0.2568239,
                         0.2423831,0.2284243,0.2149475,0.2019526,0.1894398,
                         0.1774089,0.1658599,0.1658599,0.1549876,0.1446159,
                         0.1347446,0.1253738,0.1165036,0.1081339,0.1002647,
                         0.092896,0.0860278,0.0796601,0.0796601,0.0740247,
                         0.0689825,0.0645335,0.0606777,0.057415,0.0547456,
                         0.0526694,0.0511864,0.0502966,0.05,0.05,
                         0.05,0.3234295,0.3234295,0.3235221,0.3238002,
                         0.3242636,0.3249125,0.3257467,0.3267662,0.3279712,
                         0.3293615,0.3309372,0.3326982,0.3326982,0.334654,
                         0.3368136,0.3391771,0.3417446,0.344516,0.3474912,
                         0.3506704,0.3540536,0.3576406,0.3614315,0.3614315,
                         0.3656303,0.37007,0.3747508,0.3796725,0.3848352,
                         0.3902389,0.3958836,0.4017693,0.407896,0.4142636,
                         0.4142636,0.4208908,0.4277961,0.4349794,0.4424408,
                         0.4501802,0.4581977,0.4664933,0.4750669,0.4839186,
                         0.4930484,0.4930484,0.5021967,0.511104,0.5197703,
                         0.5281957,0.53638,0.5443234,0.5520257,0.5594871,
                         0.5667075,0.5736869,0.5736869,0.5804346,0.5869598,
                         0.5932626,0.5993429,0.6052008,0.6108363,0.6162492,
                         0.6214398,0.6264078,0.6311535,0.6311535,0.6356674,
                         0.6399403,0.6439722,0.6477631,0.6513131,0.654622,
                         0.65769,0.660517,0.663103,0.665448,0.665448,
                         0.6675613,0.6694521,0.6711205,0.6725664,0.6737899,
                         0.6747909,0.6755695,0.6761256,0.6764593,0.6765705,
                         0.6765705,0.6765705,0.95,0.95,0.9496756,
                         0.9487024,0.9470803,0.9448095,0.9418898,0.9383213,
                         0.934104,0.9292379,0.923723,0.9175592,0.9175592,
                         0.9107837,0.9034336,0.8955088,0.8870093,0.8779351,
                         0.8682863,0.8580628,0.8472647,0.8358919,0.8239444,
                         0.8239444,0.8116354,0.7988074,0.7854604,0.7715942,
                         0.7572091,0.7423048,0.7268816,0.7109392,0.6944779,
                         0.6774974,0.6774974,0.6602297,0.6425355,0.624415,
                         0.6058682,0.586895,0.5674954,0.5476694,0.5274171,
                         0.5067384,0.4856334,0.4856334),
                     y=c(0,0.0001972,0.0007889,0.001775,0.0031556,
                         0.0049306,0.0071001,0.009664,0.0126224,0.0159752,
                         0.0197224,0.0197224,0.0240687,0.0289262,0.0342951,
                         0.0401753,0.0465668,0.0534697,0.0608839,0.0688093,
                         0.0772462,0.0861943,0.0861943,0.0958437,0.106092,
                         0.1169394,0.1283857,0.140431,0.1530752,0.1663185,
                         0.1801607,0.1946019,0.2096421,0.2096421,0.2253543,
                         0.2418115,0.2590139,0.2769613,0.2956538,0.3150913,
                         0.3352739,0.3562016,0.3778744,0.4002922,0.4002922,
                         1,1,0.4024836,0.3896786,0.3772681,
                         0.365252,0.3536304,0.3424032,0.3315705,0.3211322,
                         0.3110884,0.301439,0.2921841,0.2921841,0.2834405,
                         0.2750329,0.2669613,0.2592257,0.2518262,0.2447626,
                         0.2380351,0.2316435,0.225588,0.2198685,0.2198685,
                         0.2146092,0.2096421,0.2049671,0.2005844,0.1964938,
                         0.1926954,0.1891892,0.1859752,0.1830533,0.1804237,
                         0.1804237,0.1782031,0.1762162,0.1744631,0.1729438,
                         0.1716581,0.1706063,0.1697882,0.1692038,0.1688532,
                         0.1687363,0.1687363,0.1688605,0.169233,0.1698539,
                         0.1707232,0.1718408,0.1732067,0.174821,0.1766837,
                         0.1787947,0.1811541,0.1811541,0.1839372,0.187027,
                         0.1904237,0.1941271,0.1981373,0.2024543,0.2070782,
                         0.2120088,0.2172462,0.2227904,0.2227904,0.2286633,
                         0.2348868,0.2414609,0.2483857,0.2556611,0.2632871,
                         0.2712637,0.2795909,0.2882688,0.2972973,0.2972973,
                         0.3067056,0.316523,0.3267495,0.337385,0.3484295,
                         0.3598831,0.3717458,0.3840175,0.3966983,0.4097882,
                         0.4097882,1,1,0.4112491,0.3888093,
                         0.3670709,0.3460336,0.3256976,0.3060628,0.2871293,
                         0.268897,0.251366,0.2345362,0.2184076,0.2184076,
                         0.2030679,0.1883126,0.1741417,0.1605551,0.147553,
                         0.1351351,0.1233017,0.1120526,0.1013879,0.0913075,
                         0.0913075,0.0819284,0.0730752,0.064748,0.0569467,
                         0.0496713,0.0429218,0.0366983,0.0310007,0.0258291,
                         0.0211833,0.0211833,0.0171585,0.0135573,0.0103798,
                         0.007626,0.0052958,0.0033893,0.0019065,0.0008473,
                         0.0002118,0,0))
                ),
         V=list(list(color="black", 
                     x=c(0.6110294,0.3911765,0.05,0.2764706,0.4477941,
                         0.4498456,0.4518824,0.4539044,0.4559118,0.4579044,
                         0.4598824,0.4618456,0.4637941,0.4657279,0.4676471,
                         0.4676471,0.4695441,0.4714118,0.47325,0.4750588,
                         0.4768382,0.4785882,0.4803088,0.482,0.4836618,
                         0.4852941,0.4852941,0.4870515,0.4887941,0.4905221,
                         0.4922353,0.4939338,0.4956176,0.4972868,0.4989412,
                         0.5005809,0.5022059,0.5022059,0.5036838,0.5051765,
                         0.5066838,0.5082059,0.5097426,0.5112941,0.5128603,
                         0.5144412,0.5160368,0.5176471,0.5176471,0.5191397,
                         0.5206765,0.5222574,0.5238824,0.5255515,0.5272647,
                         0.5290221,0.5308235,0.5326691,0.5345588,0.5345588,
                         0.5366176,0.5386765,0.5407353,0.5427941,0.5448529,
                         0.5469118,0.5489706,0.5510294,0.5530882,0.5551471,
                         0.5551471,0.7257353,0.95,0.6110294),
                     y=c(0,0,1,1,0.4255004,
                         0.4188139,0.4120979,0.4053521,0.3985767,0.3917717,
                         0.384937,0.3780726,0.3711787,0.364255,0.3573017,
                         0.3573017,0.3503929,0.3436027,0.3369311,0.3303781,
                         0.3239437,0.3176279,0.3114307,0.3053521,0.2993921,
                         0.2935508,0.2935508,0.2867606,0.2800297,0.273358,
                         0.2667457,0.2601927,0.253699,0.2472646,0.2408895,
                         0.2345738,0.2283173,0.2283173,0.2344255,0.240593,
                         0.2468199,0.253106,0.2594514,0.2658562,0.2723202,
                         0.2788436,0.2854262,0.2920682,0.2920682,0.2977687,
                         0.3036027,0.3095701,0.3156709,0.3219051,0.3282728,
                         0.3347739,0.3414085,0.3481764,0.3550778,0.3550778,
                         0.3621868,0.3692809,0.3763603,0.3834248,0.3904744,
                         0.3975093,0.4045293,0.4115345,0.4185248,0.4255004,
                         0.4255004,1,1,0))
                ),
         W=list(list(color="black", 
                     x=c(0.7960912,0.6143322,0.5674267,0.5659023,0.5642606,
                         0.5625016,0.5606254,0.5586319,0.5565505,0.5544104,
                         0.5522117,0.5499544,0.5476384,0.545557,0.5434169,
                         0.5412182,0.5389609,0.536645,0.5345928,0.5325407,
                         0.5304886,0.5284365,0.5263844,0.5214007,0.5164169,
                         0.5114332,0.5064495,0.5014658,0.4962182,0.4910293,
                         0.485899,0.4808274,0.4758143,0.4737329,0.4715928,
                         0.4693941,0.4671368,0.4648208,0.4627394,0.4605993,
                         0.4584007,0.4561433,0.4538274,0.4517752,0.4497231,
                         0.447671,0.4456189,0.4435668,0.4415733,0.4396971,
                         0.4379381,0.4362964,0.434772,0.3864007,0.2046417,
                         0.05,0.2398208,0.3050489,0.306456,0.3077459,
                         0.3089186,0.3099739,0.3109121,0.3117622,0.3125537,
                         0.3132866,0.3139609,0.3145765,0.3151336,0.3156319,
                         0.3160717,0.3164528,0.3167752,0.3186221,0.320645,
                         0.3228436,0.3252182,0.3277687,0.3301726,0.3326938,
                         0.3353322,0.3380879,0.3409609,0.3439511,0.3470586,
                         0.3502834,0.3536254,0.3570847,0.4296417,0.572557,
                         0.645114,0.6488078,0.6522671,0.6554919,0.6584821,
                         0.6612378,0.6641107,0.6668664,0.6695049,0.6720261,
                         0.67443,0.6769805,0.679355,0.6815537,0.6835765,
                         0.6854235,0.6857166,0.6860098,0.6863029,0.6865961,
                         0.6868893,0.6872117,0.6875928,0.6880326,0.6885309,
                         0.6890879,0.6897036,0.6903779,0.6911107,0.6919023,
                         0.6927524,0.6936906,0.6947459,0.6959186,0.6972085,
                         0.6986156,0.7601792,0.95,0.7960912),
                     y=c(0,0,0.1979244,0.2037658,0.2100222,
                         0.2166938,0.2237806,0.2312824,0.2390808,0.2470571,
                         0.2552113,0.2635434,0.2720534,0.2809785,0.2899629,
                         0.2990067,0.3081097,0.3172721,0.3267013,0.3360119,
                         0.3452039,0.3542772,0.363232,0.384166,0.4054559,
                         0.4271016,0.449103,0.4714603,0.4488362,0.4266271,
                         0.4048332,0.3834544,0.3624907,0.353536,0.3444626,
                         0.3352706,0.32596,0.3165308,0.3073684,0.2982654,
                         0.2892216,0.2802372,0.2713121,0.2628021,0.25447,
                         0.2463158,0.2383395,0.2305411,0.2233062,0.216427,
                         0.2099036,0.2037361,0.1979244,0,0,
                         1,1,0.4848036,0.4730319,0.4608451,
                         0.4482431,0.4352261,0.4217939,0.4083024,0.3951075,
                         0.382209,0.3696071,0.3573017,0.3431282,0.3290734,
                         0.3151371,0.3013195,0.2876205,0.2984136,0.309444,
                         0.3207116,0.3322165,0.3439585,0.3545738,0.3656635,
                         0.3772276,0.3892661,0.4017791,0.4148851,0.4281097,
                         0.4414529,0.4549148,0.4684952,0.7197924,0.7197924,
                         0.4684952,0.4551816,0.4419274,0.4287324,0.4155967,
                         0.4025204,0.3900074,0.3779689,0.3664047,0.355315,
                         0.3446998,0.3329281,0.3213343,0.3099185,0.2986805,
                         0.2876205,0.2884211,0.2908228,0.2948258,0.3004299,
                         0.3076353,0.3160267,0.325189,0.3351223,0.3458265,
                         0.3573017,0.3693403,0.3817346,0.3944848,0.4075908,
                         0.4210526,0.4347517,0.4479763,0.4607265,0.4730022,
                         0.4848036,1,1,0))
                ),
         X=list(list(color="black", 
                     x=c(0.7229862,0.5010985,0.2784784,0.05,0.3861269,
                         0.0800244,0.3085028,0.5010985,0.6929618,0.9199756,
                         0.60655,0.95,0.7229862),
                     y=c(0,0.3654559,0,0,0.5270571,
                         1,1,0.6812454,1,1,
                         0.5270571,0,0))
                ),
         Y=list(list(color="black", 
                     x=c(0.6082651,0.6082651,0.3917349,0.3917349,0.05,
                         0.2775777,0.498527,0.7224223,0.95,0.6082651),
                     y=c(0.4121572,0,0,0.4121572,1,
                         1,0.5804299,1,1,0.4121572))
                ),
         Z=list(list(color="black", 
                     x=c(0.95,0.05,0.05,0.6245956,0.1150809,
                         0.1150809,0.9148906,0.9148906,0.340295,0.95,
                         0.95),
                     y=c(0,0,0.1549296,0.828762,0.828762,
                         1,1,0.8480356,0.171238,0.171238,
                         0))
                )
         )
  x <- letterPolygon[[toupper(ch)]]
  x <- lapply(x, function(.ele){
    if(.ele$color=="black") .ele$color <- color
    .ele
  })
  return(x)
}

checkSummary <- function(x){
  if("summary" %in% slotNames(x)){
    if(any(is.na(x@summary@xscale))){
      x@summary@xscale <- c(0, 72)
    }
    if(any(is.na(x@summary@yscale))){
      x@summary@yscale <- c(72, 0)
    }
  }
  x
}
coloredSymbols <- function(ncha, font, color, rname, alpha, 
                           fontface="bold", envir = .globals){
  symbols<-list()
  for(i in 1:ncha){
    symbols[[i]]<-checkSummary(importSVG(font, color[i], rname[i], fontface))
  }
  if(!missing(alpha)){
    for(i in 1:ncha){
      symbols[[paste0(i, "_", alpha)]]<-checkSummary(
        importSVG(font, colalpha(color, alpha)[i], rname[i], 
                  fontface, envir = envir))
    }
  }
  symbols
}

addPseudolog2<-function(x){
    x <- round(x, digits=6)
    ifelse(x < .000001, -20, log2(x)) ## !important -10.
}

## get Information Entropy from matrix
getIE<-function(x){
    addPseudolog2(nrow(x))
}

UngappedAlignment<-function(pfms, i, threshold, minimalConsensus=0, 
                            rcpostfix="(RC)", revcomp=TRUE){
    if(!is(pfms[[i]], "pfm")){
        pcms <- pfms
        pfms <- lapply(pfms, pcm2pfm)
    }else{
        pcms <- NULL
    }
    res<-getAlignedICWithoutGap(pfms[[i-1]], pfms[[i]], threshold, revcomp)
    if(!is.null(pcms)) pfms <- pcms
    if(res$max>=minimalConsensus){
        if(res$rev){
            pfms[[i]]<-matrixReverseComplement(pfms[[i]])
            pfms[[i]]@name<-paste(pfms[[i]]@name, rcpostfix, sep="")
        }
        if(res$offset>0){
            pfms[[i]]<-addBlank(pfms[[i]], res$offset, FALSE)
        }else{
            if(res[1]<0){
                pfms[1:(i-1)]<-lapply(pfms[1:(i-1)],function(.ele) 
                  addBlank(.ele, -1*res$offset, FALSE))
            }
        }
    }
    pfms
}

UngappedAlignmentAA<-function(pcms, i, threshold, minimalConsensus){
  pcm1 <- pcms[[i-1]]
  pcm2 <- pcms[[i]]
  offset1 <- 0
  offset2 <- 0
  ## trim pcm
  trimPCMbyIC <- function(pcm, t){
    ic <- getIC(pcm)
    w <- rle(ic < t)
    offL <- offR <- 0
    if(w$values[1]) {
      offL <- w$lengths[1]
    }
    if(w$values[length(w$values)]) {
      offR <- w$lengths[length(w$values)]
    }
    if(offL+1>ncol(pcm@mat)-offR){
      return(pcm=NULL, offL=offL, offR=offR)
    }
    pcm@mat <- pcm@mat[, (offL+1):(ncol(pcm@mat)-offR), drop=FALSE]
    list(pcm=pcm, offL=offL, offR=offR)
  }
  res <- trimPCMbyIC(pcm1, t=threshold)
  if(is.null(res$pcm)){
    return(pcms)
  }
  pcm1 <- res$pcm
  offset1 <- res$offL
  res <- trimPCMbyIC(pcm2, t=threshold)
  if(is.null(res$pcm)){
    return(pcms)
  }
  pcm2 <- res$pcm
  offset2 <- res$offL
  res<-compare2profiles(pcm1, pcm2, method="Smith-Waterman")
  if(res$length>=minimalConsensus){
    of1 <- offset1 + res$i_start
    of2 <- offset2 + res$j_start
    if(of1<of2){
      pcms[seq.int(i-1)] <- lapply(pcms[seq.int(i-1)], function(.ele){
        addBlank(.ele, of2-of1, FALSE)
      })
    }else{
      if(of1>of2){
        pcms[[i]] <- addBlank(pcms[[i]], of1-of2, FALSE)
      }
    }
  }
  pcms
}

getAlignedICWithoutGap<-function(pfm1, pfm2, threshold, revcomp=TRUE){
    if(!is(pfm1, "pfm") | !is(pfm2, "pfm")) 
      stop("class of pfm1 and pfm2 must be pfm")
    offset1<-getoffsetPosByIC(pfm1, pfm2, threshold)
    if(revcomp){
        pfm3<-matrixReverseComplement(pfm2)
        offset2<-getoffsetPosByIC(pfm1, pfm3, threshold)
    }else{
        offset2<-list(k=0,max=0)
    }
    offset<-0
    rev<-FALSE
    if((offset1$max < offset2$max)){
        rev<-TRUE
        max<-offset2$max
        offset<-offset2$k
    }else{
        max<-offset1$max
        offset<-offset1$k
    }
    list(offset=offset,rev=rev,max=max)    
}
getICbyBase<-function(p, pfmj){
    re<-rep(0, 5)
    re[1:4]<-pfmj * (addPseudolog2(pfmj) - addPseudolog2(p))
    re[5]<-sum(re[1:4])
    re
}
getALLRbyBase <- function(b, pfmi, pfmj){##return 2x ALLR
    sum(pfmj * (addPseudolog2(pfmi) - addPseudolog2(b))) + 
        sum(pfmi * (addPseudolog2(pfmj) - addPseudolog2(b)))
}
checkALLR <- function(...){
    getALLRbyBase(...)>0
}
SWU <- function(pattern, subject, b, 
                match=1, mismatch=-1, 
                gap=-1000){## motif length will never longer than 1000
    if(!is(pattern, "matrix") || !is(subject, "matrix")){
        stop("pattern and subject must be numeric matrix")
    }
    m <- ncol(pattern)
    n <- ncol(subject)
    score <- matrix(0, nrow=m+1, ncol=n+1)
    point <- matrix("none", nrow=m+1, ncol=n+1)
    max_i <- 1
    max_j <- 1
    max_score <- 0
    for(i in 1:m){
        for(j in 1:n){
            diagonal_score <- 0
            left_score <- 0
            up_score <- 0
            #calculate match score
            col1 <- subject[, j]
            col2 <- pattern[, i]
            #score by KLLR
            if(checkALLR(b, col1, col2)){
                diagonal_score <- score[i, j] + match
            }else{
                diagonal_score <- score[i, j] + mismatch
            }
            
            #calculate gap scores
            up_score <- score[i, j+1] + gap
            left_score <- score[i+1, j] + gap
            
            if(diagonal_score <=0 && up_score <=0 && left_score <=0){
                score[i+1, j+1] <- 0
                point[i+1, j+1] <- "none"
                next # terminate this iteration of the loop
            }
            
            #choose best score
            if(diagonal_score >= up_score){
                if(diagonal_score >= left_score){
                    score[i+1, j+1] <- diagonal_score
                    point[i+1, j+1] <- "diagonal"
                }else{
                    score[i+1, j+1] <- left_score
                    point[i+1, j+1] <- "left"
                }
            }else{
                if(up_score > left_score){
                    score[i+1, j+1] <- up_score
                    point[i+1, j+1] <- "up"
                }else{
                    score[i+1, j+1] <- left_score
                    point[i+1, j+1] <- "left"
                }
            }
            
            #set maximum score
            if(score[i+1, j+1] > max_score){
                max_i <- i+1
                max_j <- j+1
                max_score <- score[i+1, j+1]
            }
        }
    }
    #trace-back
    align1 <- c()
    align2 <- c()
    
    i <- max_i
    j <- max_j
    while(1){
        if(point[i, j]=="none"){
            break
        }
        
        if(point[i, j]=="diagonal"){
            align1 <- c(align1, j-1)
            align2 <- c(align2, i-1)
            i <- i-1
            j <- j-1
        }else{
            if(point[i, j]=="left"){
                align1 <- c(align1, j-1)
                align2 <- c(align2, 0)
                j <- j-1
            }else{
                if(point[i, j]=="up"){
                    align1 <- c(align1, 0)
                    align2 <- c(align2, i-1)
                    i <- i-1
                }
            }
        }
    }
    align1 <- rev(align1)
    align2 <- rev(align2)
    if(length(align1)<1 | length(align2)<1){
        k <- 0
    }else{
        k=align1[1]-align2[1]
    }
    list(k=k, max=max_score)
}
ungappedScore <- function(query, subject, b, threshold){
    if(!is(query, "matrix") || !is(subject, "matrix")){
        stop("query and subject must be numeric matrix")
    }
    m <- ncol(query)
    n <- ncol(subject)
    score <- matrix(0, nrow=m+1, ncol=n+1)
    for(i in 1:m){
        for(j in 1:n){
            ic1 <- getICbyBase(b, query[, i])[5]
            ic2 <- getICbyBase(b, subject[, j])[5]
            if(ic1>=threshold && ic2>=threshold){
                value <- getICbyBase(b, c(query[, i]+subject[, j])/2)[5]
            }else{
                value <- 0
            }
            score[i+1, j+1] <- score[i, j] + value
        }
    }
    score[1:m, 1:n] <- 0
    max_score <- max(score)
    if(max_score==0){
        return(list(k=0, max=0))
    }
    idx <- which(score==max_score, arr.ind=TRUE)
    
    if(nrow(idx)>1){
        idxn <- apply(idx, 1, min)
        idx <- idx[which(idxn==min(idxn))[1], , drop=FALSE]
    }
    max_r <- idx[, "row"]
    max_c <- idx[, "col"]
    k <- max_r - max_c
    list(k=as.numeric(k), max=max_score)
}
getoffsetPosByIC<-function(pfm1, pfm2, threshold){
    if(!is(pfm1, "pfm") | !is(pfm2, "pfm")) 
      stop("class of pfm1 and pfm2 must be pfm")
    res <- ungappedScore(pfm1$mat, pfm2$mat, b=pfm1$background, threshold)
    res
}

#' @importFrom stats sd cor
getoffsetPosByIC_old<-function(pfm1, pfm2, threshold){
    if(!is(pfm1, "pfm") | !is(pfm2, "pfm")) 
      stop("class of pfm1 and pfm2 must be pfm")
    score1<-rep(0, ncol(pfm1@mat))
    score2<-rep(0, ncol(pfm2@mat))
    value1<-rep(0, ncol(pfm1@mat))
    value2<-rep(0, ncol(pfm2@mat))
    for(i in 1:ncol(pfm1@mat)){
        J<-ifelse(ncol(pfm1@mat)+1-i>ncol(pfm2@mat),
                  ncol(pfm2@mat),ncol(pfm1@mat)+1-i)
        for(j in 1:J){
            ic1<-getICbyBase(pfm1@background, pfm1@mat[,i+j-1])
            ic2<-getICbyBase(pfm2@background, pfm2@mat[,j])
            #ic3<-getALLRbyBase(pfm1@background, pfm1@mat[,i+j-1], pfm2@mat[,j])
            ic3 <- getICbyBase(pfm1@background, 
                               (pfm1@mat[,i+j-1]+pfm2@mat[,j])/2)
            if(sd(pfm1@mat[,i+j-1])==0 || sd(pfm2@mat[,j])==0){
                corr <- 0
            }else{
                corr <- cor(pfm1@mat[,i+j-1], pfm2@mat[,j], method="spearman")
            }
            if(ic1[5]>threshold & ic2[5]>threshold){
                #a<-ic1[1:4]>mean(ic1[1:4])
                #b<-ic2[1:4]>mean(ic2[1:4])
                #if(ic3>0) score1[i]<-score1[i]+1
                #if(ic3>0.25) {
                #if(any(a&b)){
                if(corr>0){
                    score1[i]<-score1[i]+1
                    value1[i] <- value1[i]+ic3[5]
                }
            }
        }
    }
    for(i in 1:ncol(pfm2@mat)){
        J<-ifelse(ncol(pfm2@mat)+1-i>ncol(pfm1@mat),
                  ncol(pfm1@mat),ncol(pfm2@mat)+1-i)
        for(j in 1:J){
            ic2<-getICbyBase(pfm2@background, pfm2@mat[,i+j-1])
            ic1<-getICbyBase(pfm1@background, pfm1@mat[,j])
            #ic3<-getALLRbyBase(pfm1@background, pfm1@mat[,j], pfm2@mat[,i+j-1])
            ic3 <- getICbyBase(pfm1@background, 
                               (pfm2@mat[,i+j-1]+pfm1@mat[,j])/2)
            if(sd(pfm1@mat[,j])==0 || sd(pfm2@mat[,i+j-1])==0){
                corr <- 0
            }else{
                corr <- cor(pfm1@mat[,j], pfm2@mat[,i+j-1], method="spearman")
            }
            if(ic1[5]>threshold & ic2[5]>threshold){
                #a<-ic1[1:4]>mean(ic1[1:4])
                #b<-ic2[1:4]>mean(ic2[1:4])
                #if(ic3>0) score2[i]<-score2[i]+1
                #if(ic3>0.25) {
                #if(any(a&b)){
                if(corr>0){
                    score2[i]<-score2[i]+1
                    value2[i] <- value2[i]+ic3[5]
                }
            }
        }
    }
    k1<-match(max(score1),score1)
    k2<-match(max(score2),score2)
    sc <- (score1[k1] < score2[k2]) || ((score1[k1] == score2[k2]) && 
                                            (value1[k1] < value2[k2]))
    k <- ifelse(sc, -1*(k2-1), (k1-1))
    max <- ifelse(sc, score2[k2], score1[k1])
    val <- ifelse(sc, value2[k2], value1[k1])
    list(k=k,max=max,val=val)
}

setClass("Rect", 
         representation(x="numeric", y="numeric",
                        width="numeric", height="numeric"), 
         prototype=prototype(x=0, y=0, width=0, height=0)
)

setGeneric("isContainedIn", function(a, b) standardGeneric("isContainedIn"))

setMethod("isContainedIn", signature(a="Rect", b="Rect"), function(a, b){
    a@x >= b@x && a@y >= b@y && 
      a@x+a@width <= b@x+b@width && a@y+a@height <= b@y+b@height
})

setClass("pos",
         representation(box="Rect", beta="numeric", sig="pfm", 
                        freq="numeric", norm="numeric")
)

getPFMid <- function(pfms, nodename, rcpostfix="(RC)"){
    pfmNames <- as.character(unlist(lapply(pfms, function(.ele) .ele@name)))
    pfmNames <- gsub(rcpostfix,"",pfmNames, fixed=TRUE)
    which(pfmNames==nodename)
}

getParentNode <- function(nodelist, nodename){
    for(i in 1:length(nodelist)){
        currNode <- nodelist[[i]]
        if(currNode@left==nodename) return(c(currNode@parent, "left"))
        if(currNode@right==nodename) return(c(currNode@parent, "right"))
    }
    NULL
}

makeLeaveNames <- function(ch){
    gsub(".", "_", make.names(ch), fixed=TRUE)
}

#' @importFrom graphics par
#' @importFrom grid stringWidth stringHeight grid.rect gpar grid.text
grid.eti <- function(x, y, label, clabel, boxes=FALSE, 
                     coul = rep(1, length(x)), 
                     bg = "white", draw=TRUE){
  if (length(label) == 0) 
    return(invisible())
  if (is.null(label)) 
    return(invisible())
  if (any(label == "")) 
    return(invisible())
  cex0 <- par("cex") * clabel
  plot <- gList()
  for (i in seq_along(x)) {
    cha <- as.character(label[i])
    cha <- paste(" ", cha, " ", sep = "")
    x1 <- x[i]
    y1 <- y[i]
    xh <- cex0*stringWidth(cha)
    yh <- cex0*stringHeight(cha) * 1.66666666667
    if (boxes) {
      plot <- gList(plot,
                    rectGrob(x=x1, y=y1, width=xh, height=yh, 
                             gp=gpar(fill = bg, col = coul[i]), 
                             default.units = "native"))
    }
    plot <- gList(plot,
                  textGrob(x=x1, y=y1, label=cha, 
                           gp=gpar(cex = cex0, col = coul[i]), 
                           default.units = "native"))
  }
  if(draw){
    suppressWarnings(grid.draw(plot))
  }else{
    plot
  }
} 
