% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spls.R
\name{spls}
\alias{spls}
\title{Sparse Partial Least Squares (sPLS)}
\usage{
spls(
  X,
  Y,
  ncomp = 2,
  mode = c("regression", "canonical", "invariant", "classic"),
  keepX,
  keepY,
  scale = TRUE,
  tol = 1e-06,
  max.iter = 100,
  near.zero.var = FALSE,
  logratio = "none",
  multilevel = NULL,
  all.outputs = TRUE,
  verbose.call = FALSE
)
}
\arguments{
\item{X}{numeric matrix of predictors with the rows as individual
observations. missing values (\code{NA}s) are allowed.}

\item{Y}{numeric matrix of response(s) with the rows as individual
observations matching \code{X}. missing values (\code{NA}s) are allowed.}

\item{ncomp}{Positive Integer. The number of components to include in the
model. Default to 2.}

\item{mode}{Character string indicating the type of PLS algorithm to use. One
of \code{"regression"}, \code{"canonical"}, \code{"invariant"} or \code{"classic"}. See Details.}

\item{keepX}{numeric vector of length \code{ncomp}, the number of variables
to keep in \eqn{X}-loadings. By default all variables are kept in the model.}

\item{keepY}{numeric vector of length \code{ncomp}, the number of variables}

\item{scale}{Logical. If scale = TRUE, each block is standardized to zero
means and unit variances (default: TRUE)}

\item{tol}{Positive numeric used as convergence criteria/tolerance during the
iterative process. Default to \code{1e-06}.}

\item{max.iter}{Integer, the maximum number of iterations. Default to  100.}

\item{near.zero.var}{Logical, see the internal \code{\link{nearZeroVar}}
function (should be set to TRUE in particular for data with many zero
values). Setting this argument to FALSE (when appropriate) will speed up the
computations. Default value is FALSE.}

\item{logratio}{Character, one of ('none','CLR') specifies
the log ratio transformation to deal with compositional values that may
arise from specific normalisation in sequencing data. Default to 'none'.
See \code{?logratio.transfo} for details.}

\item{multilevel}{Numeric, design matrix for repeated measurement analysis,
where multilevel decomposition is required. For a one factor decomposition,
the repeated measures on each individual, i.e. the individuals ID is input
as the first column. For a 2 level factor decomposition then 2nd AND 3rd
columns indicate those factors. See examples.}

\item{all.outputs}{Logical. Computation can be faster when some specific
(and non-essential) outputs are not calculated. Default = \code{TRUE}.}

\item{verbose.call}{Logical (Default=FALSE), if set to TRUE then the \code{$call}
component of the returned object will contain the variable values for all 
parameters. Note that this may cause large memory usage.}
}
\value{
\code{spls} returns an object of class \code{"spls"}, a list that
contains the following components:
\item{call}{if \code{verbose.call = FALSE}, then just the function call is returned.
If \code{verbose.call = TRUE} then all the inputted values are accessable via
this component}
\item{X}{the centered and standardized original predictor matrix.}
\item{Y}{the centered and standardized original response vector or matrix.}
\item{ncomp}{the number of components included in the model.}
\item{mode}{the algorithm used to fit the model.} \item{keepX}{number of
\eqn{X} variables kept in the model on each component.} \item{keepY}{number
of \eqn{Y} variables kept in the model on each component.}
\item{variates}{list containing the variates.} \item{loadings}{list
containing the estimated loadings for the \eqn{X} and \eqn{Y} variates.}
\item{names}{list containing the names to be used for individuals and
variables.} \item{tol}{the tolerance used in the iterative algorithm, used
for subsequent S3 methods} \item{iter}{Number of iterations of the algorithm
for each component} \item{max.iter}{the maximum number of iterations, used
for subsequent S3 methods} \item{nzv}{list containing the zero- or near-zero
predictors information.} \item{scale}{whether scaling was applied per
predictor.} \item{logratio}{whether log ratio transformation for relative
proportion data was applied, and if so, which type of transformation.}
\item{prop_expl_var}{Proportion of variance explained per component (note
that contrary to PCA, this amount may not decrease as the aim of the method
is not to maximise the variance, but the covariance between data sets).}
\item{input.X}{numeric matrix of predictors in X that was input, before any
saling / logratio / multilevel transformation.} \item{mat.c}{matrix of
coefficients from the regression of X / residual matrices X on the
X-variates, to be used internally by \code{predict}.}
\item{defl.matrix}{residual matrices X for each dimension.}
}
\description{
Function to perform sparse Partial Least Squares (sPLS). The sPLS approach
combines both integration and variable selection simultaneously on two data
sets in a one-step strategy.
}
\details{
\code{spls} function fit sPLS models with \eqn{1, \ldots ,}\code{ncomp}
components. Multi-response models are fully supported. The \code{X} and
\code{Y} datasets can contain missing values.
}
\section{missing values}{
 
The estimation of the missing values can be performed using the
\code{\link{impute.nipals}} function. Otherwise, missing values are handled
by element-wise deletion in the \code{pls} function without having to delete
the rows with missing data.
}

\section{multilevel}{

Multilevel (s)PLS enables the integration of data measured on two different
data sets on the same individuals. This approach differs from multilevel
sPLS-DA as the aim is to select subsets of variables from both data sets that
are highly positively or negatively correlated across samples. The approach
is unsupervised, i.e. no prior knowledge about the sample groups is included.
}

\section{logratio and multilevel}{
 
logratio transform and multilevel analysis are performed sequentially as
internal pre-processing step, through \code{\link{logratio.transfo}} and
\code{\link{withinVariation}} respectively.
}

\examples{
data(liver.toxicity)
X <- liver.toxicity$gene
Y <- liver.toxicity$clinic

toxicity.spls <- spls(X, Y, ncomp = 2, keepX = c(50, 50),
keepY = c(10, 10))

toxicity.spls <- spls(X, Y[,1:2,drop=FALSE], ncomp = 5, keepX = c(50, 50))#,  mode="canonical")

\dontrun{

## Second example: one-factor multilevel analysis with sPLS, selecting a subset of variables
#--------------------------------------------------------------

data(liver.toxicity)
# note: we made up those data, pretending they are repeated measurements
repeat.indiv <- c(1, 2, 1, 2, 1, 2, 1, 2, 3, 3, 4, 3, 4, 3, 4, 4, 5, 6, 5, 5,
6, 5, 6, 7, 7, 8, 6, 7, 8, 7, 8, 8, 9, 10, 9, 10, 11, 9, 9,
10, 11, 12, 12, 10, 11, 12, 11, 12, 13, 14, 13, 14, 13, 14,
13, 14, 15, 16, 15, 16, 15, 16, 15, 16)
summary(as.factor(repeat.indiv)) # 16 rats, 4 measurements each

# this is a spls (unsupervised analysis) so no need to mention any factor in design
# we only perform a one level variation split
design <- data.frame(sample = repeat.indiv)
res.spls.1level <- spls(X = liver.toxicity$gene,
Y=liver.toxicity$clinic,
multilevel = design,
ncomp = 3,
keepX = c(50, 50, 50), keepY = c(5, 5, 5),
mode = 'canonical')

# set up colors and pch for plotIndiv
col.stimu <- 1:nlevels(design$stimu)

plotIndiv(res.spls.1level, rep.space = 'X-variate', ind.names = FALSE,
group = liver.toxicity$treatment$Dose.Group,
pch = 20, main = 'Gene expression subspace',
legend = TRUE)


plotIndiv(res.spls.1level, rep.space = 'Y-variate', ind.names = FALSE,
group = liver.toxicity$treatment$Dose.Group,
pch = 20, main = 'Clinical measurements ssubpace',
legend = TRUE)

plotIndiv(res.spls.1level, rep.space = 'XY-variate', ind.names = FALSE,
group = liver.toxicity$treatment$Dose.Group,
pch = 20, main = 'Both Gene expression and Clinical subspaces',
legend = TRUE)

## Third example: two-factor multilevel analysis with sPLS, selecting a subset of variables
#--------------------------------------------------------------

data(liver.toxicity)
dose <- as.factor(liver.toxicity$treatment$Dose.Group)
time <- as.factor(liver.toxicity$treatment$Time.Group)
# note: we made up those data, pretending they are repeated measurements
repeat.indiv <- c(1, 2, 1, 2, 1, 2, 1, 2, 3, 3, 4, 3, 4, 3, 4, 4, 5, 6, 5, 5,
6, 5, 6, 7, 7, 8, 6, 7, 8, 7, 8, 8, 9, 10, 9, 10, 11, 9, 9,
10, 11, 12, 12, 10, 11, 12, 11, 12, 13, 14, 13, 14, 13, 14,
13, 14, 15, 16, 15, 16, 15, 16, 15, 16)
summary(as.factor(repeat.indiv)) # 16 rats, 4 measurements each
design <- data.frame(sample = repeat.indiv, dose = dose, time = time)

res.spls.2level = spls(liver.toxicity$gene,
Y = liver.toxicity$clinic,
multilevel = design,
ncomp=2,
keepX = c(10,10), keepY = c(5,5))
}
}
\references{
Sparse PLS: canonical and regression modes:

Lê Cao, K.-A., Martin, P.G.P., Robert-Granie, C. and Besse, P. (2009).
Sparse canonical methods for biological data integration: application to a
cross-platform study. \emph{BMC Bioinformatics} \bold{10}:34.

Lê Cao, K.-A., Rossouw, D., Robert-Granie, C. and Besse, P. (2008). A sparse
PLS for variable selection when integrating Omics data. \emph{Statistical
Applications in Genetics and Molecular Biology} \bold{7}, article 35.

Sparse SVD: Shen, H. and Huang, J. Z. (2008). Sparse principal component
analysis via regularized low rank matrix approximation. \emph{Journal of
Multivariate Analysis} \bold{99}, 1015-1034.

PLS methods: Tenenhaus, M. (1998). \emph{La regression PLS: theorie et
pratique}. Paris: Editions Technic. Chapters 9 and 11.

Abdi H (2010). Partial least squares regression and projection on latent
structure regression (PLS Regression). \emph{Wiley Interdisciplinary
Reviews: Computational Statistics}, 2(1), 97-106.

Wold H. (1966). Estimation of principal components and related models by
iterative least squares. In: Krishnaiah, P. R. (editors), \emph{Multivariate
Analysis}. Academic Press, N.Y., 391-420.

On multilevel analysis:

Liquet, B., Lê Cao, K.-A., Hocini, H. and Thiebaut, R. (2012) A novel
approach for biomarker selection and the integration of repeated measures
experiments from two platforms. \emph{BMC Bioinformatics} \bold{13}:325.

Westerhuis, J. A., van Velzen, E. J., Hoefsloot, H. C., and Smilde, A. K.
(2010). Multivariate paired data analysis: multilevel PLSDA versus OPLSDA.
\emph{Metabolomics}, \bold{6}(1), 119-128.
}
\seealso{
\code{\link{pls}}, \code{\link{summary}}, \code{\link{plotIndiv}},
\code{\link{plotVar}}, \code{\link{cim}}, \code{\link{network}},
\code{\link{predict}}, \code{\link{perf}} and http://www.mixOmics.org for
more details.
}
\author{
Sébastien Déjean, Ignacio González, Florian Rohart, Kim-Anh Lê Cao,
Al J abadi
}
\keyword{multivariate}
\keyword{regression}
