% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/block.pls.R
\name{block.pls}
\alias{block.pls}
\title{N-integration with Projection to Latent Structures models (PLS)}
\usage{
block.pls(
  X,
  Y,
  indY,
  ncomp = 2,
  design,
  mode,
  scale = TRUE,
  tol = 1e-06,
  max.iter = 100,
  near.zero.var = FALSE,
  all.outputs = TRUE,
  verbose.call = FALSE
)
}
\arguments{
\item{X}{A named list of data sets (called 'blocks') measured on the same 
samples. Data in the list should be arranged in matrices, samples x variables, 
with samples order matching in all data sets.}

\item{Y}{Matrix response for a multivariate regression framework. Data
should be continuous variables (see \code{?block.plsda} for supervised
classification and factor response).}

\item{indY}{To supply if \code{Y} is missing, indicates the position of 
the matrix response in the list \code{X}.}

\item{ncomp}{the number of components to include in the model. Default to 2.
Applies to all blocks.}

\item{design}{numeric matrix of size (number of blocks in X) x (number of
blocks in X) with values between 0 and 1. Each value indicates the strenght
of the relationship to be modelled between two blocks; a value of 0
indicates no relationship, 1 is the maximum value. Alternatively, one of
c('null', 'full') indicating a disconnected or fully connected design,
respecively, or a numeric between 0 and 1 which will designate all
off-diagonal elements of a fully connected design (see examples in
\code{block.splsda}). If \code{Y} is provided instead of \code{indY}, the
\code{design} matrix is changed to include relationships to \code{Y}.}

\item{mode}{Character string indicating the type of PLS algorithm to use. One
of \code{"regression"}, \code{"canonical"}, \code{"invariant"} or \code{"classic"}. See Details.}

\item{scale}{Logical. If scale = TRUE, each block is standardized to zero
means and unit variances (default: TRUE)}

\item{tol}{Positive numeric used as convergence criteria/tolerance during the
iterative process. Default to \code{1e-06}.}

\item{max.iter}{Integer, the maximum number of iterations. Default to  100.}

\item{near.zero.var}{Logical, see the internal \code{\link{nearZeroVar}}
function (should be set to TRUE in particular for data with many zero
values). Setting this argument to FALSE (when appropriate) will speed up the
computations. Default value is FALSE.}

\item{all.outputs}{Logical. Computation can be faster when some specific
(and non-essential) outputs are not calculated. Default = \code{TRUE}.}

\item{verbose.call}{Logical (Default=FALSE), if set to TRUE then the \code{$call}
component of the returned object will contain the variable values for all 
parameters. Note that this may cause large memory usage.}
}
\value{
\code{block.pls} returns an object of class \code{'block.pls'}, a
list that contains the following components:

\item{X}{the centered and standardized original predictor matrix.}
\item{indY}{the position of the outcome Y in the output list X.}
\item{ncomp}{the number of components included in the model for each block.}
\item{mode}{the algorithm used to fit the model.} \item{variates}{list
containing the variates of each block of X.} \item{loadings}{list containing
the estimated loadings for the variates.} \item{names}{list containing the
names to be used for individuals and variables.} \item{nzv}{list containing
the zero- or near-zero predictors information.} \item{iter}{Number of
iterations of the algorithm for each component}
\item{prop_expl_var}{Percentage of explained variance for each
component and each block}
\item{call}{if \code{verbose.call = FALSE}, then just the function call is returned.
If \code{verbose.call = TRUE} then all the inputted values are accessable via
this component}
}
\description{
Integration of multiple data sets measured on the same samples or
observations, ie. N-integration. The method is partly based on Generalised
Canonical Correlation Analysis.
}
\details{
\code{block.spls} function fits a horizontal integration PLS model with a
specified number of components per block). An outcome needs to be provided,
either by \code{Y} or by its position \code{indY} in the list of blocks
\code{X}. Multi (continuous)response are supported. \code{X} and \code{Y}
can contain missing values. Missing values are handled by being disregarded
during the cross product computations in the algorithm \code{block.pls}
without having to delete rows with missing data. Alternatively, missing data
can be imputed prior using the \code{\link{impute.nipals}} function.

The type of algorithm to use is specified with the \code{mode} argument.
Four PLS algorithms are available: PLS regression \code{("regression")}, PLS
canonical analysis \code{("canonical")}, redundancy analysis
\code{("invariant")} and the classical PLS algorithm \code{("classic")} (see
References and \code{?pls} for more details). Note that the argument 'scheme' 
has now been hardcoded to 'horst' and 'init' to 'svd.single'. 

Note that our method is partly based on Generalised Canonical Correlation
Analysis and differs from the MB-PLS approaches proposed by Kowalski et al.,
1989, J Chemom 3(1) and Westerhuis et al., 1998, J Chemom, 12(5).
}
\examples{
# Example with TCGA multi omics study
# -----------------------------------
data("breast.TCGA")
# this is the X data as a list of mRNA and miRNA; the Y data set is a single data set of proteins
data = list(mrna = breast.TCGA$data.train$mrna, mirna = breast.TCGA$data.train$mirna)
# set up a full design where every block is connected
design = matrix(1, ncol = length(data), nrow = length(data),
dimnames = list(names(data), names(data)))
diag(design) =  0
design
# set number of component per data set
ncomp = c(2)

TCGA.block.pls = block.pls(X = data, Y = breast.TCGA$data.train$protein, 
                           ncomp = ncomp, design = design)
TCGA.block.pls
## use design = 'full'
TCGA.block.pls = block.pls(X = data, Y = breast.TCGA$data.train$protein, 
                           ncomp = ncomp, design = 'full')
# in plotindiv we color the samples per breast subtype group but the method is unsupervised!
# here Y is the protein data set
plotIndiv(TCGA.block.pls, group =  breast.TCGA$data.train$subtype, ind.names = FALSE)
}
\references{
Tenenhaus, M. (1998). \emph{La regression PLS: theorie et
pratique}. Paris: Editions Technic.

Wold H. (1966). Estimation of principal components and related models by
iterative least squares. In: Krishnaiah, P. R. (editors), \emph{Multivariate
Analysis}. Academic Press, N.Y., 391-420.

Tenenhaus A. and Tenenhaus M., (2011), Regularized Generalized Canonical
Correlation Analysis, Psychometrika, Vol. 76, Nr 2, pp 257-284.
}
\seealso{
\code{\link{plotIndiv}}, \code{\link{plotArrow}},
\code{\link{plotLoadings}}, \code{\link{plotVar}}, \code{\link{predict}},
\code{\link{perf}}, \code{\link{selectVar}}, \code{\link{block.spls}},
\code{\link{block.plsda}} and http://www.mixOmics.org for more details.
}
\author{
Florian Rohart, Benoit Gautier, Kim-Anh Lê Cao, Al J Abadi
}
\keyword{multivariate}
\keyword{regression}
