% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/misty.R
\name{run_misty}
\alias{run_misty}
\title{Train MISTy models}
\usage{
run_misty(
  views,
  results.folder = "results",
  seed = 42,
  target.subset = NULL,
  bypass.intra = FALSE,
  cv.folds = 10,
  cached = FALSE,
  append = FALSE,
  model.function = random_forest_model,
  ...
)
}
\arguments{
\item{views}{view composition.}

\item{results.folder}{path to the top level folder to store raw results.}

\item{seed}{seed used for random sampling to ensure reproducibility.}

\item{target.subset}{subset of targets to train models for. If \code{NULL},
models will be trained for markers in the intraview.}

\item{bypass.intra}{a \code{logical} indicating whether to train a baseline
model using the intraview data (see Details).}

\item{cv.folds}{number of cross-validation folds to consider for estimating
the performance of the multi-view models}

\item{cached}{a \code{logical} indicating whether to cache the trained models
and to reuse previously cached ones if they already exist for this sample.}

\item{append}{a \code{logical} indicating whether to append the performance
and coefficient files in the \code{results.folder}. Consider setting to
\code{TRUE} when rerunning a workflow with different \code{target.subset}
parameters.}

\item{model.function}{a function which is used to model each view, default
model is \code{random_forest_model}. Other models included in mistyR are
\code{gradient_boosting_model}, \code{bagged_mars_model},
\code{mars_model}, \code{linear_model},
\code{svm_model}, \code{mlp_model}}

\item{...}{all additional parameters are passed to the chosen ML model for
training the view-specific models}
}
\value{
Path to the results folder that can be passed to
    \code{\link{collect_results}()}.
}
\description{
Trains multi-view models for all target markers, estimates the performance,
the contributions of the view specific models and the importance of predictor
markers for each target marker.
}
\details{
If \code{bypass.intra} is set to \code{TRUE} all variable in the intraview
the intraview data will be treated as targets only. The baseline intraview
model in this case is a trivial model that predicts the average of each
target. If the intraview has only one variable this switch is automatically
set to \code{TRUE}.

\emph{Default} model to train the view-specific views is a Random Forest model
based on \code{\link[ranger]{ranger}()} --  
\code{run_misty(views, model.function = random_forest_model)}
  
The following parameters are the default
configuration: \code{num.trees = 100}, \code{importance = "impurity"},
\code{num.threads = 1}, \code{seed = seed}.

\emph{Gradient boosting} is an alternative to model each view using gradient 
boosting. The algorithm is based on \code{\link[xgboost]{xgb.train}()} --
 \code{run_misty(views, model.function = gradient_boosting_model)} 

The following parameters are the default configuration: \code{booster = "gbtree"},
\code{rounds = 10}, \code{objective = "reg:squarederror"}. Set \code{booster}
to \code{"gblinear"} for linear boosting.

\emph{Bagged MARS} is an alternative to model each view using bagged MARS,
(multivariate adaptive spline regression models) trained with 
bootstrap aggregation samples. The algorithm is based on 
\code{\link[earth]{earth}()} --
\code{run_misty(views, model.function = bagged_mars_model)}

The following parameters are the default configuration: \code{degree = 2}.
Furthermore 50 base learners are used by default (pass \code{n.bags} as
parameter via \code{...} to change this value).

\emph{MARS} is an alternative to model each view using 
multivariate adaptive spline regression model. The algorithm is based on 
\code{\link[earth]{earth}()} --
\code{run_misty(views, model.function = mars_model)}

The following parameters are the default configuration: \code{degree = 2}.

\emph{Linear model} is an alternative to model each view using a simple linear
model. The algorithm is based on \code{\link[stats]{lm}()} --
\code{run_misty(views, model.function = linear_model)}

\emph{SVM} is an alternative to model each view using a support vector 
machines. The algorithm is based on \code{\link[kernlab]{ksvm}()} --
\code{run_misty(views, model.function = svm_model)}

The following parameters are the default configuration: \code{kernel = "vanilladot"} 
(linear kernel), \code{C = 1}, \code{type = "eps-svr"}.

\emph{MLP} is an alternative to model each view using a multi-layer perceptron.
The alogorithm is based on \code{\link[RSNNS]{mlp}()} --
\code{run_misty(views, model.function = mlp_model)}

The following parameters are the default configuration: \code{size = c(10)} 
(meaning we have 1 hidden layer with 10 units).
}
\examples{
# Create a view composition of an intraview and a paraview with radius 10 then
# run MISTy for a single sample.

library(dplyr)

# get the expression data
data("synthetic")
expr <- synthetic[[1]] \%>\% select(-c(row, col, type))
# get the coordinates for each cell
pos <- synthetic[[1]] \%>\% select(row, col)

# compose
misty.views <- create_initial_view(expr) \%>\% add_paraview(pos, l = 10)

# run with default parameters
run_misty(misty.views)
}
\seealso{
\code{\link{create_initial_view}()} for
    starting a view composition.
}
