#' methInheritSim: Simulating Whole-Genome Inherited Bisulphite Sequencing Data
#'
#' This package generates simulations of multigeneration of bisulfite data 
#' using a real control dataset.
#'
#' @docType package
#'
#' @name methInheritSim-package
#'
#' @aliases methInheritSim-package methInheritSim
#'
#' @author Pascal Belleau,
#' Astrid Deschênes and 
#' Arnaud Droit
#'
#' Maintainer:
#' Pascal Belleau <pascal_belleau@hotmail.com>
#'
#' @seealso
#' \itemize{
#' \item \code{\link{runSim}} { for simulating a multigeneration methylation 
#' experiment with inheritance }
#' }
#'
#' @keywords package
NULL


#' All samples information, formated by \code{methylKit}, in a
#' \code{methylBase} format (for demo purpose).
#'
#' The object is a \code{methylBase}.
#' There is 12 samples (6 controls and 6 cases). Each
#' sample information is stored in a \code{methylRaw} object.
#' 
#' This dataset can be
#' used to test the \code{runSim} function.
#' 
#' @name samplesForChrSynthetic
#'
#' @docType data
#'
#' @aliases samplesForChrSynthetic
#'
#' @format A \code{methylBase} object contains the information for one 
#' generation. Each sample information is
#' stored in a \code{methylRaw} object. There is \code{methylRaw} objects
#'  (6 controls and 6 cases).
#'
#' @return A \code{methylBase} contains the information for one generation.
#' Each sample information is stored in a \code{methylRaw} object. 
#' There is \code{methylRaw} objects
#' (6 controls and 6 cases).
#'
#' @seealso
#' \itemize{
#' \item \code{\link{runSim}} {for running a
#' simulation analysis using methylKit info as input}
#' }
#'
#' @usage data(samplesForChrSynthetic)
#'
#' @keywords datasets
#'
#' @examples
#'
#' ## Loading dataset
#' data(samplesForChrSynthetic)
#'
#' ## Set the output directory where files will be created
#' temp_dir <- "test_samplesForChrSynthetic"
#' 
#' ## Create 4 simulated dataset (nbSimulation) 
#' ## over 3 generations (nbGenration = 3) with
#' ## 6 cases and 6 controls (nNbsample = 6) using only one set
#' ## of parameters (vpDiff = 0.85, vpDiffsd = 0.1, vDiff = 0.8)
#' runSim(outputDir = temp_dir, fileID = "F1", nbSynCHR = 1, 
#'     methData = samplesForChrSynthetic, nbSimulation = 4, 
#'     nbBlock = 10, nbCpG = 20,
#'     nbGeneration = 3, vNbSample = c(6), vpDiff = c(0.85), 
#'     vpDiffsd = c(0.1), vDiff = c(0.8), 
#'     vInheritance = c(0.5), propInherite = 0.3,
#'     rateDiff = 0.3, minRate = 0.2, propHetero = 0.5, 
#'     nbCores = 1, vSeed = 32)
#'
#' ## Delete the output directory and its content
#' if (dir.exists(temp_dir)) {
#'     unlink(temp_dir, recursive = TRUE, force = FALSE)
#' }
#' 
NULL


#' A \code{list} containing methylation information used by some internal 
#' functions (for demo purpose. 
#' 
#' @name dataSimExample
#'
#' @docType data
#'
#' @aliases dataSimExample
#'
#' @format a \code{list} containing:
#' \itemize{
#' \item \code{stateInfo} a \code{GRanges}, a synthetic chromosome as 
#' generated by \code{getSyntheticChr} function. 
#' \item \code{stateDiff} a \code{list} containing:
#' \itemize{
#' \item \code{stateDiff} a \code{vector} of \code{integer} (\code{0} 
#' and \code{1}) with length corresponding the length of \code{stateInfo}.
#' The \code{vector}
#' indicates, using \code{1}, the positions where the CpG sites are
#' differentially methylated.
#' \item \code{stateInherite} a \code{vector} of \code{integer} (\code{0} and 
#' \code{1}) with length corresponding the length of \code{stateInfo}. The 
#' \code{vector}
#' indicates, using \code{1}, the positions where the CpG values are
#' inherited.
#' }
#' \item \code{treatment} a \code{vector} of \code{integer} (\code{0} 
#' and \code{1}) with length corresponding the number of samples. The 
#' \code{vector} indicates which samples are control (0) which samples are 
#' case (1).
#' \item \code{sample.id} a \code{list} of 3 \code{list}. Each entry of the
#' \code{list} correspond to one generation (first entry = first generation, 
#' etc..). Each  
#' \code{list} contains a \code{list} of 12 entries each containing a string
#' of \code{character}, the name of the sample. 
#' }
#'
#' @return  a \code{list} containing:
#' \itemize{
#' \item \code{stateInfo} a \code{GRanges}, a synthetic chromosome as 
#' generated by \code{getSyntheticChr} function. 
#' \item \code{stateDiff} a \code{list} containing:
#' \itemize{
#' \item \code{stateDiff} a \code{vector} of \code{integer} (\code{0} 
#' and \code{1}) with length corresponding the length of \code{stateInfo}.
#' The \code{vector}
#' indicates, using \code{1}, the positions where the CpG sites are
#' differentially methylated.
#' \item \code{stateInherite} a \code{vector} of \code{integer} (\code{0} and 
#' \code{1})
#' with length corresponding the length of \code{stateInfo}. The 
#' \code{vector}
#' indicates, using \code{1}, the positions where the CpG values are
#' inherited.
#' }
#' \item \code{treatment} a \code{vector} of \code{integer} (\code{0} 
#' and \code{1}) with length corresponding the number of samples. The 
#' \code{vector} indicates which samples are control (0) which samples are 
#' case (1).
#' \item \code{sample.id} a \code{list} of 3 \code{list}. Each entry of the
#' \code{list} correspond to one generation (first entry = first generation, 
#' etc..). Each  
#' \code{list} contains a \code{list} of 12 entries each containing a string
#' of \code{character}, the name of the sample.
#' }
#'
#' @seealso
#' \itemize{
#' \item \code{\link{runSim}} {for running a
#' simulation analysis using methylKit info as input}
#' }
#'
#' @usage data(dataSimExample)
#'
#' @keywords datasets
#'
#' @examples
#'
#' ## Loading dataset
#' data(dataSimExample)
#'
#' ## Identify differentially methylated sites and among those, the ones
#' ## that are inherited
#' methInheritSim:::getDiffMeth(stateInfo = dataSimExample$stateInfo, 
#'     rateDiff = 0.2, minRate = 0.3,propInherite = 0.3)
#' 
NULL
