#' Plot gistic results as a bubble plot
#' @description Plots significantly altered cytobands as a function of number samples in which it is altered and number genes it contains. Size of each bubble is according to -log10 transformed q values.
#' @param gistic an object of class \code{GISTIC} generated by \code{readGistic}
#' @param fdrCutOff fdr cutoff to use. Default 0.1
#' @param markBands any cytobands to label. Can be cytoband labels, or number of top bands to highlight. Default top 5 lowest q values.
#' @param color colors for Amp and Del events.
#' @param txtSize label size for bubbles.
#' @param log_y log10 scale y-axis (# genes affected). Default TRUE
#' @examples
#' all.lesions <- system.file("extdata", "all_lesions.conf_99.txt", package = "maftools")
#' amp.genes <- system.file("extdata", "amp_genes.conf_99.txt", package = "maftools")
#' del.genes <- system.file("extdata", "del_genes.conf_99.txt", package = "maftools")
#' scores.gistic <- system.file("extdata", "scores.gistic", package = "maftools")
#' laml.gistic = readGistic(gisticAllLesionsFile = all.lesions, gisticAmpGenesFile = amp.genes, gisticDelGenesFile = del.genes, gisticScoresFile = scores.gistic)
#' gisticBubblePlot(gistic = laml.gistic, markBands = "")
#' @return Nothing
#' @export
#'
gisticBubblePlot = function(gistic = NULL, color = NULL, markBands = NULL, fdrCutOff = 0.1, log_y = TRUE,
                            txtSize = 3){

  g = getCytobandSummary(gistic)
  g = g[qvalues < fdrCutOff]
  g[,Chromosome := sapply(strsplit(x = g$Wide_Peak_Limits, split = ':'), '[', 1)]
  g[,loc := sapply(strsplit(x = g$Wide_Peak_Limits, split = ':'), '[', 2)]
  g[,Start_Position := sapply(strsplit(x = g$loc, split = '-'), '[', 1)]
  g[,End_Position := sapply(strsplit(x = g$loc, split = '-'), '[', 2)]
  g.lin = transformSegments(segmentedData = g[,.(Chromosome, Start_Position, End_Position, qvalues, Cytoband, Variant_Classification)])

  if(is.null(color)){
    color = c('Amp' = grDevices::adjustcolor('red', alpha.f = 0.8),
              'Del' = grDevices::adjustcolor('blue', alpha.f = 0.8))
  }

  g$lab = sapply(strsplit(x = g$Unique_Name, split = ':'), '[', 2)
  if(log_y){
    g[, nGenes := log10(nGenes)]
  }
  g$pos = ifelse(test = g$Variant_Classification %in% 'Amp', yes = g$nGenes, no = -1 * g$nGenes)

  g$color = ifelse(test = g$Variant_Classification == "Amp", yes = color[1], no = color[2])
  g[, log_q := -log10(qvalues)]


  if(is.null(markBands)){
    markBands = g[1:5, Cytoband]
  }else{
    if(is.numeric(markBands)){
      markBands = g[1:markBands, Cytoband]
    }else if(markBands[1] == "all"){
      markBands = g[, Cytoband]
    }
  }

  if(!is.null(markBands)){
    g.labs = g[Cytoband %in% markBands]
    if(nrow(g.labs) == 0){
      warning("Could not find provided bands for labelling", immediate. = TRUE)
      g.labs = NULL
    }
  }

  par(mar = c(4, 4, 2, 0))
  bubble_plot(
    plot_dat = g,
    lab_dat = g.labs,
    x_var = "nSamples",
    y_var = "pos",
    bubble_var = "log_q",
    text_var = "lab",
    col_var = "color",
    return_dat = FALSE,
    showscale = TRUE,
    xlab = "# samples",
    ylab = ifelse(test = log_y, yes = "# Genes (log10)", no = "# Genes")
  )
}
