% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/chargeCalculations.R
\name{chargeCalculationLocal}
\alias{chargeCalculationLocal}
\title{Charge Calculation Along a Protein Sequence}
\usage{
chargeCalculationLocal(
  sequence,
  window = 9,
  proteinName = NA,
  pH = 7,
  pKaSet = "IPC_protein",
  printCitation = FALSE,
  plotResults = FALSE,
  ...
)
}
\arguments{
\item{sequence}{amino acid sequence as a single character string
or vector of single characters.
It also supports a single character string that specifies
the location of a .fasta or .fa file.}

\item{window}{a positive, odd integer. 9 by default.
Sets the size of sliding window, must be an odd number.
The window determines the number of residues to be analyzed and averaged
for each position along the sequence.}

\item{proteinName}{character string, optional. Used to add protein name
to the title in ggplot. Ignored if \code{plotResults = FALSE}.}

\item{pH}{numeric value, 7.0 by default.
The environmental pH used to calculate residue charge.}

\item{pKaSet}{A character string or data frame. "IPC_protein" by default.
Character string to load specific, preloaded pKa sets.
 c("EMBOSS", "DTASelect", "Solomons", "Sillero", "Rodwell",
  "Lehninger", "Toseland", "Thurlkill", "Nozaki", "Dawson",
  "Bjellqvist", "ProMoST", "Vollhardt", "IPC_protein", "IPC_peptide")
 Alternatively, the user may supply a custom pKa dataset.
 The format must be a data frame where:
 Column 1 must be a character vector of residues named "AA" AND
 Column 2 must be a numeric vector of pKa values.}

\item{printCitation}{Logical value. FALSE by default.
When \code{printCitation = TRUE} the citation for the pKa set is printed.
This allows for the user to easily obtain the dataset citation.
Will not print if there is a custom dataset.}

\item{plotResults}{logical value. TRUE by default.
If \code{plotResults = TRUE}, a ggplot of window charges are returned.
If \code{plotResults = FALSE}, a data frame of window charges are returned.}

\item{...}{any additional parameters, especially those for plotting.}
}
\value{
see plotResults argument
}
\description{
This calculates the charge, as determined by the Henderson-Hasselbalch
  equation, for each window along the sequence. This function uses a
  sliding window. The output is either a graph or a data frame of
  calculated charges.
}
\section{Plot Colors}{

  For users who wish to keep a common aesthetic, the following colors are
  used when plotResults = TRUE. \cr
  \itemize{
  \item Dynamic line colors: \itemize{
  \item Close to -1 = "#92140C"
  \item Close to +1 = "#348AA7"
  \item Close to 0 (midpoint) = "grey65" or "#A6A6A6"}}
}

\examples{
 #Amino acid sequences can be character strings
aaString <- "ACDEFGHIKLMNPQRSTVWY"
#Amino acid sequences can also be character vectors
aaVector <- c("A", "C", "D", "E", "F",
              "G", "H", "I", "K", "L",
              "M", "N", "P", "Q", "R",
              "S", "T", "V", "W", "Y")
#Alternatively, .fasta files can also be used by providing
# a character string of the path to the file.
exampleDF <- chargeCalculationLocal(aaString)
exampleDF <- chargeCalculationLocal(aaVector)
head(exampleDF)

#Changing window will alter the number of residues analyzed
exampleDF_window3 <- chargeCalculationLocal(aaString,
                                            window = 3)
head(exampleDF_window3)
exampleDF_window15 <- chargeCalculationLocal(aaString,
                                             window = 15)
head(exampleDF_window15)

#Changing pKa set or pH used for calculations
exampleDF_pH5 <- chargeCalculationLocal(aaString,
                                        pH = 5)
head(exampleDF_pH5)
exampleDF_pH7 <- chargeCalculationLocal(aaString,
                                       pH = 7)
head(exampleDF_pH7)
exampleDF_EMBOSS <- chargeCalculationLocal(aaString,
                                           pH = 7,
                                           pKa = "EMBOSS")
head(exampleDF_EMBOSS)

#plotResults = TRUE will output a ggplot
  chargeCalculationLocal(aaString,
                         plot = TRUE)

  #since it is a ggplot, you can change or annotate the plot
  gg <- chargeCalculationLocal(aaVector,
                               window = 3,
                               plot = TRUE)
  gg <- gg + ggplot2::ylab("Local Charge")
  gg <- gg + ggplot2::geom_text(data = exampleDF_window3,
                                ggplot2::aes(label = CenterResidue,
                                             y = windowCharge + 0.1))
 plot(gg)
}
\seealso{
\code{\link{pKaData}} for residue pKa values and citations. See
  \code{\link{hendersonHasselbalch}} for charge calculations.

Other charge functions: 
\code{\link{chargeCalculationGlobal}()},
\code{\link{hendersonHasselbalch}()},
\code{\link{netCharge}()}
}
\concept{charge functions}
