\name{geom_h5vc}
\alias{geom_h5vc}

\title{geom_h5vc}
\description{
Plotting function that returns a \code{ggplot2} layer representing the specified dataset for the specified samples in the region \code{[positon - windowsize, position + windowsize]}.
}
\usage{
geom_h5vc( data, sampledata, samples=sampledata$Sample, windowsize, position, dataset, ... )
}

\arguments{
  \item{data}{ The data to be plotted. Returned by \code{h5dapply}. Must
  be centered on \code{position}, extend by \code{windowsize} in each
  direction and contain a slot named like the \code{dataset} argument}
  \item{sampledata}{ The sampledata for the cohort represented by
    \code{data}. Returned by \code{getSampleData} }
  \item{samples}{A character vector listing the names of samples to be
    plotted, defaults to all samples as described in \code{sampledata}}
  \item{windowsize}{Size of the window in which to plot on each
    side. The total interval that is plotted will be
    [position-windowsize,position+windowsize]}
  \item{position}{The position at which the plot shall be centered}
  \item{dataset}{The slot in the \code{data} argument that should be plotted}
  \item{...}{Paramteters to be passed to the internally used \code{geom_rect}, see \code{\link[ggplot2]{geom_rect}} for details}
}
\details{
  Creates a ggplot layer centered on \code{position} using the specified \code{dataset} from list \code{data}, annotating it with sample information provided in the data.frame \code{sampledata} and showing all samples listed in \code{sample}. The resulting plot uses \code{ggplot2}'s \code{geom_rect} to draw boxes representing the values from \code{dataset}. The x-axis is the position and will span the interval \code{[positon - windowsize, position + windowsize]}. The x-axis is centered at 0 and additional layers to be added to the plot should be centered at 0 also.
  
  Ths function allows for fast creation of overview plots similar to \code{\link{mismatchPlot}} (without the stacking of tracks). The example below shows how one can create a plot showing the coverage and number of mismatches per position (but not the alternative allele) for a given region.
}
\value{
  A \code{ggplot} layer object containing the plot of the specified dataset, this can be used
  like any other ggplot layer, i.e. it may be added to another plot.
  }
\author{
Paul Pyl
}

\examples{
  # loading library and example data
  library(h5vc)
  library(ggplot2)
  tallyFile <- system.file( "extdata", "example.tally.hfs5", package = "h5vcData" )
  sampleData <- getSampleData( tallyFile, "/ExampleStudy/16" )
  position <- 29979629
  windowsize <- 30
  samples <- sampleData$Sample[sampleData$Patient == "Patient8"]
  data <- h5dapply(
    filename = tallyFile,
    group = "/ExampleStudy/16",
    blocksize = windowsize * 3, #choose blocksize larger than range so that all needed data is collected as one block
    names = c("Coverages", "Counts", "Deletions"),
    range = c(position - windowsize, position + windowsize)
  )[[1]]
  # Summing up all mismatches irrespective of the alternative allele
  data$CountsAggregate = colSums(data$Counts)
  # Simple overview plot showing number of mismatches per position
  p <- ggplot() + 
  geom_h5vc( data=data, sampledata=sampleData, windowsize = 35, position = 500, dataset = "Coverages", fill = "gray" ) + 
  geom_h5vc( data=data, sampledata=sampleData, windowsize = 35, position = 500, dataset = "CountsAggregate", fill = "#D50000" ) + 
  facet_wrap( ~ Sample, ncol = 2 )
  print(p)
}