% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/internal_functions.R
\name{intern.crude}
\alias{intern.crude}
\alias{intern.focus}
\alias{intern.conva}
\title{Internal function}
\usage{
intern.crude(y, X, mu, phi, perm)

intern.focus(y, X, mu, phi, perm, focus)

intern.conva(y, X, mu, phi, perm, offset)
}
\arguments{
\item{y}{response variable: numeric vector of length \code{n}}

\item{X}{covariate set: numeric matrix with \code{n} rows (samples)
and \code{p} columns (covariates)}

\item{mu}{mean parameters: numeric vector of length \code{n}}

\item{phi}{dispersion parameter: non-negative real number}

\item{perm}{permutations: matrix with \code{n} rows (see example)}

\item{focus}{number between 0 and 1}

\item{offset}{numeric vector of length \code{n}}
}
\value{
Each function returns a dataframe,
with the p-value in the first row,
and the test statistic in the second row.
}
\description{
Using the parameter estimates \code{mu} and \code{phi}
and the permutation matrix \code{perm}, these functions
tests for global association between \code{y} and \code{X}.
The function \code{\link{intern.crude}} calculates
p-values by permutation (without repetitions).
The functions \code{\link{intern.focus}} and
\code{\link{intern.conva}} use different tricks
to increase precision and decrease computational expense.
}
\details{
The function \code{\link{intern.focus}}
uses permutations in chunks.
If the remaining permutations do not allow
to reach a specified significance level,
it stops and rounds the p-value to one.

The function \code{\link{intern.conva}}
uses the method of control variates
from Senchaudhuri et al. (1995).
Roughly speaking, if the test statistics
from Rauschenberger et al. (2016)
and Goeman et al. (2004) are highly correlated,
it returns the asymptotic p-value from Goeman et al. (2004).
}
\examples{
# simulate high-dimensional data
n <- 30
p <- 100
# set.seed(1)
y <- rnbinom(n,mu=10,size=1/0.25)
X <- matrix(rnorm(n*p),nrow=n,ncol=p)

# prepare arguments
mu <- rep(mean(y),n)
phi <- (var(y)-mu)/mu^2
perm <- intern.permu(n=n,it=99,group=NULL,kind=1)

# perform tests
intern.crude(y,X,mu,phi,perm)
intern.focus(y,X,mu,phi,perm,focus=0.01)
intern.conva(y,X,mu,phi,perm,NULL)

}
\references{
P Senchaudhuri, CR Mehta, and NR Patel (1995).
"Estimating exact p values by the method of control variates
or Monte Carlo rescue",
\emph{Journal of the American Statistical Association}.
90:640-648
\href{http://dx.doi.org/10.2307/2291077}{html}
\href{http://www.jstor.org/stable/pdf/2291077.pdf?acceptTC=true}{pdf}
(restricted access)

A Rauschenberger, MA Jonker, MA van de Wiel, and RX Menezes (2016).
"Testing for association between RNA-Seq and high-dimensional data",
\emph{BMC Bioinformatics}. 17:118.
\href{http://dx.doi.org/10.1186/s12859-016-0961-5}{html}
\href{http://www.biomedcentral.com/content/pdf/s12859-016-0961-5.pdf}{pdf}
(open access)

JJ Goeman, SA van de Geer, F de Kort, and HC van Houwelingen (2004).
"A global test for groups of genes:
testing association with a clinical outcome",
\emph{Bioinformatics}. 20:93-99.
\href{http://dx.doi.org/10.1093/bioinformatics/btg382}{html}
\href{http://bioinformatics.oxfordjournals.org/content/20/1/93.full.pdf}{pdf}
(open access)
}
\seealso{
These are \code{\link{internal}} functions. The user functions 
of the R package \code{\link{globalSeq}} are \code{\link{cursus}},
\code{\link{omnibus}}, and \code{\link{proprius}}.
}
\keyword{internal}
