#' Save a Visualization to File (PDF, PNG, or SVG)
#'
#' This function saves a visualization object to a file in the specified
#' format and directory. It supports visualizations generated by
#' `plotVenn()`, `plotUpSet()`, `ggplot2`, or any other plot object that
#' can be rendered using `print()` inside a graphics device. Optionally,
#' the current date (stored in the `today` variable) can be prepended to
#' the filename.
#'
#' @param viz A visualization object typically created by either
#'   `plotVenn()` or `plotUpSet()`, but can also be a `ggplot2` plot or
#'   any other plot object printable with `print()`.
#' @param output_dir A string specifying the output directory. Defaults
#'   to `"."`.
#' @param output_file A string specifying the base filename (without
#'   extension). Defaults to `"viz_genomicVenn"`.
#' @param format Output format. One of `"pdf"`, `"png"`, or `"svg"`.
#'   Defaults to `"pdf"`.
#' @param bg Background color for the plot. Default is `"white"`.
#'   Use `"transparent"` for a transparent background. This parameter
#'   is passed to the graphics device (pdf, png, or svg).
#' @param with_date Logical (default `TRUE`). Whether to prepend the
#'   current date (from `today`) to the filename.
#' @param width Width of the output file in inches. Default is 5.
#' @param height Height of the output file in inches. Default is 5.
#' @param resolution Resolution in DPI (only used for PNG). Default is
#'   300.
#' @param verbose Logical. If `TRUE`, print a message with the saved
#'   path. Default `TRUE`.
#'
#' @return The visualization is saved to a file on disk. Invisibly
#'   returns the full path to the saved file.
#' @export
#'
#' @examples
#' # Example with a built-in set dataset
#'   data(gene_list)
#'   ov_sets <- computeOverlaps(gene_list)
#'   venn_plot <- plotVenn(ov_sets)
#'   saveViz(venn_plot, output_dir = tempdir(), output_file = "venn_sets")
#'
#'   # Example with a built-in genomic dataset
#'   data(a549_chipseq_peaks)
#'   ov_genomic <- computeOverlaps(a549_chipseq_peaks)
#'   upset_plot <- plotUpSet(ov_genomic)
#'   saveViz(upset_plot, output_dir = tempdir(), output_file = "upset_genomic")
#'
#'   # Save as PNG instead of PDF
#'   saveViz(upset_plot, format = "png", output_dir = tempdir(), output_file = "upset_example")
#'
#'   # Save as SVG
#'   saveViz(venn_plot, format = "svg", output_dir = tempdir(), output_file = "venn_example")
#'
#'   # Save with transparent background
#'   saveViz(venn_plot, format = "png", bg = "transparent",
#'           output_dir = tempdir(), output_file = "venn_transparent")
saveViz <- function(viz,
                    output_dir = ".",
                    output_file = "figure_gVenn",
                    format = "pdf",
                    with_date = TRUE,
                    width = 5,
                    height = 5,
                    resolution = 300,
                    bg = "white",
                    verbose = TRUE) {
    format <- match.arg(format, choices = c("pdf", "png", "svg"))

    if (!dir.exists(output_dir)) {
        dir.create(output_dir, recursive = TRUE)
    }

    if (isTRUE(with_date)) {
        output_file <- paste0(today, "_", output_file)
    }

    filepath <- file.path(output_dir, paste0(output_file, ".", format))

    switch(format,
           pdf = grDevices::pdf(file = filepath, width = width, height = height, bg = bg),
           png = grDevices::png(file = filepath, width = width, height = height, units = "in", res = resolution, bg = bg),
           svg = grDevices::svg(file = filepath, width = width, height = height, bg = bg))

    print(viz)
    grDevices::dev.off()

    if (isTRUE(verbose)) {
        message(" > Visualization (", format, ") saved in ", filepath)
    }
    invisible(filepath)
}
