% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/runEISA.R
\name{runEISA}
\alias{runEISA}
\title{Run Exon-Intron Split Analysis.}
\usage{
runEISA(
  cntEx,
  cntIn,
  cond,
  method = NULL,
  modelSamples = TRUE,
  geneSelection = c("filterByExpr", "none", "Gaidatzis2015"),
  statFramework = c("QLF", "LRT"),
  legacyQLF = FALSE,
  effects = c("predFC", "Gaidatzis2015"),
  pscnt = 2,
  sizeFactor = c("exon", "intron", "individual"),
  recalcNormFactAfterFilt = TRUE,
  recalcLibSizeAfterFilt = FALSE,
  ...
)
}
\arguments{
\item{cntEx}{Gene by sample \code{matrix} with exonic counts, OR a
\code{SummarizedExperiment} with two assays named \code{exon} and
\code{intron}, containing exonic and intronic counts, respectively. If
\code{cntEx} is a \code{SummarizedExperiment}, \code{cntIn} will be
disregarded.}

\item{cntIn}{Gene by sample \code{matrix} with intronic counts. Must have the
same structure as \code{cntEx} (same number and order of rows and columns)
if \code{cntEx} is a matrix. Will be disregarded if \code{cntEx} is a
\code{SummarizedExperiment}.}

\item{cond}{\code{numeric}, \code{character} or \code{factor} with two levels
that groups the samples (columns of \code{cntEx} and \code{cntIn}) into two
conditions. The contrast will be defined as secondLevel - firstLevel.}

\item{method}{One of \code{NULL} (the default) or \code{"Gaidatzis2015"}. If
\code{"Gaidatzis2015"}, gene filtering, statistical analysis and
calculation of contrasts is performed as described in Gaidatzis et al.
2015, and the statistical analysis is based on \code{\link[edgeR]{glmFit}}
and \code{\link[edgeR:glmFit]{glmLRT}}. This is done by setting the arguments
\code{modelSamples}, \code{geneSelection}, \code{effects}, \code{pscnt},
\code{statFramework}, \code{sizeFactor}, \code{recalcNormFactAfterFilt}
and \code{recalcLibSizeAfterFilt}
to appropriate values (see details), overriding the defaults or any value
passed to these arguments. If \code{NULL}, the default values of the arguments
will be used instead (recommended).}

\item{modelSamples}{Whether to include a sample identifier in the design matrix
of the statistical model. If \code{TRUE}, potential sample effects
that affect both exonic and intronic counts of that sample will be taken
into account, which could result in higher sensitivity (default: \code{TRUE}).}

\item{geneSelection}{Controls how to select quantifyable genes. One of the
following:\describe{
\item{\code{"filterByExpr"}: }{(default) First, counts are normalized using
\code{\link[edgeR]{calcNormFactors}}, treating intronic and exonic counts
as individual samples. Then, \code{\link[edgeR]{filterByExpr}} is used
with default parameters to select quantifyable genes.}
\item{\code{"none"}: }{This will use all the genes provided in the count
tables, assuming that an appropriate selection of quantifyable genes has
already been done.}
\item{\code{"Gaidatzis2015"}: }{First, intronic and exonic counts are
linearly scaled to the mean library size (estimated as the sum of all
intronic or exonic counts, respectively). Then, quantifyable genes are
selected as the genes with counts \code{x} that fulfill
\code{log2(x + 8) > 5} in both exons and introns.}
}}

\item{statFramework}{Selects the framework within \code{edgeR} that is used
for the statistical analysis. One of:\describe{
\item{\code{"QLF"}: }{(default) Quasi-likelihood F-test using
\code{\link[edgeR:glmQLFTest]{glmQLFit}} and \code{\link[edgeR]{glmQLFTest}}. This
framework is highly recommended as it gives stricter error rate control
by accounting for the uncertainty in dispersion estimation.}
\item{\code{"LRT"}: }{Likelihood ratio test using \code{\link[edgeR]{glmFit}}
and \code{\link[edgeR:glmFit]{glmLRT}}}.
}}

\item{legacyQLF}{Whether to use the 'legacy' version of
\code{\link[edgeR:glmQLFTest]{glmQLFit}}. See \code{\link[edgeR:glmQLFTest]{glmQLFit}}
for more details. If \code{FALSE}, the new method introduced in
\code{edgeR} 4.0.0 is used.}

\item{effects}{How the effects (contrasts or log2 fold-changes) are calculated.
One of:\describe{
\item{\code{"predFC"}: }{(default) Fold-changes are calculated using
the fitted model with \code{\link[edgeR]{predFC}} with
\code{prior.count = pscnt}. Please note that if a sample factor is
included in the model (\code{modelSamples=TRUE}), effects cannot be
obtained from that model. In that case, effects are obtained from a
simpler model without sample effects.}
\item{\code{"Gaidatzis2015"}: }{Fold-changes are calculated using the
formula \code{log2((x + pscnt)/(y + pscnt))}. If \code{pscnt} is not
set to 8, \code{runEISA} will warn that this deviates from the method
used in Gaidatzis et al., 2015.}
}}

\item{pscnt}{\code{numeric(1)} with pseudocount to add to read counts
(default: 2). For \code{method = "Gaidatzis2015"}, it is set to 8.
It is added to scaled read counts used in \code{geneSelection = "Gaidatzis2015"}
and \code{effects = "Gaidatzis2015"}, or else used in \code{cpm(..., prior.count = pscnt)}
and \code{predFC(..., prior.count = pscnt)}.}

\item{sizeFactor}{How the size factors are calculated in the analysis.
If 'exon' (default), the exon-derived size factors are used also for the
columns corresponding to intronic counts. If 'intron', the intron-derived
size factors are used also for the columns corresponding to exonic
counts. If 'individual', column-wise size factors are calculated.}

\item{recalcNormFactAfterFilt}{Logical, indicating whether normalization
factors should be recalculated after filtering out lowly
expressed genes.}

\item{recalcLibSizeAfterFilt}{Logical, indicating whether library
sizes should be recalculated after filtering out lowly
expressed genes.}

\item{...}{additional arguments passed to the \code{\link[edgeR]{DGEList}}
constructor, such as \code{lib.size} or \code{genes}.}
}
\value{
a \code{list} with elements \describe{
\item{fracIn}{fraction intronic counts in each sample}
\item{contrastName}{contrast name}
\item{contrasts}{contrast matrix for quantifyable genes, with average log2
fold-changes in exons (\code{Dex}), in introns (\code{Din}), and average
difference between log2 fold-changes in exons and introns (\code{Dex.Din})}
\item{DGEList}{\code{\link[edgeR]{DGEList}} object used in model fitting}
\item{tab.ExIn}{statisical results for differential changes between exonic
and intronic contrast, an indication for post-transcriptional regulation.}
\item{contr.ExIn}{contrast vector used for testing the difference between
exonic and intronic contrast (results in \code{tab.ExIn})}
\item{designMatrix}{design matrix used for testing the difference between
exonic and intronic contrast (results in \code{tab.ExIn})}
\item{params}{a \code{list} with parameter values used to run EISA}
}
}
\description{
Starting from count tables with exonic and intronic counts
for two conditions, perform all the steps in EISA (normalize, identify
quantifyable genes, calculate contrasts and their significance).
}
\details{
Setting \code{method = "Gaidatzis2015"} has precedence over other
argument values and corresponds to setting:
\code{modelSamples = FALSE, geneSelection = "Gaidatzis2015",
  statFramework = "LRT", effects = "Gaidatzis2015", pscnt = 8, 
  sizeFactor = "individual", recalcNormFactAfterFilt = TRUE, 
  recalcLibSizeAfterFilt = FALSE}.
}
\examples{
cntEx <- readRDS(system.file("extdata", "Fig3abc_GSE33252_rawcounts_exonic.rds",
                             package = "eisaR"))[,-1]
cntIn <- readRDS(system.file("extdata", "Fig3abc_GSE33252_rawcounts_intronic.rds",
                             package = "eisaR"))[,-1]
cond <- factor(c("ES","ES","TN","TN"))
res <- runEISA(cntEx, cntIn, cond)
plotEISA(res)

}
\references{
Analysis of intronic and exonic reads in RNA-seq data characterizes
transcriptional and post-transcriptional regulation.
Dimos Gaidatzis, Lukas Burger, Maria Florescu and Michael B. Stadler
Nature Biotechnology, 2015 Jul;33(7):722-9. doi: 10.1038/nbt.3269.
}
\seealso{
\code{\link[edgeR]{DGEList}} for \code{DGEList} object construction,
\code{\link[edgeR]{calcNormFactors}} for normalization,
\code{\link[edgeR]{filterByExpr}} for gene selection,
\code{\link[edgeR]{glmFit}} and \code{\link[edgeR:glmQLFTest]{glmQLFit}} for statistical
analysis.
}
\author{
Michael Stadler
}
