% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AllGenerics.R, R/deSet-methods.R
\docType{methods}
\name{kl_clust}
\alias{kl_clust}
\alias{kl_clust,deSet,deFit-method}
\alias{kl_clust,deSet,missing-method}
\title{Modular optimal discovery procedure (mODP)}
\usage{
kl_clust(object, de.fit = NULL, n.mods = 50)

\S4method{kl_clust}{deSet,missing}(object, de.fit = NULL, n.mods = 50)

\S4method{kl_clust}{deSet,deFit}(object, de.fit = NULL, n.mods = 50)
}
\arguments{
\item{object}{\code{S4 object}: \code{\linkS4class{deSet}}.}

\item{de.fit}{\code{S4 object}: \code{\linkS4class{deFit}}.}

\item{n.mods}{\code{integer}: number of modules (i.e., clusters).}
}
\value{
A list with the following slots:
\itemize{
  \item {mu.full: cluster averaged fitted values from full model.}
  \item {mu.null: cluster averaged fitted values from null model.}
  \item {sig.full: cluster standard deviations from full model.}
  \item {sig.null: cluster standard deviations from null model.}
  \item {n.per.mod: total members in each cluster.}
  \item {clustMembers: cluster membership for each gene.}
}
}
\description{
\code{kl_clust} is an implementation of mODP that assigns genes to modules
based on of the Kullback-Leibler distance.
}
\details{
mODP utilizes a k-means clustering algorithm where genes are
assigned to a cluster based on the Kullback-Leiber distance. Each gene is
assigned an module-average parameter to calculate the ODP score (See Woo,
Leek and Storey 2010 for more details). The mODP and full ODP produce nearly
exact results but mODP has the advantage of being computationally
faster.
}
\note{
The results are generally insensitive to the number of modules after a 
  certain threshold of about n.mods>=50 in our experience. It is recommended
  that users experiment with the number of modules. If the number of modules
  is equal to the number of genes then the original ODP is implemented.
}
\examples{
# import data
library(splines)
data(kidney)
age <- kidney$age
sex <- kidney$sex
kidexpr <- kidney$kidexpr
cov <- data.frame(sex = sex, age = age)

# create models
null_model <- ~sex
full_model <- ~sex + ns(age, df = 4)

# create deSet object from data
de_obj <- build_models(data = kidexpr, cov = cov, null.model = null_model,
full.model = full_model)

# mODP method
de_clust <- kl_clust(de_obj)

# change the number of clusters
de_clust <- kl_clust(de_obj, n.mods = 10)

# input a deFit object
de_fit <- fit_models(de_obj, stat.type = "odp")
de_clust <- kl_clust(de_obj, de.fit = de_fit)

}
\author{
John Storey, Jeffrey Leek
}
\references{
Storey JD. (2007) The optimal discovery procedure: A new approach to
simultaneous significance testing. Journal of the Royal Statistical
Society, Series B, 69: 347-368.

Storey JD, Dai JY, and Leek JT. (2007) The optimal discovery procedure for
large-scale significance testing, with applications to comparative
microarray experiments. Biostatistics, 8: 414-432.

Woo S, Leek JT, Storey JD (2010) A computationally efficient modular optimal
 discovery procedure. Bioinformatics, 27(4): 509-515.
}
\seealso{
\code{\link{odp}}, \code{\link{fit_models}}
}

