% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/explore_biomarkers.R
\name{explore_biomarkers}
\alias{explore_biomarkers}
\title{Explore biomarkers of immune response}
\usage{
explore_biomarkers(
  pathways = NULL,
  immunecells = NULL,
  tfs = NULL,
  lrpairs = NULL,
  ccpairs = NULL,
  cancer_type,
  patient_label = NULL,
  verbose = TRUE
)
}
\arguments{
\item{pathways}{numeric matrix with pathways activity
(rows = samples; columns = pathways). This is the
output from \code{compute_pathway_activity}.}

\item{immunecells}{numeric matrix with immune cell quantification
(rows = samples; columns = cell types). This is the
output from \code{compute_cell_fractions}.}

\item{tfs}{numeric matrix with transcription factors activity
(rows = samples; columns = transcription factors). This is the
output from \code{compute_TF_activity}.}

\item{lrpairs}{numeric matrix with ligand-receptor weights
(rows = samples; columns = ligand-receptor pairs). This is the
output from \code{compute_LR_pairs}.}

\item{ccpairs}{numeric matrix with cell-cell scores
(rows = samples; columns = cell-cell pairs). This is the
output from \code{compute_CC_pairs}.}

\item{cancer_type}{character string indicating which cancer-specific
model should be used to compute the predictions. This should be
available from the cancer-specific models. The following cancer types
have a corresponding model available: "BLCA", "BRCA", "CESC", "CRC",
"GBM", "HNSC", "KIRC", "KIRP", "LIHC", "LUAD", "LUSC", "NSCLC", "OV",
"PAAD", "PRAD", "SKCM", "STAD", "THCA" and "UCEC".}

\item{patient_label}{character vector with two factor levels,
e.g. NR (Non-responders) vs R (Responders), pre- vs on- treatment.}

\item{verbose}{logical flag indicating whether to display messages
about the process.}
}
\value{
\itemize{
\item{A combined plot for each type of quantitative descriptors,
showing the original distribution of the features and the importance
of these features for the trained models}
#' \item{Volcano plot displaying relevant biomarkers differentiating
responders vs non-responders patients.}
}
}
\description{
Provides a good overview of the computed features
(biomarkers) including the corresponding weights from the
trained model. If \code{patient_response} is provided,
this function shows statistically significant biomarkers
between responders (R) and non-responders (NR) patients.
}
\examples{
# using a SummarizedExperiment object
library(SummarizedExperiment)
# Using example exemplary dataset (Mariathasan et al., Nature, 2018)
# from easierData. Original processed data is available from
# IMvigor210CoreBiologies package.
library("easierData")

dataset_mariathasan <- easierData::get_Mariathasan2018_PDL1_treatment()
RNA_tpm <- assays(dataset_mariathasan)[["tpm"]]
cancer_type <- metadata(dataset_mariathasan)[["cancertype"]]

# Select a subset of patients to reduce vignette building time.
pat_subset <- c(
  "SAM76a431ba6ce1", "SAMd3bd67996035", "SAMd3601288319e",
  "SAMba1a34b5a060", "SAM18a4dabbc557"
)
RNA_tpm <- RNA_tpm[, colnames(RNA_tpm) \%in\% pat_subset]

# Computation of TF activity
tf_activity <- compute_TF_activity(
  RNA_tpm = RNA_tpm
)

# retrieve clinical response
patient_ICBresponse <- colData(dataset_mariathasan)[["BOR"]]
names(patient_ICBresponse) <- colData(dataset_mariathasan)[["pat_id"]]
patient_ICBresponse <- patient_ICBresponse[names(patient_ICBresponse) \%in\% pat_subset]

# Investigate possible biomarkers
output_biomarkers <- explore_biomarkers(
  tfs = tf_activity,
  cancer_type = cancer_type,
  patient_label = patient_ICBresponse
)

\donttest{

RNA_counts <- assays(dataset_mariathasan)[["counts"]]
RNA_counts <- RNA_counts[, colnames(RNA_counts) \%in\% pat_subset]

# Computation of cell fractions
cell_fractions <- compute_cell_fractions(RNA_tpm = RNA_tpm)

# Computation of pathway scores
pathway_activity <- compute_pathway_activity(
  RNA_counts = RNA_counts,
  remove_sig_genes_immune_response = TRUE
)

# Computation of ligand-receptor pair weights
lrpair_weights <- compute_LR_pairs(
  RNA_tpm = RNA_tpm,
  cancer_type = "pancan"
)

# Computation of cell-cell interaction scores
ccpair_scores <- compute_CC_pairs(
  lrpairs = lrpair_weights,
  cancer_type = "pancan"
)

# Investigate possible biomarkers
output_biomarkers <- explore_biomarkers(
  pathways = pathway_activity,
  immunecells = cell_fractions,
  lrpairs = lrpair_weights,
  tfs = tf_activity,
  ccpairs = ccpair_scores,
  cancer_type = cancer_type,
  patient_label = patient_ICBresponse
)
}
}
