% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compute_LR_pairs.R
\name{compute_LR_pairs}
\alias{compute_LR_pairs}
\title{Compute ligand-receptor pair weights from TPM bulk gene expression}
\usage{
compute_LR_pairs(RNA_tpm = NULL, cancer_type = "pancan", verbose = TRUE)
}
\arguments{
\item{RNA_tpm}{A data.frame containing TPM values with HGNC symbols
in rows and samples in columns.}

\item{cancer_type}{A string detailing the cancer type whose
ligand-receptor pairs network will be used.
A pan-cancer network is selected by default, whose network represents
the union of all ligand-receptor pairs present across the 18 cancer
types studied in Lapuente-Santana et al., Patterns, 2021.}

\item{verbose}{A logical value indicating whether to display messages
about the number of ligand-receptor genes found in the gene expression
data provided.}
}
\value{
A matrix of weights with samples in rows and ligand-receptor
pairs in columns.
}
\description{
Quantifies ligand-receptor interactions in the tumor microenvironment
from TPM bulk gene expression (Lapuente-Santana et al., Patterns, 2021)
by using prior knowledge coming from ligand-receptor pair annotations
from the database of Ramilowski (Ramilowski et al., Nat Commun, 2015).
Each ligand-receptor weight is defined as the minimum of the log2(TPM+1)
expression of the ligand and the receptor.
}
\examples{
# using a SummarizedExperiment object
library(SummarizedExperiment)
# Using example exemplary dataset (Mariathasan et al., Nature, 2018)
# from easierData. Original processed data is available from
# IMvigor210CoreBiologies package.
library("easierData")

dataset_mariathasan <- easierData::get_Mariathasan2018_PDL1_treatment()
RNA_tpm <- assays(dataset_mariathasan)[["tpm"]]

# Select a subset of patients to reduce vignette building time.
pat_subset <- c(
  "SAM76a431ba6ce1", "SAMd3bd67996035", "SAMd3601288319e",
  "SAMba1a34b5a060", "SAM18a4dabbc557"
)
RNA_tpm <- RNA_tpm[, colnames(RNA_tpm) \%in\% pat_subset]

# Computation of ligand-receptor pair weights
lrpair_weights <- compute_LR_pairs(
  RNA_tpm = RNA_tpm,
  cancer_type = "pancan"
)
lrpair_weights[1:5, 1:5]
}
\references{
Oscar Lapuente-Santana, Maisa van Genderen,
Peter A. J. Hilbers, Francesca Finotello, and Federica Eduati.
2021. Interpretable Systems Biomarkers Predict Response to
Immune-Checkpoint Inhibitors. Patterns, 100293.
https://doi.org/10.1016/j.patter.2021.100293.

Ramilowski, J., Goldberg, T., Harshbarger, J. et al.
A draft network of ligand–receptor-mediated multicellular signalling
in human. Nat Commun 6, 7866 (2015). https://doi.org/10.1038/ncomms8866
}
