#' Perspective projection to obtain simplex
#'
#' This function reduces data dimension by loading matrix and then project
#' dimension-reduced data to the hyperplane orthogonal to c(1,0,...,0),
#' i.e., the first axis in the new coordinate system..
#' @param data A data set that will be internally coerced into a matrix.
#'     Each row is a gene and each column is a sample. Missing values
#'     are not supported.
#'     All-zero rows will be removed internally.
#' @param W The matrix whose rows are loading vectors;
#'     should be obtained from \code{\link{CAM}}/\code{\link{CAMPrep}} function
#'     with accessor method \code{\link{PCAmat}}.
#' @details This function can project gene expression vectors to simplex plot
#' generated by \code{\link{CAM}}/\code{\link{CAMPrep}}. Using slot \code{Xproj}
#' in "\code{\link{CAMPrepObj}}" can only show the simplex of genes after
#' filtering. This function helps observe all genes in simplex plot.
#' @return  The data after perspective projection.
#' @export
#' @examples
#' #obtain data
#' data(ratMix3)
#' data <- ratMix3$X
#'
#' #preprocess data
#' rPrep <- CAMPrep(data, dim.rdc = 3, thres.low = 0.50, thres.high = 0.90)
#'
#' #obtain simplex
#' Xproj <- XWProj(data, PCAmat(rPrep))
#' #plot simplex in 3d space
#' #plot3d(Xproj[,-1]) #The first dimension is constant after projection
XWProj <- function(data, W){
    if (is(data, "data.frame")) {
        data <- as.matrix(data)
    } else if (is(data, "SummarizedExperiment")) {
        data <- SummarizedExperiment::assay(data)
    } else if (is(data, "ExpressionSet")) {
        data <- Biobase::exprs(data)
    } else if (is(data, "matrix") == FALSE) {
        stop("Only matrix, data frame, SummarizedExperiment and ExpressionSet
            object are supported for expression data!")
    }

    data <- data[rowSums(data) > 0,]

    Xp <- data %*% t(W)

    L <- nrow(W)
    Xcenter <- c(1, rep(0, L-1))
    Xp / c(Xp %*% Xcenter)
}
