% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/load_data.R
\name{load_data_peaks}
\alias{load_data_peaks}
\title{Load genome-wide binding data and associated peak files or GRanges objects}
\usage{
load_data_peaks(
  binding_profiles_path = NULL,
  peaks_path = NULL,
  binding_profiles = NULL,
  peaks = NULL,
  drop_samples = NULL,
  maxgap_loci = 1000,
  quantile_norm = FALSE,
  organism = "drosophila melanogaster",
  ensdb_genes = NULL,
  BPPARAM = BiocParallel::bpparam(),
  plot_diagnostics = interactive()
)
}
\arguments{
\item{binding_profiles_path}{Character vector. Path(s) to directories or file
globs containing log2 ratio binding tracks in bedGraph format. Wildcards ('*') supported.}

\item{peaks_path}{Character vector. Path(s) to directories or file globs containing
the peak calls in GFF or BED format.}

\item{binding_profiles}{List of GRanges objects with binding profiles, one
per sample.}

\item{peaks}{List of GRanges objects representing peak regions.}

\item{drop_samples}{A character vector of sample names or patterns to remove.
Matching samples are removed from the analysis before normalisation and
occupancy calculation. This can be useful for excluding samples that fail
initial quality checks. Default: `NULL` (no samples are dropped).}

\item{maxgap_loci}{Integer, the maximum bp distance between a peak boundary
and a gene to associate that peak with the gene. Default: 1000.}

\item{quantile_norm}{Logical (default: FALSE). If TRUE, quantile-normalise
the signal columns across all datasets.}

\item{organism}{Organism string (lower case) to obtain genome annotation from
(if not providing a custom `ensdb_genes` object)
Default: "drosophila melanogaster".}

\item{ensdb_genes}{GRanges object: gene annotation. Automatically obtained
from `organism` if NULL.}

\item{BPPARAM}{BiocParallel function (defaults to BiocParallel::bpparam())}

\item{plot_diagnostics}{Logical. If `TRUE` (the default in interactive sessions),
diagnostic plots (PCA and correlation heatmap) will be generated and
displayed for both the raw binding data and the summarised occupancy data.}
}
\value{
A list with components:
  \item{binding_profiles_data}{data.frame: Signal matrix for all regions, with columns chr, start, end, sample columns.}
  \item{peaks}{list(GRanges): All loaded peak regions from input files or directly supplied.}
  \item{pr}{GRanges: Reduced (union) peak regions across samples.}
  \item{occupancy}{data.frame: Binding values summarised over reduced peaks, with overlap annotations.}
  \item{test_category}{Character scalar; will be "bound".}
}
\description{
Reads DamID-seq log2 ratio binding data either from bedGraph files or
directly from a list of GRanges objects, and associated peak regions either
from GFF/bed files or from a list of GRanges objects.
This function is suitable for transcription factor binding analyses.
For peak discovery, use an external peak caller (e.g. 'find_peaks').
}
\details{
One of `binding_profiles_path` or `binding_profiles` must be provided.
Similarly, one of `peaks_path` or `peaks` must be provided.

When supplying GRanges lists, each GRanges should contain exactly one numeric
metadata column representing the binding signal, and all GRanges should be supplied
as a named list, with element names used as sample names.
}
\examples{
# Create a mock GRanges object for gene annotation
# This object, based on the package's unit tests, avoids network access
# and includes a very long gene to ensure overlaps with sample data.
mock_genes_gr <- GenomicRanges::GRanges(
    seqnames = S4Vectors::Rle("2L", 7),
    ranges = IRanges::IRanges(
        start = c(1000, 2000, 3000, 5000, 6000, 7000, 8000),
        end = c(1500, 2500, 3500, 5500, 6500, 7500, 20000000)
    ),
    strand = S4Vectors::Rle(GenomicRanges::strand(c("+", "-", "+", "+", "-", "-", "+"))),
    gene_id = c("FBgn001", "FBgn002", "FBgn003", "FBgn004", "FBgn005", "FBgn006", "FBgn007"),
    gene_name = c("geneA", "geneB", "geneC", "geneD", "geneE", "geneF", "LargeTestGene")
)

# Get path to sample data files included with the package
data_dir <- system.file("extdata", package = "damidBind")

# Run loading function using sample files and mock gene annotations
loaded_data <- load_data_peaks(
    binding_profiles_path = data_dir,
    peaks_path = data_dir,
    ensdb_genes = mock_genes_gr,
    quantile_norm = TRUE
)

# View the structure of the output
str(loaded_data, max.level = 1)

}
