#' Reformat TSA data into TSA Tms
#'
#' This function is used to output calculated Tm data from TSA analysis. The
#'     input data frame will be transformed into
#'     a new format that is helpful for user reading and automated analysis.
#'     The Tm values can be listed as a data frame of individual wells or the
#'     Tms from identical conditions can be averaged. When condition_average
#'     is TRUE (the default), samples with identical condition IDs will be
#'     aggregated and the average / standard deviation will be calculated
#'     where appropriate. To analyze multiple TSA experiments, use merge_TSA()
#'     to make a single data frame for analysis.
#'
#' @importFrom  dplyr distinct bind_rows
#'
#' @param analysis_data a data frame that is unmerged and generated by
#'     TSAR::read_analysis() or a merged TSA data frame generated by
#'     TSAR::merge_TSA(). Data frames require a column named 'condition_ID' for
#'     averaging.
#' @param condition_average logical value; \code{n = TRUE} by default.
#'     When TRUE, the average Tm is calculated by matched condition IDs within
#'     the data frame. When FALSE, each well is reported as a unique value with
#'     the corresponding Tm.
#' @return A data frame of Tm values.
#' @family TSAR Formatting
#' @seealso \code{\link{merge_TSA}}, \code{\link{read_raw_data}},
#'     and \code{\link{read_analysis}} for preparing input.
#' @examples
#' data("example_tsar_data")
#' TSA_Tms(example_tsar_data)
#' @export
#'

TSA_Tms <- function(
    analysis_data,
    condition_average = TRUE) {
    if (condition_average) { #--- Code to average by condition
        n_conditions <- condition_IDs(analysis_data, n = TRUE)
        tm_df_list <- as.data.frame(matrix(ncol = 0, nrow = 1))
        condition_data_list <- lapply(1:n_conditions, function(i) {
            condition_i <- condition_IDs(analysis_data)[i]
            analysis_data_i <- analysis_data[analysis_data$condition_ID == condition_i, ]
            tm_df_i <- distinct(analysis_data_i, well_ID, Tm)
            n_tms_i <- nrow(tm_df_i)
            avg_tm_i <- mean(tm_df_i$Tm, na.rm = TRUE)
            sd_tm_i <- sd(tm_df_i$Tm, na.rm = TRUE)

            tm_df_list[i, seq_len((n_tms_i + 3))] <-
                c(condition_i, avg_tm_i, sd_tm_i, tm_df_i$Tm)
            tm_df_list[i,]
        })
        tm_df <- bind_rows(condition_data_list)


        #--- Format col names and data type
        names(tm_df) <- c(
            "condition_ID", "Avg_Tm", "SD_Tm",
            paste("Tm_", (4:ncol(tm_df) - 3))
        )
        tm_df[, seq.int(2, ncol(tm_df))] <-
            lapply(tm_df[, seq.int(2, ncol(tm_df))], as.numeric)
        #--- Matching values to get Protein and Ligand into table
        tm_df$Protein <-
            analysis_data$Protein[match(
                tm_df$condition_ID,
                analysis_data$condition_ID
            )]
        tm_df$Ligand <-
            analysis_data$Ligand[match(
                tm_df$condition_ID,
                analysis_data$condition_ID
            )]
    } else { # When only reporting wells, not conditions
        tm_df <- dplyr::distinct(
            analysis_data, well_ID, Tm,
            condition_ID, Protein, Ligand
        )
    }
    return(tm_df)
}
