% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cameraPR.matrix.R
\name{cameraPR.matrix}
\alias{cameraPR.matrix}
\title{Competitive Gene Set Test Accounting for Inter-gene Correlation}
\usage{
\method{cameraPR}{matrix}(
  statistic,
  index,
  use.ranks = FALSE,
  inter.gene.cor = 0.01,
  sort = TRUE,
  alternative = c("two.sided", "less", "greater"),
  adjust.globally = FALSE,
  min.size = 2L,
  ...
)
}
\arguments{
\item{statistic}{a matrix of statistics (moderated z-statistics preferred)
with genes/molecules as row names and one or more contrasts or coefficients
as column names. Missing values are allowed.}

\item{index}{a named list of sets to test. Passed to
\code{\link{sparseIncidence}}. \code{index} must be a list of character
vectors, not the result of \code{\link[limma]{ids2indices}}, so it is more
restrictive than what \code{\link[limma]{cameraPR.default}} allows.}

\item{use.ranks}{logical; whether to perform a parametric test (\code{FALSE};
default) or a rank-based test (\code{TRUE}).}

\item{inter.gene.cor}{numeric; the inter-gene correlation within tested sets.
May be a single value or a named vector with names matching those of
\code{index}.}

\item{sort}{logical; should the results of each contrast be sorted by
p-value? Default is \code{TRUE}.}

\item{alternative}{character; the alternative hypothesis. Must be one of
"\code{two.sided}" (default), "\code{greater}", or "\code{less}". May be
abbreviated.}

\item{adjust.globally}{logical; whether p-values from different contrasts
should be adjusted together. It is recommended to set this to \code{TRUE}
when testing a set of closely related contrasts. See Section 13.3 of the
LIMMA User's Guide (\code{\link[limma]{limmaUsersGuide}}) for details.
Default is \code{FALSE}.}

\item{min.size}{integer; the minimum set size. To be considered for testing,
sets must have at least \code{min.size} elements with non-missing values in
all contrasts. The default value of 2 is the minimum possible set size
required for testing, though a value of 10 or higher is recommended; higher
values tend to produce more robust results.}

\item{...}{other arguments are not currently used}
}
\value{
A \code{data.frame} with the following columns:

\item{Contrast}{factor; the contrast of interest.}
\item{GeneSet}{character; the gene set being tested.}
\item{NGenes}{integer; number of genes in the set with values in the
\code{statistic} matrix for a given contrast.}
\item{Correlation}{numeric; inter-gene correlation (only included if
\code{inter.gene.cor} was not a single value).}
\item{Direction}{character; direction of change ("Up" or "Down").}
\item{TwoSampleT}{numeric; two-sample t-statistic (only included if
\code{use.ranks=FALSE}).}
\item{df}{integer; degrees of freedom (only included if
\code{use.ranks=FALSE}). Two less than the number of non-missing values in
each column of the \code{statistic} matrix.}
\item{ZScore}{numeric; the z-score equivalent of \code{TwoSampleT}.}
\item{PValue}{numeric; one- or two-sided (if
\code{alternative="two.sided"}) p-value.}
\item{FDR}{numeric; Benjamini and Hochberg FDR adjusted p-value.}
}
\description{
Pre-ranked Correlation-Adjusted MEan RAnk gene set testing
(CAMERA-PR) tests whether a set of genes is highly ranked relative to other
genes in terms of some measure of differential expression, accounting for
inter-gene correlation (Wu & Smyth, 2012). See
\code{\link[limma]{cameraPR}} for details.

While the language is gene-centric, any \emph{a priori} groups of molecules
may be tested.
}
\section{Test Assumptions}{


If \code{use.ranks=FALSE}, the parametric version of CAMERA-PR will be
used. Since this is a modification of Student's two sample t-test, it is
assumed that the statistics in each column of \code{statistic} are
approximately Normally distributed. In \code{\link[limma]{camera.default}},
the moderated t-statistics are converted to z-statistics with
\code{\link[limma]{zscoreT}} and used for the analysis.

If \code{use.ranks=TRUE}, a modified Wilcoxon-Mann-Whitney rank sum test
will be used.
}

\examples{
require(stats)

# Simulate experimental data with control and treatment groups (3 samples
# each)
group <- rep(c("control", "treatment"), each = 3)
design <- model.matrix(~ 0 + group)
contrasts <- makeContrasts(contrasts = "grouptreatment - groupcontrol",
                           levels = colnames(design))

ngenes <- 1000L
nsamples <- length(group)

set.seed(0)
y <- matrix(data = rnorm(ngenes * nsamples),
            nrow = ngenes, ncol = nsamples,
            dimnames = list(paste0("gene", seq_len(ngenes)),
                            make.unique(group)))

# First set of 20 genes are genuinely differentially expressed
# (trt1 and trt2 are lower than control)
index1 <- 1:20
y[index1, 1:3] <- y[index1, 1:3] + 1

# Second set of 20 genes are not DE
index2 <- 21:40

# Generate matrix of moderated t-statistics
fit <- lmFit(y, design)
fit.contr <- contrasts.fit(fit, contrasts = contrasts)
fit.smooth <- eBayes(fit.contr)

index <- list(set1 = rownames(y)[index1],
              set2 = rownames(y)[index2])

# Compute z-score equivalents of moderated t-statistics
statistic <- zscoreT(fit.smooth$t, fit.smooth$df.total)
head(statistic)

# Only set1 is DE
cameraPR(statistic = statistic, index = index)

# Non-parametric version
cameraPR(statistic = statistic, index = index, use.ranks = TRUE)
}
\references{
Wu, D., and Smyth, G. K. (2012). Camera: a competitive gene set
test accounting for inter-gene correlation. \emph{Nucleic Acids Research}
40, e133. doi:\href{https://doi.org/10.1093/nar/gks461
}{10.1093/nar/gks461}.

Goeman, J. J., and Bühlmann, P. (2007). Analyzing gene expression data in
terms of gene sets: methodological issues. \emph{Bioinformatics} 23,
980-987. doi:\href{https://doi.org/10.1093/bioinformatics/btm051
}{10.1093/bioinformatics/btm051}.
}
\seealso{
\code{\link[limma]{cameraPR}},
\code{\link[limma]{rankSumTestWithCorrelation}}
}
\author{
CAMERA-PR was developed by Di Wu and Gordon Smyth (2012). With
permission, Tyler Sagendorf modified their original code to create the
matrix method. If using \code{cameraPR.matrix}, please cite the original
paper, as well as the TMSig R package.
}
