% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Spectra.R
\name{query}
\alias{query}
\alias{query,Spectra-method}
\title{Query a Spectra object using MassQL}
\usage{
\S4method{query}{Spectra}(x, query = character(), ...)
}
\arguments{
\item{x}{The \code{Spectra} object to query.}

\item{query}{\code{character(1)} with the MassQL query.}

\item{...}{currently ignored.}
}
\value{
Depending on the \verb{<type of data>} part of the MassQL query.
}
\description{
The \code{query} function allows to query and subset/filter a \code{Spectra} object
using a Mass Spec Query Language
\href{https://mwang87.github.io/MassQueryLanguage_Documentation/}{MassQL}
expression.

A MassQL query is expressed in the form \code{"QUERY <type of data> WHERE <condition> AND <condition> FILTER <filter> AND <filter>"}, multiple
\emph{conditions} and \emph{filters} can be combined with logical \emph{and} operations. In
the MassQL definition, \emph{conditions} subsets the data to specific spectra
while \emph{filter} restricts the data within a spectrum. Note that at present
MassQL \emph{filters} are not supported. Also note that MassQL queries are
interpreted case insensitive in \code{SpectraQL}.

See also the package vignette for more details.
}
\section{Type of data}{


The \code{"<type of data>"} allows to define which data should be extracted from
the selected spectra. MassQL defines \emph{type of data} being \code{"MS1DATA"} or
\code{"MS2DATA"} to retrieve data from MS1 or MS2 scans. By default peak data will
be returned, but in addition, MASSQL defines additional functions that can
be applied to modify the data or select different data to be returned. In
addition \emph{SpectraQL} defines the special type of data \code{"*"} which will return
the results as a \code{Spectra} object. \emph{SpectraQL} supports:
\itemize{
\item \code{"*"}: select all data and return the data subset as a
\code{\link[Spectra:Spectra]{Spectra::Spectra()}} object.
\item \code{"MS1DATA"}: return the \code{\link[Spectra:spectraData]{Spectra::peaksData()}} from all selected \strong{MS1}
spectra, i.e. a \code{list} with two column matrices with the peaks' m/z and
intensity values.
\item \code{"MS2DATA"}: return the \code{\link[Spectra:spectraData]{Spectra::peaksData()}} from all selected \strong{MS2}
spectra, i.e. a \code{list} with two column matrices with the peaks' m/z and
intensity values.
\item \code{"scaninfo(MS1DATA)"}, \code{"scaninfo(MS2DATA)"}: return the
\code{\link[Spectra:spectraData]{Spectra::spectraData()}} of all selected spectra.
\item \code{"scansum(MS1DATA)"}, \code{"scansum(MS2DATA)"}: sum of the peak intensities of
the selected spectra (TIC, or XIC if combined with \code{"FILTER"}).
}
}

\section{Conditions}{


Conditions define to which spectra the data set should be subsetted. A
\emph{condition} will subset a \code{Spectra} object to selected spectra, but will not
(unlike \emph{Filters}, see further below) filter peaks from a spectrum. Several
conditions can be combined with \code{"and"} (case insensitive). The syntax for a
condition is \code{"<condition> = <value>"}, e.g. \code{"MS2PROD = 144.1"}. Such
conditions can be further refined by additional expressions that allow for
example to define acceptable tolerances for m/z differences. \code{SpectraQL}
supports (case insensitive):
\itemize{
\item \code{"RTMIN"}: minimum retention time (in \strong{seconds}).
\item \code{"RTMAX"}: maximum retention time (in \strong{seconds}).
\item \code{"SCANMIN"}: the minimum scan number (acquisition number).
\item \code{"SCANMAX"}: the maximum scan number (acquisition number).
\item \code{"CHARGE"}: the charge for MS2 spectra.
\item \code{"POLARITY"}: the polarity of the spectra (can be \code{"positive"},
\code{"negative"}, \code{"pos"} or \code{"neg"}, case insensitive).
\item \code{"MS2PROD"} or \code{"MS2MZ"}: allows to select MS2 spectra that contain a peak
with particular m/z value(s). See below for examples.
\item \code{"MS2PREC"}: allows to select MS2 spectra with the defined precursor m/z
value(s). See below for examples.
\item \code{"MS1MZ"}: allows to select MS1 spectra containing peak(s) with the defined
m/z value(s).
\item \code{"MS2NL"}: allows to look for a neutral loss from precursor in MS2 spectra.
}

All conditions involving m/z values allow to specify a mass accuracy using
the optional fields \code{"TOLERANCEMZ"} and \code{"TOLERANCEPPM"} that define the
absolute and m/z-relative acceptable difference in m/z values. One or both
fields can be attached to a \emph{condition} such as
\code{"MS2PREC=100:TOLERANCEMZ=0.1:TOLERANCEPPM=20"} to select for example all
MS2 spectra with a precursor m/z equal to 100 accepting a difference of 0.1
and 20 ppm. Note that in contrast to MassQL, the default tolarance and ppm
is 0 for all calls.
}

\section{Filters}{


Filters subset the data within spectra, i.e. select which peaks within
spectra should be retrieved. \emph{SpectraQL} supports the following filters:
\itemize{
\item \code{"MS1MZ"}: filters MS1 spectra keeping only peaks with matching m/z values
(tolerance can be specified with \code{"TOLERANCEMZ"} and \code{"TOLERANCEPPM"} as
for conditions).
\item \code{"MS2MZ"}: filters MS2 spectra keeping only peaks with matching m/z values
(tolerance can be specified with \code{"TOLERANCEMZ"} and \code{"TOLERANCEPPM"} as
for conditions).
}
}

\examples{

## Read a data file with MS1 and MS2 spectra
library(msdata)
library(Spectra)
fls <- dir(system.file("TripleTOF-SWATH", package = "msdata"),
    full.names = TRUE)
sps_dda <- Spectra(fls[1L])

## Subset to spectra measured between 300 and 400 seconds
query(sps_dda, "QUERY * WHERE RTMIN = 300 AND RTMAX = 400")

## To extract peaks data from MS1 or MS2 spectra use "MS1DATA" or "MS2DATA"
## instead of *. Note also that queries are case-insensitive.
pks <- query(sps_dda, "query ms1data where rtmin = 300 and rtmax = 400")
pks
head(pks[[1L]])

## To select (MS2) spectra with a certain precursor m/z the MS2PREC condition
## can be used. Below we extract all spectra with a precursor m/z of 99.9
## accepting also a difference of 10ppm
query(sps_dda, "QUERY * WHERE MS2PREC = 99.967:TOLERANCEPPM=10")

## It is also possible to specify multiple precursor m/z values:
query(sps_dda, "QUERY * WHERE MS2PREC = (99.967 OR 428.88):TOLERANCEPPM=10")

## To select all MS1 spectra that contain a peak with a certain m/z we can
## use the MS1MZ condition. Below we combine this with an absolute tolerance
## using TOLERANCEMZ.
query(sps_dda, "QUERY * WHERE MS1MZ = 100:TOLERANCEMZ=1")

## Using MS2DATA in combination with MS1MZ will not return any spectra.
query(sps_dda, "QUERY MS2DATA WHERE MS1MZ = 100:TOLERANCEMZ=1")

## In contrast, do select MS2 spectra containing a peak with a certain m/z
## we have to use the condition MS2PROD
query(sps_dda, "QUERY * WHERE MS2PROD = 100:TOLERANCEMZ=1")

## MS2MZ can be used as alternative to MS2PROD
query(sps_dda, "QUERY * WHERE MS2MZ = 100:TOLERANCEMZ=1")

## Select MS2 spectra containing a peak with neutral loss from
## precursor of 100 allowing a m/z relative ppm tolerance of 5)
res <- query(sps_dda, "QUERY MS2DATA WHERE MS2NL=100:TOLERANCEPPM=5")

## Combine two different conditions: selection of spectra with positive
## polarity and retention time greater than 200
res <- query(sps_dda, "QUERY * WHERE RTMIN = 200 AND POLARITY = Positive")
}
\author{
Andrea Vicini, Johannes Rainer
}
