% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SPsimSeq.R
\name{SPsimSeq}
\alias{SPsimSeq}
\title{A function to simulate bulk or single cell RNA sequencing data}
\usage{
SPsimSeq(
  n.sim = 1,
  s.data,
  batch = rep(1, ncol(s.data)),
  group = rep(1, ncol(s.data)),
  n.genes = 1000,
  batch.config = 1,
  group.config = 1,
  pDE = 0.1,
  cand.DE.genes = NULL,
  lfc.thrld = 0.5,
  t.thrld = 2.5,
  llStat.thrld = 5,
  tot.samples = ncol(s.data),
  model.zero.prob = FALSE,
  genewiseCor = TRUE,
  log.CPM.transform = TRUE,
  lib.size.params = NULL,
  variable.lib.size = FALSE,
  w = NULL,
  result.format = "SCE",
  return.details = FALSE,
  verbose = TRUE,
  prior.count = 1,
  const.mult = 1e+06,
  n.mean.class = 0.2,
  minFracZeroes = 0.25
)
}
\arguments{
\item{n.sim}{an integer for the number of simulations to be generated}

\item{s.data}{a real source dataset (a SingleCellExperiment class or a matrix/data.frame of counts with genes in rows and samples in columns)}

\item{batch}{NULL or a vector containing batch indicators for each sample/cell in the source data}

\item{group}{NULL or a vector containing group indicators for each sample/cell in the source data}

\item{n.genes}{a numeric value for the total number of genes to be simulated}

\item{batch.config}{a numerical vector containing fractions for the composition of samples/cells per batch. The fractions must sum to 1. The number of batches to be simulated is equal to the size of the vector. (Example, batch.config=c(0.6, 0.4)   means simulate 2 batches with 60\% of the simulated samples/cells in batch 1 and the rest 40\% in the second batch. Another example,  batch.config=c(0.3, 0.35, 0.25) means simulate 3 batches with the first, second and third batches contain 30\%, 35\% and 25\% of the samples/cells, respectively).}

\item{group.config}{a numerical vector containing fractions for the composition of samples/cells per group. Similar definition to 'batch.config'. The number of groups to be simulated is equal to the size of the vector. The fractions must sum to 1.}

\item{pDE}{a numeric value between 0 and 1 indicating the desired fraction of DE genes in the simulated data}

\item{cand.DE.genes}{a list object contatining canidiate null and non-null (DE/predictor) genes.
If NULL (the default), an internal function determines candidate genes based on log-fold-change and other statistics.
The user can also pass a list of canidate null and non-null genes (they must be disjoint). In particular, the list should contain
two character vectors (for the name of the features/genes in the source data) with names 'null.genes' and 'nonnull.genes'.
For example, cand.DE.genes=list(null.genes=c('A', 'B'), nonnull.genes=c('C', 'D')).}

\item{lfc.thrld}{a positive numeric value for the minimum absolute log-fold-change for selecting
candidate DE genes in the source data (when group is not NULL, pDE>0 and cand.DE.genes is NULL)}

\item{t.thrld}{a positive numeric value for the minimum absolute t-test statistic for the
log-fold-changes of genes for selecting candidate DE genes in the source data (when group
is not NULL, pDE>0 and cand.DE.genes is NULL)}

\item{llStat.thrld}{a positive numeric value for the minimum squared test statistics from
the log-linear model to select candidate DE genes in the source data (when group is not NULL,
pDE>0 and cand.DE.genes is NULL)}

\item{tot.samples}{a numerical value for the total number of samples/cells to be simulated.}

\item{model.zero.prob}{a logical value whether to model the zero expression probability separately (suitable for simulating of single-cell RNA-seq data or zero-inflated data)}

\item{genewiseCor}{a logical value, if TRUE (default) the simulation will retain the gene-to-gene correlation structure of the source data using Gausian-copulas . Note that if it is TRUE, the program will be slow or it may fail for a limited memory size.}

\item{log.CPM.transform}{a logical value. If TRUE, the source data will be transformed into log-(CPM+const) before estimating the probability distributions}

\item{lib.size.params}{NULL or a named numerical vector containing parameters for simulating library sizes from log-normal distribution. If lib.size.params =NULL (default), 
then the package will fit a log-normal distribution for the library sizes in the source data to simulate new library sizes. 
If the user would like to specify the parameters of the log-normal distribution for the desired library sizes, 
then the log-mean and log-SD params of rlnorm() functions can be passed using this argument. 
Example, lib.size.params = c(meanlog=10, sdlog=0.2). See also ?rlnorm.}

\item{variable.lib.size}{a logical value. If FALSE (default), then the expected library sizes are simulated once and remains the same for every replication (if n.sim>1).}

\item{w}{see ?hist}

\item{result.format}{a character value for the type of format for the output. Choice can be 'SCE' for SingleCellExperiment class or "list" for a list object that contains the simulated count, column information and row information.}

\item{return.details}{a logical value. If TRUE, detailed results including estimated parameters and densities will be returned}

\item{verbose}{a logical value, if TRUE a message about the status of the simulation will be printed on the console}

\item{prior.count}{a positive constant to be added to the CPM before log transformation, to avoid log(0). The default is 1.}

\item{const.mult}{A constant by which the count are scaled. Usually 1e6 to get CPM}

\item{n.mean.class}{a fraction of the number of genes
for the number of groups to be created for the mean log CPM of genes}

\item{minFracZeroes}{minimum fraction of zeroes before a zero inflation model
is fitted}
}
\value{
a list of SingleCellExperiment/list objects each containing simulated counts (not normalized), smple/cell level information in colData, and gene/feature level information in rowData.
}
\description{
This function simulates (bulk/single cell) RNA-seq dataset from semi-parametrically estimated distributions of gene expression levels in a given real source RNA-seq dataset
}
\details{
This function uses a specially designed exponential family for density estimation
to constructs the distribution of gene expression levels from a given real gene expression data
(e.g. single-cell or bulk sequencing data), and subsequently, simulates a new
from the estimated distributions.#'
For simulation of single-cell RNA-seq data (or any zero inflated gene expression data), the programm
involves an additional step to explicitly account for the high abundance of zero counts (if required).
This step models the probability of zero counts as a function the mean
expression of the gene and the library size of the cell (both in log scale) to add excess zeros.
This can be done by using \emph{model.zero.prob=TRUE}. Note that,
for extremly large size data, it is recomended to use a random sample of cells to
reduce computation time. To enable this, add the argument \emph{subset.data=TRUE} and you
can specify the number of cells to be used using \emph{n.samples} argument.
For example \emph{n.samples=400}.
Given known groups of samples/cells in the source data, DGE is simulated by independently
sampling data from distributions constructed for each group seprately. In particular, this procedure is
applied on a set of genes with absolute log-fold-change in the source data more than a given threshold (\emph{lfc.thrld}).
Moreover, when  the source dataset involves samples/cells processed in different batches, our
simulation procedure incorporates this batch effect in the simulated data, if required.
Different experimental designs can be simulated using the group and batch configuration arguments to
simulate biologica/experimental conditions and batchs, respectively.
Also, it is important to filter the source data so that genes with suffient expression will be used to
estimate the probability distributions.
}
\examples{
#----------------------------------------------------------------
# Example 1: simulating bulk RNA-seq

# load the Zhang bulk RNA-seq data (availabl with the package)
data("zhang.data.sub")

zhang.counts <- zhang.data.sub$counts
MYCN.status  <- zhang.data.sub$MYCN.status

# We simulate only a single data (n.sim = 1) with the following property
# - 1000 genes ( n.genes = 1000)
# - 40 samples (tot.samples = 40)
# - the samples are equally divided into 2 groups each with 90 samples
#   (group.config = c(0.5, 0.5))
# - all samples are from a single batch (batch = NULL, batch.config = 1)
# - we add 10\% DE genes (pDE = 0.1)
# - the DE genes have a log-fold-change of at least 0.5 in
#   the source data (lfc.thrld = 0.5)
# - we do not model the zeroes separately, they are the part of density
#    estimation (model.zero.prob = FALSE)

# simulate data
set.seed(6452)
sim.data.bulk <- SPsimSeq(n.sim = 1, s.data = zhang.counts,
                          group = MYCN.status, n.genes = 1000, batch.config = 1,
                          group.config = c(0.5, 0.5), tot.samples = 40,
                          pDE = 0.1, lfc.thrld = 0.5, result.format = "list")

head(sim.data.bulk$counts[[1]][, seq_len(5)])  # count data
head(sim.data.bulk$colData)        # sample info
head(sim.data.bulk$rowData)        # gene info

#----------------------------------------------------------------
# Example 2: simulating single cell RNA-seq from a single batch (read-counts)
# we simulate only a single scRNA-seq data (n.sim = 1) with the following property
# - 2000 genes (n.genes = 2000)
# - 100 cells (tot.samples = 100)
# - the cells are equally divided into 2 groups each with 50 cells
#   (group.config = c(0.5, 0.5))
# - all cells are from a single batch (batch = NULL, batch.config = 1)
# - we add 10\% DE genes (pDE = 0.1)
# - the DE genes have a log-fold-change of at least 0.5
# - we model the zeroes separately (model.zero.prob = TRUE)
# - the ouput will be in SingleCellExperiment class object (result.format = "SCE")

library(SingleCellExperiment)

# load the NGP nutlin data (availabl with the package, processed with
# SMARTer/C1 protocol, and contains read-counts)
data("scNGP.data")

# filter genes with sufficient expression (important step to avoid bugs)
treatment <- ifelse(scNGP.data$characteristics..treatment=="nutlin",2,1)

set.seed(654321)

# simulate data (we simulate here only a single data, n.sim = 1)
sim.data.sc <- SPsimSeq(n.sim = 1, s.data = scNGP.data, group = treatment,
 n.genes = 2000, batch.config = 1, group.config = c(0.5, 0.5), 
 tot.samples = 100, pDE = 0.1, lfc.thrld = 0.5, model.zero.prob = TRUE,
                    result.format = "SCE")

sim.data.sc1 <- sim.data.sc[[1]]
class(sim.data.sc1)
head(counts(sim.data.sc1)[, seq_len(5)])
colData(sim.data.sc1)
rowData(sim.data.sc1)

}
\references{
\itemize{
\item Assefa, A. T., Vandesompele, J., & Thas, O. (2020). SPsimSeq: semi-parametric simulation of bulk and single cell RNA sequencing data. \emph{Bioinformatics}, doi: https://doi.org/10.1093/bioinformatics/btaa105.
\item Efron, B., & Tibshirani, R. (1996). Using specially designed exponential families for density estimation. \emph{The Annals of Statistics}, 24(6), 2431-2461.
}
}
