% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spicey_plot.R
\name{spicey_heatmap}
\alias{spicey_heatmap}
\title{SPICEY heatmap for gene specificity across cell types}
\usage{
spicey_heatmap(
  df,
  top_n = 5,
  spicey_measure = c("RETSI", "GETSI", "SPICEY"),
  combined_score = FALSE
)
}
\arguments{
\item{df}{A data frame with at least the following columns:
\describe{
\item{gene_id}{Identifier of the gene. Must be an official gene symbol (e.g., \code{GAPDH}).
If you only have ATAC data, link to nearest gene (\code{\link{annotate_with_nearest}})
or using coaccessibility (\code{\link{annotate_with_coaccessibility}}).}
\item{\code{cell_type}}{Cell type or cluster label (e.g., \code{Acinar})}
\item{\code{RETSI}}{Numeric. RETSI specificity scores (optional unless used).}
\item{\code{GETSI}}{Numeric. GETSI specificity scores (optional unless used).}}}

\item{top_n}{Integer. Number of top-ranked genes to include per cell type (default \code{"5"})}

\item{spicey_measure}{Character. Score type to visualize. Must be one of the following:
\describe{
\item{\code{"RETSI"}}{Only RETSI will be plotted.}
\item{\code{"GETSI"}}{Only GETSI will be plotted.}
\item{\code{"SPICEY"}}{Both RETSI and GETSI are used (requires both columns)}}}

\item{combined_score}{Logical. Only relevant if \code{spicey_measure = "SPICEY"}.
If \code{TRUE}, a single heatmap of mean RETSI/GETSI score is generated.
If \code{FALSE}, two heatmaps are produced side by side (RETSI and GETSI).}
}
\value{
A \code{ggplot2} object, or a \code{patchwork} layout if two heatmaps are returned.
}
\description{
Visualizes gene-level specificity scores (RETSI and/or GETSI) across cell types
using a SPICEY scored heatmap representation. Depending on the chosen mode, the function
can display either RETSI or GETSI scores independently, or compute and visualize
a combined SPICEY score (mean score of RETSI and GETSI).
If \code{spicey_measure = "SPICEY"} and \code{combined_score = TRUE}, RETSI and GETSI
scores are scaled, averaged, and shown in a unified heatmap. Otherwise, separate
heatmaps are produced for RETSI and GETSI, respectively.
}
\examples{
library(TxDb.Hsapiens.UCSC.hg38.knownGene)
library(org.Hs.eg.db)

data(rna)
data(atac)
data(cicero_links)

# Obtain annotatin with coaccessibility
peaks <- unique(unlist(atac)[, c("region_id")])
annotation_coacc <- annotate_with_coaccessibility(
  peaks = peaks,
  txdb = TxDb.Hsapiens.UCSC.hg38.knownGene,
  links_df = cicero_links,
  annot_dbi = org.Hs.eg.db,
  protein_coding_only = TRUE,
  verbose = TRUE,
  add_tss_annotation = FALSE,
  upstream = 2000,
  downstream = 2000
)

# Obtain linked SPICEY measures
spicey_coacc <- SPICEY(
  rna = rna,
  atac = atac,
  annotation = annotation_coacc
)

# Make plots
retsi <- spicey_coacc$RETSI |> dplyr::left_join(annotation_coacc, by = c("region_id"))
spicey_heatmap(retsi, spicey_measure = "RETSI")

spicey_heatmap(spicey_coacc$GETSI, spicey_measure = "GETSI")

spicey_heatmap(spicey_coacc$linked, spicey_measure = "SPICEY", combined_score = FALSE)

spicey_heatmap(spicey_coacc$linked, spicey_measure = "SPICEY", combined_score = TRUE)
}
\seealso{
\code{\link{SPICEY}}, \code{\link{prepare_heatmap_data}}, \code{\link{plot_heatmap}}
}
