% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/synthetic.R
\encoding{UTF-8}
\name{prepSynthetic}
\alias{prepSynthetic}
\title{Add information related to the synthetic profiles
(study and synthetic reference profiles information) into a Profile GDS file}
\usage{
prepSynthetic(
  fileProfileGDS,
  listSampleRef,
  profileID,
  studyDF,
  nbSim = 1L,
  prefix = "",
  verbose = FALSE
)
}
\arguments{
\item{fileProfileGDS}{a \code{character} string representing the file name
of the Profile GDS file containing the information about the reference
profiles used to generate the synthetic profiles.}

\item{listSampleRef}{a \code{vector} of \code{character} string
representing the
identifiers of the selected 1KG profiles that will be used as reference to
generate the synthetic profiles.}

\item{profileID}{a \code{character} string representing the profile
identifier present in the \code{fileProfileGDS} that will be used to
generate synthetic profiles.}

\item{studyDF}{a \code{data.frame} containing the information about the
study associated to the analysed sample(s). The \code{data.frame} must have
those 2 columns: "study.id" and "study.desc". Those 2 columns
must be in \code{character} strings (no factor). Other columns can be
present, such as "study.platform", but won't be used.}

\item{nbSim}{a single positive \code{integer} representing the number of
simulations per combination of sample and 1KG reference. Default: \code{1L}.}

\item{prefix}{a single \code{character} string representing the prefix that
is going to be added to the name of the synthetic profile. The prefix
enables the creation of multiple synthetic profile using the same
combination of sample and 1KG reference. Default: \code{""}.}

\item{verbose}{a \code{logical} indicating if messages should be printed
to show how the different steps in the function. Default: \code{FALSE}.}
}
\value{
\code{0L} when successful.
}
\description{
This function add entries related to synthetic profiles into
a Profile GDS file. The entries are related to two types of information:
the synthetic study and the synthetic profiles.

The study information is appended to the Profile GDS file "study.list" node.
The "study.platform" entry is always set to 'Synthetic'.

The profile information, for all selected synthetic profiles, is appended to
the Profile GDS file "study.annot" node. Both the "Source" and the
"Sample.Type" entries are always set to 'Synthetic'.

The synthetic profiles are assigned unique names by combining:
\code{prefix}.\code{data.id.profile}.\code{listSampleRef}.\code{simulation
number(1 to nbSim)}
}
\examples{

## Required library
library(gdsfmt)

## Path to the demo 1KG GDS file is located in this package
dataDir <- system.file("extdata/tests", package="RAIDS")

## Temporary Profile GDS file
fileNameGDS <- file.path(tempdir(), "ex1.gds")

## Copy the Profile GDS file demo that has been pruned and annotated
file.copy(file.path(dataDir, "ex1_demo_with_pruning_and_1KG_annot.gds"),
                 fileNameGDS)

## Information about the synthetic data set
syntheticStudyDF <- data.frame(study.id="MYDATA.Synthetic",
        study.desc="MYDATA synthetic data", study.platform="PLATFORM",
        stringsAsFactors=FALSE)

## Add information related to the synthetic profiles into the Profile GDS
prepSynthetic(fileProfileGDS=fileNameGDS,
        listSampleRef=c("HG00243", "HG00150"), profileID="ex1",
        studyDF=syntheticStudyDF, nbSim=1L, prefix="synthetic",
        verbose=FALSE)

## Open Profile GDS file
profileGDS <- openfn.gds(fileNameGDS)

## The synthetic profiles should be added in the 'study.annot' entry
tail(read.gdsn(index.gdsn(profileGDS, "study.annot")))

## The synthetic study information should be added to
## the 'study.list' entry
tail(read.gdsn(index.gdsn(profileGDS, "study.list")))

## Close GDS file (important)
closefn.gds(profileGDS)

## Remove Profile GDS file (created for demo purpose)
unlink(fileNameGDS, force=TRUE)


}
\author{
Pascal Belleau, Astrid Deschênes and Alexander Krasnitz
}
