#' @title Generate accuracy graph
#'
#' @description This function extracts the required information from an
#' output generated by RAIDS to create a graphic representation of the
#' accuracy for different values of PCA dimensions and K-neighbors through
#' all tested ancestries.
#'
#' @param fileRDS a \code{character} string representing the path and file
#' name of the RDS file containing the ancestry information as generated by
#' RAIDS.
#'
#' @param title a \code{character} string representing the title of the graph.
#' Default: \code{""}.
#'
#' @param selectD a \code{array} of \code{integer} representing the selected
#' PCA dimensions to plot. The length of the \code{array} cannot be more than
#' 5 entries. The dimensions must tested by RAIDS (i.e. be present in the
#' RDS file). Default: \code{c(3,7,11)}.
#'
#' @param selectColor a \code{array} of \code{character} strings representing
#' the selected colors for the associated PCA dimensions to plot. The length
#' of the \code{array} must correspond to the length of the \code{selectD}
#' parameter. In addition, the length of the \code{array} cannot be more than
#' 5 entries.
#' Default: \code{c("#5e688a", "#cd5700", "#CC79A7")}.
#'
#' @return a \code{ggplot} object containing the graphic representation of the
#' accuracy for different values of PCA dimensions and K-neighbors through
#' all tested ancestries.
#'
#' @examples
#'
#' ## Required library
#' library(ggplot2)
#'
#' ## Path to RDS file with ancestry information generated by RAIDS (demo file)
#' dataDir <- system.file("extdata", package="RAIDS")
#' fileRDS <- file.path(dataDir, "TEST_01.infoCall.RDS")
#'
#' ## Create accuracy graph
#' accuracyGraph <- createAccuracyGraph(fileRDS=fileRDS, title="Test 01",
#'     selectD=c(3,6,9,12,15),
#'     selectColor=c("steelblue", "darkorange", "violet", "pink", "gray80"))
#'
#' accuracyGraph
#'
#' @author Astrid Deschênes and Pascal Belleau
#' @importFrom ggplot2 ggplot geom_ribbon geom_line theme_classic ylim
#' @importFrom ggplot2 ylab facet_grid theme scale_colour_manual ggtitle
#' @importFrom ggplot2 element_text element_rect aes
#' @importFrom rlang .data
#' @encoding UTF-8
#' @export
createAccuracyGraph <- function(fileRDS, title="",
    selectD=c(3,7,11), selectColor=c("#5e688a", "#cd5700", "#CC79A7")) {

    ## Validate parameters
    validateCreateAccuracyGraph(fileRDS=fileRDS, title=title, selectD=selectD,
                                    selectColor=selectColor)

    ## Extract required information from RDS file
    info <- readRDS(fileRDS)
    dfAUROC <- info$paraSample$dfAUROC

    if (!all(selectD %in% unique(dfAUROC$pcaD))) {
        stop("Not all values in \'selectD\' are present in the RDS file.")
    }

    ## Retained selected dimensions
    dfAUROC <- dfAUROC[which(dfAUROC$pcaD %in% selectD), ]
    dfAUROC$pcaD <- as.factor(dfAUROC$pcaD)

    colnames(dfAUROC)[colnames(dfAUROC) == "pcaD"] <- "D"

    ## Set y axis minimum value
    ymin <- min(c(dfAUROC$L)) - 0.008

    ## Generate graph
    accuracy <- ggplot(dfAUROC, aes(x=.data$K, y=.data$AUC, group=.data$D,
                                color=.data$D, linetype=.data$D)) +
        geom_ribbon(aes(ymin=.data$L, ymax=.data$H, group=.data$D),
                linetype="dotted", linewidth=2, alpha=0.1) +
        geom_line(linewidth=2) + facet_grid(. ~ Call) +
        ylim(c(ymin, 1)) + ggtitle(title) + ylab(label = "AUROC") +
        scale_colour_manual(aesthetics = c("colour", "fill"),
                breaks=selectD, values=selectColor) +
        theme_classic() + 
        theme(axis.text=element_text(size=20, colour = "black"),
            panel.background = element_rect(color="black"),
            axis.text.x=element_text(size=20, angle=90,
                    vjust = 0.5, hjust=1, colour="black"),
            plot.title = element_text(size=22, face="bold",
                    colour="gray20", hjust=0.5),
            axis.title=element_text(size=30,face="bold.italic"),
            strip.text  = element_text(size=20, face="bold"),
            strip.background = element_rect(fill="gray90"),
            legend.text=element_text(size=19),
            legend.title=element_text(size=22, face="bold.italic"))

    ## Successful
    return(accuracy)
}

#' @title Generate accuracy graph
#'
#' @description This function extracts the required information from an
#' output generated by RAIDS to create a graphic representation of the
#' accuracy for different values of PCA dimensions and K-neighbors through
#' all tested ancestries.
#'
#' @param dfAUROC a \code{data.frame} corresponding to res$paraSample$dfAUROC
#' where res is the result of inferAncestry() or inferAncestryGeneAware() 
#' functions.
#'
#' @param title a \code{character} string representing the title of the graph.
#' Default: \code{""}.
#'
#' @param selectD a \code{array} of \code{integer} representing the selected
#' PCA dimensions to plot. The length of the \code{array} cannot be more than
#' 5 entries. The dimensions must tested by RAIDS (i.e. be present in the
#' RDS file). Default: \code{c(3,7,11)}.
#'
#' @param selectColor a \code{array} of \code{character} strings representing
#' the selected colors for the associated PCA dimensions to plot. The length
#' of the \code{array} must correspond to the length of the \code{selectD}
#' parameter. In addition, the length of the \code{array} cannot be more than
#' 5 entries.
#' Default: \code{c("#5e688a", "#cd5700", "#CC79A7")}.
#'
#' @return a \code{ggplot} object containing the graphic representation of the
#' accuracy for different values of PCA dimensions and K-neighbors through
#' all tested ancestries.
#'
#' @examples
#'
#' ## Required library
#' library(ggplot2)
#'
#' ## Path to RDS file with ancestry information generated by RAIDS (demo file)
#' dataDir <- system.file("extdata", package="RAIDS")
#' fileRDS <- file.path(dataDir, "TEST_01.infoCall.RDS")
#' info <- readRDS(fileRDS)
#' dfAUROC <- info$paraSample$dfAUROC
#' 
#' ## Some of the column names must be updated to fit new standards
#' colnames(dfAUROC) <- c("D", "K", "Call", "L", "AUROC", "H")
#' 
#' ## Create accuracy graph
#' accuracyGraph <- createAUROCGraph(dfAUROC=dfAUROC, title="Test 01",
#'     selectD=c(3, 6, 9, 12, 15),
#'     selectColor=c("steelblue", "darkorange", "violet", "pink", "gray40"))
#'
#' accuracyGraph
#'
#' @author Astrid Deschênes and Pascal Belleau
#' @importFrom ggplot2 ggplot geom_ribbon geom_line theme_classic ylim
#' @importFrom ggplot2 ylab facet_grid theme scale_colour_manual ggtitle
#' @importFrom ggplot2 element_text element_rect aes
#' @importFrom rlang .data
#' @encoding UTF-8
#' @export
createAUROCGraph <- function(dfAUROC, title="",
        selectD=c(3,7,11), selectColor=c("#5e688a", "#cd5700", "#CC79A7")) {

    ## Validate parameters
    validatecreateAUROCGraph(dfAUROC=dfAUROC, title=title, selectD=selectD,
                                    selectColor=selectColor)

    if (!all(selectD %in% unique(dfAUROC$D))) {
        stop("Not all values in \'selectD\' are present in the \'dfAUROC\' ", 
                "data frame.")
    }

    ## Retained selected dimensions
    dfAUROC <- dfAUROC[which(dfAUROC$D %in% selectD), ]
    dfAUROC$D <- as.factor(dfAUROC$D)

    ## Set y axis minimum value
    ymin <- min(c(dfAUROC$L)) - 0.008

    ## Generate graph
    accuracy <- ggplot(dfAUROC, aes(x=.data$K, y=.data$AUROC, group=.data$D,
                                    color=.data$D, linetype=.data$D)) +
        geom_ribbon(aes(ymin=.data$L, ymax=.data$H, group=.data$D),
                    linetype="dotted", linewidth=2, alpha=0.1) +
        geom_line(linewidth=2) + facet_grid(. ~ Call) +
        scale_colour_manual(aesthetics = c("colour", "fill"),
                            breaks=selectD, values=selectColor) +
        ylim(c(ymin, 1)) + ggtitle(title) + ylab(label = "AUROC") + 
        theme_classic() +
        theme(axis.text=element_text(size=20, colour="black"),
            panel.background = element_rect(color="black"),
            axis.text.x=element_text(size=20, angle=90,
                                       vjust = 0.5, hjust=1, colour="black"),
            plot.title=element_text(size=22, face="bold",
                                        colour="gray20", hjust=0.5),
            axis.title=element_text(size=30, face="bold.italic"),
            strip.text=element_text(size=20, face="bold"),
            strip.background = element_rect(fill="gray90"),
            legend.text=element_text(size=19),
            legend.title=element_text(size=22, face="bold.italic"))

    ## Successful
    return(accuracy)
}

