## ------------------
## Utility functions
## ------------------

##' @title Count Unique Features
##'
##' @description
##'
##' This function counts the number of unique features per sample. A
##' grouping structure can be provided to count higher level features
##' from assays, for example counting the number of unique proteins
##' from PSM data.
##'
##' @param object An object of class `QFeatures`.
##'
##' @param i  A `numeric()` or `character()` vector indicating from
##'     which assays the `rowData` should be taken.
##'
##' @param groupBy A `character(1)` indicating the variable name in
##'     the `rowData` that contains the grouping variable, for
##'     instance to count the unique number of peptides or proteins
##'     expressed in each samples (column). If `groupBy` is missing,
##'     the number of non zero elements per sample will be stored.
##'
##' @param colDataName A `character(1)` giving the name of the new
##'     variable in the `colData` where the number of unique features
##'     will be stored. The name cannot already exist in the
##'     `colData`.
##'
##' @export
##'
##' @return An object of class `QFeatures`.
##'
##' @examples
##' data("ft_na")
##' ## Count number of (non-missing) PSMs
##' ft_na <- countUniqueFeatures(ft_na,
##'                              i = "na",
##'                              colDataName = "counts")
##' ft_na$counts
##' ## Count number of unique rowData feature
##' ft_na <- countUniqueFeatures(ft_na,
##'                              i = "na",
##'                              groupBy = "Y",
##'                              colDataName = "Y_counts")
##' ft_na$Y_counts
##'
countUniqueFeatures <- function(object,
                                i,
                                groupBy = NULL,
                                colDataName = "count") {
    ## Check the colData does not already contain the name
    if (colDataName %in% colnames(colData(object)))
        stop("'", colDataName, "' is already present in the colData.")

    snames <- unlist(colnames(object)[i])
    ## Avoid that a sample is contained in 2 different assays
    if (anyDuplicated(snames))
        stop("The same sample is present in multiple assays.")

    ## Initialize the vector containing the feature counts
    fcounts <- vector(length = length(snames), mode = "integer")
    names(fcounts) <- snames

    if (is.null(groupBy)) {
        ## If no  grouping is supplied, count the non-missing
        ## features per sample
        for (ii in i) {
            fcount <- apply(assay(object[[ii]]), 2, function(x) {
                sum(!is.na(x))
            })
            fcounts[names(fcount)] <- fcount
        }
    } else {
        ## Count the number of unique entries of groupBy
        for (ii in i) {
            fcount <- apply(assay(object[[ii]]), 2, function(x) {
                length(unique(rowData(object[[ii]])[!is.na(x), groupBy]))
            })
            fcounts[names(fcount)] <- fcount
        }
    }

    ## Store the counts in the colData
    colData(object)[names(fcounts), colDataName] <- fcounts
    object
}

##' @title Create precursor identfiers
##'
##' @description
##'
##' The [createPrecursorId()] is used to create new precursor identifier columns
##' in a `QFeatures` object's assays (more precisely in their rowData). The new
##' variable is called by default `"Precursor.Id"`, and is generated by the
##' concatenation of other rowData variables that, together, should create
##' unique identifiers.
##'
##' These precursor identifiers, assuming they are unique, can then be used to
##' join assays using [joinAssays()], rather than using the rownames, as
##' illustrated below.
##'
##' @param object An instance of class `QFeatures`.
##'
##' @param name `character(1)` with the name of the new rowData
##'     variable. Default in `"Precursor.Id"`.
##'
##' @param fcols `character()` with the rowData variables names that need to be
##'     [paste0()]ed together to create the new `name` variable. Default is
##'     `c("Modified.Sequence", "Precursor.Charge")`. Note that these must be
##'     present in _all_ assays.
##'
##' @param i The assays of `object` whose rowData need to be updated. By
##'     default, all assays are considered.
##'
##' @return An updated `QFeatures` instance.
##'
##' @author Laurent Gatto
##'
##' @export
##'
##' @examples
##'
##' ## Let use PSM assay of feat3, that don't have any precursor identifiers
##' data(feat4)
##' feat4
##' rowDataNames(feat4)
##'
##' ## Create precursor identifiers by concatenating the charge and the
##' ## sequencing
##' feat4 <- createPrecursorId(feat4,
##'                            name = "Precursor.Id",
##'                            fcols = c("charge", "Sequence"))
##' rowDataNames(feat4)
##' rowData(feat4[[1]])[, c("Sequence", "charge", "Precursor.Id")]
##'
##' ## As can be seen below, some precursors are duplicated, which will be
##' ## problematic when joining the assays. Should we join `1SYGFNAAR` in the
##' ## second assay with the first or the second `1SYGFNAAR` in the first assay?
##' rowData(feat4[[1]])[, "Precursor.Id", drop = FALSE]
##' rowData(feat4[[2]])[, "Precursor.Id", drop = FALSE]
##'
##' ## Here, one can either aggregate PSMs into PSMs with unique identifers (see
##' ## ?aggregateFeatures) or remove duplicated entries.
##' nrows(feat4) ## before filtering
##' feat4 <- filterFeatures(feat4, ~ !isDuplicated(Precursor.Id))
##' nrows(feat4) ## after filtering
##'
##' ## The assays can now be joined, using the newly created identifier rather
##' ## than the (default) rownames.
##' feat4 <- joinAssays(feat4, i = 1:2,
##'                     name = "Precursors",
##'                     fcol = "Precursor.Id")
##' feat4
createPrecursorId <- function(object,
                              name = "Precursor.Id",
                              fcols = c("Modified.Sequence", "Precursor.Charge"),
                              i = seq_along(object)) {
    sapply(rowDataNames(object)[i],
           function(rnms) {
               ## Check that 'name' does not already exist
               if (name %in% rnms)
                   stop("'", name, "' already exists.")
               ## Check that fcols do exist
               if (!all(fcols %in% rnms))
                   stop("'fcols' not found in some assays.")
           })
    i <- .normIndex(object, i, allowAbsent = FALSE)
    rdlist <- rowData(object)
    for (ii in i)
        rdlist[[ii]][[name]] <-
            do.call(paste0, rdlist[[ii]][, fcols, drop = FALSE])
    rowData(object) <- rdlist
    object
}


## ------------------
## Private functions
## ------------------

main_assay <- function(object)
    which.max(vapply(experiments(object),
                     nrow,
                     numeric(1)))

number_assays_in_se <- function(object)
    lengths(lapply(experiments(object), assays))