% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PomaUnivariate.R
\name{PomaUnivariate}
\alias{PomaUnivariate}
\title{Univariate Statistical Test}
\usage{
PomaUnivariate(
  data,
  method = "ttest",
  covs = NULL,
  error = NULL,
  paired = FALSE,
  var_equal = FALSE,
  adjust = "fdr",
  run_post_hoc = TRUE
)
}
\arguments{
\item{data}{A \code{SummarizedExperiment} object.}

\item{method}{Character. The univariate statistical test to be performed. Available options include "ttest" (T-test), "anova" (analysis of variance), "mann" (Wilcoxon rank-sum test), and "kruskal" (Kruskal-Wallis test).}

\item{covs}{Character vector. Indicates the names of \code{colData} columns to be included as covariates. Default is NULL (no covariates). If not NULL, an ANCOVA model will be fitted using the specified covariates. Note: The order of the covariates is important and should be listed in increasing order of importance in the experimental design.}

\item{error}{Character vector. Indicates the name of a \code{colData} column to be included as an error term (e.g., replicates). Default is NULL (no error term).}

\item{paired}{Logical. Indicates if the data is paired or not. Default is FALSE.}

\item{var_equal}{Logical. Indicates if the data variances are assumed to be equal or not. Default is FALSE.}

\item{adjust}{Character. Multiple comparisons correction method to adjust p-values. Available options are: "fdr" (false discovery rate), "holm", "hochberg", "hommel", "bonferroni", "BH" (Benjamini-Hochberg), and "BY" (Benjamini-Yekutieli).}

\item{run_post_hoc}{Logical. Indicates if computing post-hoc tests or not. Setting this parameter to FALSE can save time for large datasets.}
}
\value{
A \code{tibble} for "ttest" and "mann". A \code{list} for "anova" and "kruskal".
}
\description{
\code{PomaUnivariate} performs parametric and non-parametric univariate statistical tests on a \code{SummarizedExperiment} object to compare groups or conditions. Available methods include T-test, ANOVA, ANCOVA, Mann Whitney U Test (Wilcoxon Rank Sum Test), and Kruskal-Wallis.
}
\examples{
# Two groups
## Output columns: feature, fold_change, diff_means, pvalue, adj_pvalue, mean_xxx (group 1) mean_yyy (group 2), sd_xxx (group 1), sd_yyy (group 2)
data <- POMA::st000336 # Example SummarizedExperiment object included in POMA

## Perform T-test
ttest_results <- st000336 \%>\% 
  PomaImpute() \%>\% 
  PomaUnivariate(method = "ttest",
                 paired = FALSE,
                 var_equal = FALSE,
                 adjust = "fdr")

ttest_results \%>\% 
  dplyr::slice(1:10)

## Volcano plot
ttest_results \%>\% 
  dplyr::select(feature, fold_change, pvalue) \%>\% 
  PomaVolcano(labels = TRUE)

## Boxplot of top features
data \%>\% 
  PomaBoxplots(x = "features", 
               outcome = "group", # factorial variable to group by (e.g., treatment, sex, etc)
               feature_name = ttest_results$feature[1:10])

## Heatmap of top features
data[rownames(data) \%in\% ttest_results$feature[1:10]] \%>\% 
  PomaHeatmap(covs = c("group"), # covariates to plot (e.g., treatment, sex, etc)
              feature_names = TRUE)

## Perform Mann-Whitney U test
mann_whitney_results <- st000336 \%>\% 
  PomaImpute() \%>\% 
  PomaUnivariate(method = "mann",
                 paired = FALSE,
                 var_equal = FALSE,
                 adjust = "fdr")

mann_whitney_results \%>\% 
  dplyr::slice(1:10)

## Volcano plot
mann_whitney_results \%>\% 
  dplyr::select(feature, fold_change, pvalue) \%>\% 
  PomaVolcano(labels = TRUE)

## Boxplot of top features
data \%>\% 
  PomaBoxplots(x = "features", 
               outcome = "group", # factorial variable to group by (e.g., treatment, sex, etc)
               feature_name = mann_whitney_results$feature[1:10])

## Heatmap of top features
data[rownames(data) \%in\% mann_whitney_results$feature[1:10]] \%>\% 
  PomaHeatmap(covs = c("group"), # covariates to plot (e.g., treatment, sex, etc)
              feature_names = TRUE)

# More than 2 groups
## Output is a list with objects `result` and `post_hoc_tests`
data <- POMA::st000284 # Example SummarizedExperiment object included in POMA

## Perform Two-Way ANOVA
anova_results <- data \%>\% 
  PomaUnivariate(method = "anova",
                 covs = c("gender"),
                 error = NULL,
                 adjust = "fdr",
                 run_post_hoc = TRUE)

anova_results$result \%>\% 
  dplyr::slice(1:10)

anova_results$post_hoc_tests \%>\% 
  dplyr::slice(1:10)

## Boxplot of top features
data \%>\% 
  PomaBoxplots(x = "features",
               outcome = "factors", # factorial variable to group by (e.g., treatment, sex, etc)
               feature_name = anova_results$result$feature[1:10])

## Boxplot of top significant pairwise features (after posthoc test)
data \%>\% 
  PomaBoxplots(x = "features",
               outcome = "factors", # factorial variable to group by (e.g., treatment, sex, etc)
               feature_name = unique(anova_results$post_hoc_tests$feature)[1:10])

## Heatmap of top features
data[rownames(data) \%in\% anova_results$result$feature[1:10]] \%>\% 
  PomaHeatmap(covs = c("factors"), # covariates to plot (e.g., treatment, sex, etc)
              feature_names = TRUE)

## Perform Three-Way ANOVA
data \%>\% 
  PomaUnivariate(method = "anova", 
                 covs = c("gender", "smoking_condition"))

## Perform ANCOVA with one numeric covariate and one factor covariate
data \%>\% 
  PomaUnivariate(method = "anova", 
                 covs = c("age_at_consent", "smoking_condition"))

# Perform Kruskal-Wallis test
data \%>\% 
  PomaUnivariate(method = "kruskal", 
                 adjust = "holm",
                 run_post_hoc = TRUE)
}
\author{
Pol Castellano-Escuder
}
