% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plsda_batch.R
\name{PLSDA_batch}
\alias{PLSDA_batch}
\title{Partial Least Squares Discriminant Analysis for Batch Effect Correction}
\usage{
PLSDA_batch(
    X,
    Y.trt = NULL,
    Y.bat,
    ncomp.trt = 2,
    ncomp.bat = 2,
    keepX.trt = rep(ncol(X), ncomp.trt),
    keepX.bat = rep(ncol(X), ncomp.bat),
    max.iter = 500,
    tol = 1e-06,
    near.zero.var = TRUE,
    balance = TRUE
)
}
\arguments{
\item{X}{A numeric matrix as an explanatory matrix.
\code{NA}s are not allowed.}

\item{Y.trt}{A factor or a class vector for the treatment grouping
information (categorical outcome variable). Without the input of
\code{Y.trt}, treatment variation cannot be preserved before correcting for
batch effects.}

\item{Y.bat}{A factor or a class vector for the batch grouping information
(categorical outcome variable).}

\item{ncomp.trt}{Integer, the number of treatment associated dimensions to
include in the model.}

\item{ncomp.bat}{Integer, the number of batch associated dimensions to
include in the model.}

\item{keepX.trt}{A numeric vector of length \code{ncomp.trt}, the number
of variables to keep in \eqn{X}-loadings. By default all variables are kept
in the model. A valid input of \code{keepX.trt} extends \code{PLSDA-batch}
to a sparse version.}

\item{keepX.bat}{A numeric vector of length \code{ncomp.bat}, the number
of variables to keep in \eqn{X}-loadings. By default all variables are kept
in the model. We usually use the default setting.}

\item{max.iter}{Integer, the maximum number of iterations.}

\item{tol}{Numeric, convergence stopping value.}

\item{near.zero.var}{Logical, should be set to \code{TRUE} in particular for
data with many zero values. Setting this argument to \code{FALSE}
(when appropriate) will speed up the computations.
Default value is \code{TRUE}.}

\item{balance}{Logical, should be set to \code{TRUE}, if the
\code{batch x treatment design} is balanced (or complete). Setting this
argument to \code{FALSE} extends \code{PLSDA-batch} to
\code{weighted PLSDA-batch}. \code{wPLSDA-batch} can deal with highly
unbalanced designs but not the nested design. Default value is \code{TRUE}.}
}
\value{
\code{PLSDA_batch} returns a list that contains
the following components:

\item{X}{The original explanatory matrix \code{X}.}
\item{X.nobatch}{The batch corrected matrix with the same dimension
as the input matrix.}
\item{X.notrt}{The matrix from which treatment variation is removed.}
\item{Y}{The original outcome variables \code{Y.trt} and \code{Y.bat}.}
\item{latent_var.trt}{The treatment associated latent components
calculated with corresponding latent dimensions.}
\item{latent_var.bat}{The batch associated latent components calculated
with corresponding latent dimensions.}
\item{loadings.trt}{The estimated treatment associated latent dimensions.}
\item{loadings.bat}{The estimated batch associated latent dimensions.}
\item{tol}{The tolerance used in the iterative algorithm, convergence
stopping value.}
\item{max.iter}{The maximum number of iterations.}
\item{iter.trt}{Number of iterations of the algorthm for each treatment
associated component.}
\item{iter.bat}{Number of iterations of the algorthm for each batch
associated component.}
\item{explained_variance.trt}{The amount of data variance explained per
treatment associated component.}
\item{explained_variance.bat}{The amount of data variance explained per
batch associated component.}
\item{weight}{The sample weights, all \eqn{1} for a balanced
\code{batch x treatment design}.}
}
\description{
This function removes batch variation from the input data given the batch
grouping information and the number of associated components with
PLSDA-batch. For sparse PLSDA-batch, the number of variables to keep for each
treatment related component is needed (\code{keepX.trt}). For weighted
PLSDA-batch, the \code{balance} should be set to \code{FALSE}, and it cannot
deal with the nested \code{batch x treatment design}.
}
\examples{
## First example
## PLSDA-batch
library(TreeSummarizedExperiment) # for functions assays(),rowData()
data('AD_data')
X <- assays(AD_data$EgData)$Clr_value # centered log ratio transformed data
Y.trt <- rowData(AD_data$EgData)$Y.trt # treatment information
Y.bat <- rowData(AD_data$EgData)$Y.bat # batch information
names(Y.bat) <- names(Y.trt) <- rownames(AD_data$EgData)
ad_plsda_batch <- PLSDA_batch(X, Y.trt, Y.bat, ncomp.trt = 1, ncomp.bat = 5)
ad_X.corrected <- ad_plsda_batch$X.nobatch # batch corrected data

## Second example
## sparse PLSDA-batch
ad_splsda_batch <- PLSDA_batch(X, Y.trt, Y.bat, ncomp.trt = 1,
                                keepX.trt = 30, ncomp.bat = 5)

}
\references{
\insertRef{wang2020managing}{PLSDAbatch}

\insertRef{wang2020multivariate}{PLSDAbatch}
}
\seealso{
\code{\link{linear_regres}} and \code{\link{percentile_norm}} as
the other methods for batch effect management.
}
\author{
Yiwen Wang, Kim-Anh Lê Cao
}
