#!/usr/bin/env Rscript

#
#  This file is part of the `OmnipathR` R package
#
#  Copyright
#  2018-2025
#  Saez Lab, Uniklinik RWTH Aachen, Heidelberg University
#
#  File author(s): Alberto Valdeolivas
#                  Dénes Türei (turei.denes@gmail.com)
#                  Attila Gábor
#
#  Distributed under the MIT (Expat) License.
#  See accompanying file `LICENSE` or find a copy at
#      https://directory.fsf.org/wiki/License:Expat
#
#  Website: https://r.omnipathdb.org/
#  Git repo: https://github.com/saezlab/OmnipathR
#


#' Print OmniPath interactions
#'
#' Prints the interactions or enzyme-substrate relationships in a nice format.
#'
#' @param interactions Data frame with the interactions generated by any of the
#'     functions in \code{\link{omnipath-interactions}}.
#' @param refs Logical: include PubMed IDs where available.
#'
#' @export
#'
#' @return Returns `NULL`.
#'
#' @examples
#' enzsub <- enzyme_substrate()
#' print_interactions(head(enzsub))
#' print_interactions(tail(enzsub), refs = TRUE)
#' print_interactions(
#'     dplyr::filter(
#'         enzsub,
#'         enzyme_genesymbol == 'MAP2K1',
#'         substrate_genesymbol == 'MAPK3'
#'     )
#' )
#'
#' signor <- omnipath(resources = "SIGNOR")
#' print_interactions(head(signor))
#' #            source interaction            target n_resources
#' # 6 MAPK14 (Q16539)  ==( + )==> MAPKAPK2 (P49137)          23
#' # 4  TRPM7 (Q96QT4)  ==( + )==>    ANXA1 (P04083)          10
#' # 1  PRKG1 (Q13976)  ==( - )==>    TRPC3 (Q13507)           8
#' # 2  PTPN1 (P18031)  ==( - )==>    TRPV6 (Q9H1D0)           6
#' # 5 PRKACA (P17612)  ==( - )==>   MCOLN1 (Q9GZU1)           6
#' # 3  RACK1 (P63244)  ==( - )==>    TRPM6 (Q9BX84)           2
print_interactions <- function(interactions, refs=FALSE){

    if(nrow(interactions) == 0){
        message('No interactions')
        return(invisible(NULL))
    }

    if('enzyme' %in% colnames(interactions)){  #PTMS
        interactions <-
            interactions[
                order(
                    interactions$n_references,
                    interactions$n_resources,
                    decreasing = TRUE
                ),
            ]
        interactions$enzyme <-
            paste0(interactions$enzyme_genesymbol, ' (', interactions$enzyme ,')')
        interactions$substrate <-paste0(interactions$substrate_genesymbol,'_',
            interactions$residue_type,interactions$residue_offset,' (',
            interactions$substrate,')')

        signs <- ifelse(interactions$is_stimulation == 1,
            ifelse(interactions$is_inhibition == 1,'(+/-)','( + )'),
            ifelse(interactions$is_inhibition == 1,'( - )','( ? )'))
        interactions$interaction <- paste0('==', signs, '==>')
        if(refs){
            interactions[,c('enzyme', 'interaction', 'substrate', 'modification',
            'n_resources', 'n_references', 'references')]
        }else{
            interactions[,c('enzyme', 'interaction', 'substrate', 'modification',
            'n_resources')]
        }
    }else{
        if ('n_references' %in% colnames(interactions)){
            interactions <-
                interactions[order(
                    interactions$n_references,
                    interactions$n_resources,
                    decreasing = TRUE
                ),]
        }else{
            interactions <- interactions[order(
                interactions$n_resources,
                decreasing = TRUE
            ),]
        }
        interactions$source <- paste0(interactions$source_genesymbol, ' (',
            interactions$source, ')')
        interactions$target <- paste0(interactions$target_genesymbol, ' (',
            interactions$target, ')')

        signs <- ifelse(interactions$is_stimulation == 1,
            ifelse(interactions$is_inhibition == 1,'(+/-)','( + )'),
            ifelse(interactions$is_inhibition == 1,'( - )','( ? )'))

        direction <- ifelse(interactions$is_directed == 1, '>','')
        interactions$interaction <- paste0('==', signs, '==', direction)

        if(refs){
            if ('n_references' %in% colnames(interactions)){
                interactions[,c('source','interaction','target','n_resources',
                'n_references','references')]
            }else{
                interactions[,c('source','interaction','target','n_resources')]
            }
        }else{
            if ('n_references' %in% colnames(interactions)){
                interactions[,c('source','interaction','target','n_resources',
                'n_references')]
            }else{
                interactions[,c('source','interaction','target','n_resources')]
            }
        }
    }
}


#' Prints network paths in an edge sequence
#'
#' Pretty prints the interactions in a path.
#'
#' @param edges An igraph edge sequence object.
#' @param G igraph object (from ptms or any interaction dataset)
#'
#' @return Returns `NULL`.
#'
#' @importFrom igraph tail_of head_of
#' @export
#'
#' @examples
#' interactions <- omnipath(resources = "SignaLink3")
#' OPI_g <- interaction_graph(interactions = interactions)
#' print_path_es(
#'     igraph::shortest_paths(
#'         OPI_g,
#'         from = 'TYRO3',
#'         to = 'STAT3',
#'         output = 'epath'
#'     )$epath[[1]],
#'     OPI_g
#' )
#'
#' @seealso \itemize{
#'     \item{\code{\link{print_path_vs}}}
#' }
print_path_es <- function(edges, G){

    if(length(edges) == 0){
        message('Empty path')
        return(NULL)
    }
    signs <- ifelse(edges$is_stimulation == 1,
        ifelse(edges$is_inhibition == 1,'(+/-)','( + )'),
        ifelse(edges$is_inhibition == 1,'( - )','( ? )')
    )
    interaction <- paste0('==', signs,'==>')

    if(!is.null(edges$residue_type)){
        edges$residue_type
        if(!is.null(edges$n_references)){
            df <- data.frame(
                source = paste(
                    tail_of(G, edges)$name, ' (',
                    tail_of(G, edges)$up_ids, ')',
                    sep = ''
                ),
                interaction = interaction,
                target = paste(
                    paste0(
                        head_of(G, edges)$name, '_',
                        edges$residue_type,edges$residue_offset
                    ),
                    ' (',
                    head_of(G, edges)$up_ids, ')',
                    sep = ''
                ),
                n_resources = edges$n_resources,
                n_references = edges$n_references
            )
        }else{
            df <- data.frame(
                source = paste(
                    tail_of(G, edges)$name, ' (',
                    tail_of(G, edges)$up_ids, ')',
                    sep = ''
                ),
                interaction = interaction,
                target = paste(
                    paste0(
                        head_of(G, edges)$name, '_',
                        edges$residue_type,
                        edges$residue_offset
                    ),
                    ' (',
                    head_of(G, edges)$up_ids, ')',
                    sep = ''
                ),
                n_resources = edges$n_resources
            )
        }
    }else{
        if(!is.null(edges$n_references)){
            df <- data.frame(
                source = paste(
                    tail_of(G, edges)$name, ' (',
                    tail_of(G, edges)$up_ids, ')',
                    sep = ''
                ),
                interaction = interaction,
                target = paste(
                    head_of(G, edges)$name, ' (',
                    head_of(G, edges)$up_ids, ')',
                    sep = ''
                ),
                n_resources = edges$n_resources,
                n_references = edges$n_references
            )
        }else{
            df <- data.frame(
                source = paste(
                    tail_of(G, edges)$name, ' (',
                    tail_of(G, edges)$up_ids, ')',
                    sep = ''
                ),
                interaction = interaction,
                target = paste(
                    head_of(G, edges)$name, ' (',
                    head_of(G, edges)$up_ids, ')',
                    sep = ''
                ),
                n_resources = edges$n_resources
            )
        }
    }

    return(df)

}


#' Convert vertex sequence to named sequence to find unique
#'
#' Takes a list of nodesuences converts them to names and takes the unique
#' paths.
#'
#' @importFrom magrittr %>%
#' @importFrom purrr map
#' @noRd
unique_node_seq <- function(nodes_list){

    nodes_list %>%
    map(names) %>%
    unique

}


#' Print networks paths given by node sequence
#'
#' Prints the interactions in the path in a nice format.
#'
#' @param nodes An igraph node sequence object.
#' @param G An igraph graph object (from ptms or interactions)
#'
#' @importFrom igraph E %->%
#' @export
#'
#' @return Returns `NULL`.
#'
#' @examples
#' interactions <- omnipath(resources = "SignaLink3")
#' OPI_g <- interaction_graph(interactions = interactions)
#' print_path_vs(
#'     igraph::all_shortest_paths(
#'         OPI_g,
#'         from = 'TYRO3',
#'         to = 'STAT3'
#'     )$vpath,
#'     OPI_g
#' )
#' enzsub <- enzyme_substrate(resources=c("PhosphoSite", "SIGNOR"))
#' enzsub_g <- enzsub_graph(enzsub)
#' print_path_vs(
#'     igraph::all_shortest_paths(
#'         enzsub_g,
#'         from = 'SRC',
#'         to = 'STAT1'
#'     )$res,
#'     enzsub_g
#' )
#'
#' @seealso \code{\link{print_path_es}}
print_path_vs <- function(nodes, G){

    if(length(nodes) == 0){
        message('Empty path')
        return(invisible(NULL))
    }
    nodes_names <- unique_node_seq(nodes)
    for(i in seq(nodes_names)){
        message(paste0('Pathway ', i, ': ',
            paste(nodes_names[[i]], collapse = ' -> ')))
        edges <- c()
        for(j in 2:length(nodes_names[[i]])){
            edges <- c(edges, E(G)[nodes_names[[i]][[j-1]]  %->%
                nodes_names[[i]][[j]]])
        }
        print_path_es(E(G)[edges], G)
    }
}
