% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/motif_similarity.R
\name{motif_similarity}
\alias{motif_similarity}
\title{Compare motifs from segregated sequences}
\usage{
motif_similarity(
  streme_out,
  method = "PCC",
  normalise.scores = TRUE,
  BPPARAM = BiocParallel::bpparam(),
  ...
)
}
\arguments{
\item{streme_out}{Output from \code{\link{denovo_motifs}}.}

\item{method}{\code{character(1)} One of PCC, EUCL, SW, KL, ALLR, BHAT, HELL,
SEUCL, MAN, ALLR_LL, WEUCL, WPCC. See details.}

\item{normalise.scores}{\code{logical(1)} Favour alignments which leave fewer
unaligned positions, as well as alignments between motifs of similar length.
Similarity scores are multiplied by the ratio of
aligned positions to the total number of positions in the larger motif,
and the inverse for distance scores.}

\item{BPPARAM}{A \code{\link[BiocParallel]{BiocParallelParam-class}} object
specifying run parameters. (default = bpparam())}

\item{...}{
  Arguments passed on to \code{\link[universalmotif:compare_motifs]{universalmotif::compare_motifs}}
  \describe{
    \item{\code{motifs}}{See \code{\link[universalmotif:convert_motifs]{convert_motifs()}} for acceptable motif formats.}
    \item{\code{compare.to}}{\code{numeric} If missing, compares all motifs to all other motifs.
Otherwise compares all motifs to the specified motif(s).}
    \item{\code{db.scores}}{\code{data.frame} or \code{DataFrame}. See \code{details}.}
    \item{\code{use.freq}}{\code{numeric(1)}. For comparing the \code{multifreq} slot.}
    \item{\code{use.type}}{\code{character(1)} One of \code{'PPM'} and \code{'ICM'}.
The latter allows for taking into account the background
frequencies if \code{relative_entropy = TRUE}. Note that \code{'ICM'} is not
allowed when \code{method = c("ALLR", "ALLR_LL")}.}
    \item{\code{tryRC}}{\code{logical(1)} Try the reverse complement of the motifs as well,
report the best score.}
    \item{\code{min.overlap}}{\code{numeric(1)} Minimum overlap required when aligning the
motifs. Setting this to a number higher then the width of the motifs
will not allow any overhangs. Can also be a number between 0 and 1,
representing the minimum fraction that the motifs must overlap.}
    \item{\code{min.mean.ic}}{\code{numeric(1)} Minimum mean information content between the
two motifs for an alignment to be scored. This helps prevent scoring
alignments between low information content regions of two motifs. Note that
this can result in some comparisons failing if no alignment passes the
mean IC threshold. Use \code{\link[universalmotif:average_ic]{average_ic()}} to filter out low IC motifs to get around
this if you want to avoid getting \code{NA}s in your output.}
    \item{\code{min.position.ic}}{\code{numeric(1)} Minimum information content required between
individual alignment positions for it to be counted in the final alignment
score. It is recommended to use this together with \code{normalise.scores = TRUE},
as this will help punish scores resulting from only a fraction of an
alignment.}
    \item{\code{relative_entropy}}{\code{logical(1)} Change the ICM calculation affecting
\code{min.position.ic} and \code{min.mean.ic}. See \code{\link[universalmotif:convert_type]{convert_type()}}.}
    \item{\code{max.p}}{\code{numeric(1)} Maximum P-value allowed in reporting matches.
Only used if \code{compare.to} is set.}
    \item{\code{max.e}}{\code{numeric(1)} Maximum E-value allowed in reporting matches.
Only used if \code{compare.to} is set. The E-value is the P-value multiplied
by the number of input motifs times two.}
    \item{\code{nthreads}}{\code{numeric(1)} Run \code{\link[universalmotif:compare_motifs]{compare_motifs()}} in parallel with \code{nthreads}
threads. \code{nthreads = 0} uses all available threads.}
    \item{\code{score.strat}}{\code{character(1)} How to handle column scores calculated from
motif alignments. "sum": add up all scores. "a.mean": take the arithmetic
mean. "g.mean": take the geometric mean. "median": take the median.
"wa.mean", "wg.mean": weighted arithmetic/geometric mean. "fzt": Fisher
Z-transform. Weights are the
total information content shared between aligned columns.}
    \item{\code{output.report}}{\code{character(1)} Provide a filename for \code{\link[universalmotif:compare_motifs]{compare_motifs()}}
to write an html ouput report to. The top matches are shown alongside
figures of the match alignments. This requires the \code{knitr} and \code{rmarkdown}
packages. (Note: still in development.)}
    \item{\code{output.report.max.print}}{\code{numeric(1)} Maximum number of top matches to
print.}
  }}
}
\value{
A list of matrices containing the similarity scores between motifs
from different groups of sequences. The order of comparison is as follows,
with first element representing the rows and second element representing the
columns of the matrix:
\itemize{
    \item 1. \strong{Common motifs comparison:} Common seqs from reference
    (1) <-> comparison (2)
    \item 2. \strong{Unique motifs comparison:} Unique seqs from reference
    (1) <-> comparison (2)
    \item 3. \strong{Cross motifs comparison 1:} Unique seqs from reference
    (1) <-> comparison (1)
    \item 4. \strong{Cross motifs comparison 2:} Unique seqs from comparison
    (2) <-> reference (1)
}
The list is repeated for each set of comparison groups in input.
}
\description{
Compute motif similarity scores between motifs discovered from segregated
sequences. Wrapper around \code{\link[universalmotif]{compare_motifs}} to
compare motifs from different groups of sequences. To see the possible
similarity measures available, refer to details.
}
\details{
\subsection{Available metrics}{

The following metrics are available:
\itemize{
\item Euclidean distance (\code{EUCL}) (Choi et al. 2004)
\item Weighted Euclidean distance (\code{WEUCL})
\item Kullback-Leibler divergence (\code{KL}) (Kullback and Leibler 1951; Roepcke et al. 2005)
\item Hellinger distance (\code{HELL}) (Hellinger 1909)
\item Squared Euclidean distance (\code{SEUCL})
\item Manhattan distance (\code{MAN})
\item Pearson correlation coefficient (\code{PCC})
\item Weighted Pearson correlation coefficient (\code{WPCC})
\item Sandelin-Wasserman similarity (\code{SW}), or sum of squared distances (Sandelin and Wasserman 2004)
\item Average log-likelihood ratio (\code{ALLR}) (Wang and Stormo 2003)
\item Lower limit ALLR (\code{ALLR_LL}) (Mahony et al. 2007)
\item Bhattacharyya coefficient (\code{BHAT}) (Bhattacharyya 1943)
}

Comparisons are calculated between two motifs at a time. All possible alignments
are scored, and the best score is reported. In an alignment scores are calculated
individually between columns. How those scores are combined to generate the final
alignment scores depends on \code{score.strat}.

See the "Motif comparisons and P-values" vignette for a description of the
various metrics. Note that \code{PCC}, \code{WPCC}, \code{SW}, \code{ALLR}, \code{ALLR_LL} and \code{BHAT}
are similarities;
higher values mean more similar motifs. For the remaining metrics, values closer
to zero represent more similar motifs.

Small pseudocounts are automatically added when one of the following methods
is used: \code{KL}, \code{ALLR}, \code{ALLR_LL}, \code{IS}. This is avoid
zeros in the calculations.
}

\subsection{Calculating P-values}{

To note regarding p-values: P-values are pre-computed using the
\code{\link[universalmotif:make_DBscores]{make_DBscores()}} function. If not given, then uses a set of internal
precomputed P-values from the JASPAR2018 CORE motifs. These precalculated
scores are dependent on the length of the motifs being compared. This takes
into account that comparing small motifs with larger motifs leads to higher
scores, since the probability of finding a higher scoring alignment is
higher.

The default P-values have been precalculated for regular DNA motifs. They
are of little use for motifs with a different number of alphabet letters
(or even the \code{multifreq} slot).
}
}
\examples{
if (memes::meme_is_installed()) {
    data("CTCF_TIP_peaks", package = "MotifPeeker")
    data("CTCF_ChIP_peaks", package = "MotifPeeker")

    if (requireNamespace("BSgenome.Hsapiens.UCSC.hg38")) {
        genome_build <-
            BSgenome.Hsapiens.UCSC.hg38::BSgenome.Hsapiens.UCSC.hg38
        segregated_peaks <- segregate_seqs(CTCF_TIP_peaks, CTCF_ChIP_peaks)
        denovo_motifs <- denovo_motifs(unlist(segregated_peaks),
                            trim_seq_width = 50,
                            genome_build = genome_build,
                            discover_motifs_count = 1,
                            filter_n = 6,
                            maxw = 8,
                            minw = 8,
                            out_dir = tempdir())
        similarity_matrices <- motif_similarity(denovo_motifs)
        print(similarity_matrices)
    }
}

}
