% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods-MSnExp.R
\name{chromatogram,MSnExp-method}
\alias{chromatogram,MSnExp-method}
\alias{chromatogram}
\title{Extract chromatogram object(s)}
\usage{
\S4method{chromatogram}{MSnExp}(
  object,
  rt,
  mz,
  aggregationFun = "sum",
  missing = NA_real_,
  msLevel = 1L,
  BPPARAM = bpparam()
)
}
\arguments{
\item{object}{For \code{chromatogram}: a \code{\linkS4class{MSnExp}} or
\code{\linkS4class{OnDiskMSnExp}} object from which the chromatogram
should be extracted.}

\item{rt}{A \code{numeric(2)} or two-column \code{matrix} defining the lower
and upper boundary for the retention time range/window(s) for the
chromatogram(s). If a \code{matrix} is provided, a chromatogram is
extracted for each row. If not specified, a chromatogram representing the
full retention time range is extracted. See examples below for details.}

\item{mz}{A \code{numeric(2)} or two-column \code{matrix} defining the
mass-to-charge (mz) range(s) for the chromatogram(s). For each
spectrum/retention time, all intensity values within this mz range are
aggregated to result in the intensity value for the spectrum/retention
time. If not specified, the full mz range is considered. See examples
below for details.}

\item{aggregationFun}{\code{character} defining the function to be used for
intensity value aggregation along the mz dimension. Allowed values are
\code{"sum"} (TIC), \code{"max"} (BPC), \code{"min"} and \code{"mean"}.}

\item{missing}{\code{numeric(1)} allowing to specify the intensity value for
if for a given retention time (spectrum) no signal was measured within
the mz range. Defaults to \code{NA_real_}.}

\item{msLevel}{\code{integer} specifying the MS level from which the
chromatogram should be extracted. Defaults to \code{msLevel = 1L}.}

\item{BPPARAM}{Parallelisation backend to be used, which will
depend on the architecture. Default is
\code{BiocParallel::bpparam()}.}
}
\value{
\code{chromatogram} returns a \code{\link{MChromatograms}} object with
the number of columns corresponding to the number of files in
\code{object} and number of rows the number of specified ranges (i.e.
number of rows of matrices provided with arguments \code{mz} and/or
\code{rt}). The \code{featureData} of the returned object contains columns
\code{"mzmin"} and \code{"mzmax"} with the values from input argument
\code{mz} (if used) and \code{"rtmin"} and \code{"rtmax"} if the input
argument \code{rt} was used.
}
\description{
The \code{chromatogram} method extracts chromatogram(s) from an
\code{\linkS4class{MSnExp}} or \code{\linkS4class{OnDiskMSnExp}} object.
Depending on the provided parameters this can be a total ion chromatogram
(TIC), a base peak chromatogram (BPC) or an extracted ion chromatogram
(XIC) extracted from each sample/file.
}
\details{
Arguments \code{rt} and \code{mz} allow to specify the MS
data slice from which the chromatogram should be extracted.
The parameter \code{aggregationSum} allows to specify the function to be
used to aggregate the intensities across the mz range for the same
retention time. Setting \code{aggregationFun = "sum"} would e.g. allow
to calculate the \emph{total ion chromatogram} (TIC),
\code{aggregationFun = "max"} the \emph{base peak chromatogram} (BPC).
The length of the extracted \code{\link{Chromatogram}} object,
i.e. the number of available data points, corresponds to the number of
scans/spectra measured in the specified retention time range. If in a
specific scan (for a give retention time) no signal was measured in the
specified mz range, a \code{NA_real_} is reported as intensity for the
retention time (see Notes for more information). This can be changed
using the \code{missing} parameter.

\if{html}{\out{<div class="sourceCode">}}\preformatted{By default or if \code{mz} and/or \code{rt} are numeric vectors, the
function extracts one \code{\link{Chromatogram}} object for each file
in the \code{\linkS4class{MSnExp}} or \code{\linkS4class{OnDiskMSnExp}}
object. Providing a numeric matrix with argument \code{mz} or \code{rt}
enables to extract multiple chromatograms per file, one for each row in
the matrix. If the number of columns of \code{mz} or \code{rt} are not
equal to 2, \code{range} is called on each row of the matrix.
}\if{html}{\out{</div>}}
}
\examples{
## Read a test data file.
library(BiocParallel)
register(SerialParam())
library(msdata)
f <- c(system.file("microtofq/MM14.mzML", package = "msdata"),
     system.file("microtofq/MM8.mzML", package = "msdata"))

## Read the data as an MSnExp
msd <- readMSData(f, msLevel = 1)

## Extract the total ion chromatogram for each file:
tic <- chromatogram(msd)

tic

## Extract the TIC for the second file:
tic[1, 2]

## Plot the TIC for the first file
plot(rtime(tic[1, 1]), intensity(tic[1, 1]), type = "l",
    xlab = "rtime", ylab = "intensity", main = "TIC")

## Extract chromatograms for a MS data slices defined by retention time
## and mz ranges.
rtr <- rbind(c(10, 60), c(280, 300))
mzr <- rbind(c(140, 160), c(300, 320))
chrs <- chromatogram(msd, rt = rtr, mz = mzr)

## Each row of the returned MChromatograms object corresponds to one mz-rt
## range. The Chromatogram for the first range in the first file is empty,
## because the retention time range is outside of the file's rt range:
chrs[1, 1]

## The mz and/or rt ranges used are provided as featureData of the object
fData(chrs)

## The mz method can be used to extract the m/z ranges directly
mz(chrs)

## Also the Chromatogram for the second range in the second file is empty
chrs[2, 2]

## Get the extracted chromatogram for the first range in the second file
chr <- chrs[1, 2]
chr

plot(rtime(chr), intensity(chr), xlab = "rtime", ylab = "intensity")
}
\seealso{
\code{\link{Chromatogram}} and \code{\link{MChromatograms}} for the
classes that represent single and multiple chromatograms.
}
\author{
Johannes Rainer
}
