% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions-Chromatogram.R,
%   R/methods-Chromatogram.R
\name{Chromatogram}
\alias{Chromatogram}
\alias{aggregationFun}
\alias{Chromatogram-class}
\alias{compareChromatograms}
\alias{transformIntensity}
\alias{show,Chromatogram-method}
\alias{rtime,Chromatogram-method}
\alias{intensity,Chromatogram-method}
\alias{mz,Chromatogram-method}
\alias{precursorMz,Chromatogram-method}
\alias{fromFile,Chromatogram-method}
\alias{length,Chromatogram-method}
\alias{as.data.frame,Chromatogram-method}
\alias{filterRt,Chromatogram-method}
\alias{clean,Chromatogram-method}
\alias{plot,Chromatogram,ANY-method}
\alias{msLevel,Chromatogram-method}
\alias{isEmpty,Chromatogram-method}
\alias{productMz,Chromatogram-method}
\alias{bin,Chromatogram-method}
\alias{normalize,Chromatogram-method}
\alias{filterIntensity,Chromatogram-method}
\alias{alignRt,Chromatogram,Chromatogram-method}
\alias{compareChromatograms,Chromatogram,Chromatogram-method}
\alias{transformIntensity,Chromatogram-method}
\title{Representation of chromatographic MS data}
\usage{
Chromatogram(
  rtime = numeric(),
  intensity = numeric(),
  mz = c(NA_real_, NA_real_),
  filterMz = c(NA_real_, NA_real_),
  precursorMz = c(NA_real_, NA_real_),
  productMz = c(NA_real_, NA_real_),
  fromFile = integer(),
  aggregationFun = character(),
  msLevel = 1L
)

aggregationFun(object)

\S4method{show}{Chromatogram}(object)

\S4method{rtime}{Chromatogram}(object)

\S4method{intensity}{Chromatogram}(object)

\S4method{mz}{Chromatogram}(object, filter = FALSE)

\S4method{precursorMz}{Chromatogram}(object)

\S4method{fromFile}{Chromatogram}(object)

\S4method{length}{Chromatogram}(x)

\S4method{as.data.frame}{Chromatogram}(x)

\S4method{filterRt}{Chromatogram}(object, rt)

\S4method{clean}{Chromatogram}(object, all = FALSE, na.rm = FALSE)

\S4method{plot}{Chromatogram,ANY}(
  x,
  col = "#00000060",
  lty = 1,
  type = "l",
  xlab = "retention time",
  ylab = "intensity",
  main = NULL,
  ...
)

\S4method{msLevel}{Chromatogram}(object)

\S4method{isEmpty}{Chromatogram}(x)

\S4method{productMz}{Chromatogram}(object)

\S4method{bin}{Chromatogram}(
  x,
  binSize = 0.5,
  breaks = seq(floor(min(rtime(x))), ceiling(max(rtime(x))), by = binSize),
  fun = max
)

\S4method{normalize}{Chromatogram}(object, method = c("max", "sum"))

\S4method{filterIntensity}{Chromatogram}(object, intensity = 0, ...)

\S4method{alignRt}{Chromatogram,Chromatogram}(x, y, method = c("closest", "approx"), ...)

\S4method{compareChromatograms}{Chromatogram,Chromatogram}(
  x,
  y,
  ALIGNFUN = alignRt,
  ALIGNFUNARGS = list(),
  FUN = cor,
  FUNARGS = list(use = "pairwise.complete.obs"),
  ...
)

\S4method{transformIntensity}{Chromatogram}(object, FUN = identity)
}
\arguments{
\item{rtime}{for \code{Chromatogram}: \code{numeric} with the retention times (length
has to be equal to the length of \code{intensity}).}

\item{intensity}{for \code{Chromatogram}: \code{numeric} with the intensity values
(length has to be equal to the length of \code{rtime}). For \code{filterIntensity}:
\code{numeric(1)} or \code{function} to use to filter intensities. See description
for details.}

\item{mz}{for \code{Chromatogram}: \code{numeric(2)} representing the mz value range
(min, max) on which the chromatogram was created. This is supposed to
contain the \emph{real} range of mz values in contrast to \code{filterMz}.
If not applicable use \code{mzrange = c(0, 0)}.}

\item{filterMz}{for \code{Chromatogram}: \code{numeric(2)} representing the mz value
range (min, max) that was used to filter the original object on m/z
dimension. If not applicable use \code{filterMz = c(0, 0)}.}

\item{precursorMz}{for \code{Chromatogram}: \code{numeric(2)} for SRM/MRM transitions.
Represents the mz of the precursor ion. See details for more information.}

\item{productMz}{for \code{Chromatogram}: \code{numeric(2)} for SRM/MRM transitions.
Represents the mz of the product. See details for more information.}

\item{fromFile}{for \code{Chromatogram}: \code{integer(1)} the index of the file within
the \code{OnDiskMSnExp} or \code{MSnExp} from which the chromatogram was extracted.}

\item{aggregationFun}{for \code{Chromatogram}: \code{character} string specifying
the function that was used to aggregate intensity values for the same
retention time across the mz range. Supported are \code{"sum"} (total ion
chromatogram), \code{"max"} (base peak chromatogram), \code{"min"} and \code{"mean"}.}

\item{msLevel}{for \code{Chromatogram}: \code{integer(1)} with the MS level from
which the chromatogram was extracted.}

\item{object}{\code{Chromatogram} object.}

\item{filter}{for \code{mz}: \code{logical(1)} defining whether the m/z range to filter
the originating object (e.g. \code{MSnExp} object) should be returned or the
m/z range of the actual data. Defaults to \code{filter = FALSE}.}

\item{x}{\code{Chromatogram} object.}

\item{rt}{for \code{filterRt}: \code{numeric(2)} defining the lower and upper retention
time to which the \code{Chromatogram} should be subsetted.}

\item{all}{for \code{clean}: \code{logical(1)} whether all 0 intensities should be
removed. Defaults to \code{all = FALSE}. See \code{\link[=clean]{clean()}} for details.}

\item{na.rm}{for \code{clean}: if all \code{NA} intensities should be removed before
cleaning the \code{Chromatogram}. Defaults to \code{clean = FALSE}.}

\item{col}{for \code{plot}: the color to be used for plotting.}

\item{lty}{for \code{plot}: the line type. See help page of \code{plot} in
the \code{graphics} package for details.}

\item{type}{for \code{plot}: the type of plot. See help page of \code{plot} in
the \code{graphics} package for details.}

\item{xlab}{for \code{plot}: the x-axis label.}

\item{ylab}{for \code{plot}: the y-axis label.}

\item{main}{for \code{plot}: the plot title. If not provided the mz range
will be used as plot title.}

\item{...}{for \code{plot}: additional arguments to be passed to the
base \code{plot} function. For \code{filterIntensity}: additional parameters passed
along to the function provided with \code{intensity}.
For \code{compareChromatograms}: ignored}

\item{binSize}{for \code{bin}: \code{numeric(1)} with the size of the bins
(in seconds). Defaults to \code{binSize = 0.5}.}

\item{breaks}{for \code{bin}: \code{numeric} defining the bins. Usually not
required as the function calculates the bins automatically based on
\code{binSize}.}

\item{fun}{for \code{bin}: function to be used to aggregate the intensity
values falling within each bin. Defaults to \code{fun = max}.}

\item{method}{\code{character(1)}. For \code{normalise}: defining whether each
chromatogram should be normalized to its maximum signal
(\code{method = "max"}) or total signal (\code{method = "sum"}).
For \code{alignRt}: aligning approach that should be used (see description).
Defaults to \code{method = "closest"}.}

\item{y}{for \code{alignRt}: \code{Chromatogram} against which \code{x} should be aligned
against.}

\item{ALIGNFUN}{for \code{compareChromatograms}: function to align chromatogram
\code{x} against chromatogram \code{y}. Defaults to \code{alignRt}.}

\item{ALIGNFUNARGS}{\code{list} of parameters to be passed to \code{ALIGNFUN}.}

\item{FUN}{for \code{compareChromatograms}: function to calculate a similarity
score on the intensity values of the compared and aligned chromatograms.
Defaults to \code{FUN = cor}. For \code{transformIntensity}: function to transform
chromatograms' intensity values. Defaults to \code{FUN = identity}.}

\item{FUNARGS}{for \code{compareChromatograms}: \code{list} with additional parameters
for \code{FUN}. Defaults to \code{FUNARGS = list(use = "pairwise.complete.obs")}.}
}
\description{
The \code{Chromatogram} class is designed to store
chromatographic MS data, i.e. pairs of retention time and intensity
values. Instances of the class can be created with the
\code{Chromatogram} constructor function but in most cases the dedicated
methods for \linkS4class{OnDiskMSnExp} and
\linkS4class{MSnExp} objects extracting chromatograms should be
used instead (i.e. the \code{\link[=chromatogram]{chromatogram()}} method).
}
\details{
The \code{mz}, \code{filterMz}, \code{precursorMz} and
\code{productMz} are stored as a \code{numeric(2)} representing a range
even if the chromatogram was generated for only a single ion (i.e. a
single mz value). Using ranges for \code{mz} values allow this class to
be used also for e.g. total ion chromatograms or base peak chromatograms.

\if{html}{\out{<div class="sourceCode">}}\preformatted{The slots `precursorMz` and `productMz` allow to represent SRM
(single reaction monitoring) and MRM (multiple SRM) chromatograms. As
example, a `Chromatogram` for a SRM transition 273 -> 153 will have
a `@precursorMz = c(273, 273)` and a
`@productMz = c(153, 153)`.
}\if{html}{\out{</div>}}
}
\section{Object creation}{


\code{Chromatogram} objects can be extracted from an \code{MSnExp} or \code{OnDiskMSnExp}
object with the \code{chromatogram()} function.

Alternatively, the constructor function \code{Chromatogram} can be used, which
takes arguments \code{rtime}, \code{intensity}, \code{mz}, \code{filterMz}, \code{precursorMz},
\code{productMz}, \code{fromFile}, \code{aggregationFun} and \code{msLevel}.
}

\section{Data access and coercion}{

\itemize{
\item \code{aggregationFun}: gets the aggregation function used to create the
\code{Chromatogram}.
\item \code{as.data.frame}: returns a \code{data.frame} with columns \code{"rtime"} and
\code{"intensity"}.
\item \code{fromFile}: returns an \code{integer(1)} with the index of the originating file.
\item \code{intensity}: returns the intensities from the \code{Chromatogram}.
\item \code{isEmpty}: returns \code{TRUE} if the chromatogram is empty or has only \code{NA}
intensities.
\item \code{length}: returns the length (i.e. number of data points) of the
\code{Chromatogram}.
\item \code{msLevel}: returns an \code{integer(1)} with the MS level of the chromatogram.
\item \code{mz}: get the m/z (range) from the \code{Chromatogram}. The function returns
a \code{numeric(2)} with the lower and upper boundaries. Parameter \code{filter}
allows to specify whether the m/z range used to filter the originating
object should be returned or the m/z range of the actual data.
\item \code{precursorMz}: get the m/z of the precursor ion. The function returns a
\code{numeric(2)} with the lower and upper boundary.
\item \code{productMz}: get the m/z of the producto chromatogram/ion. The function
returns a \code{numeric(2)} with the lower and upper m/z value.
\item \code{rtime}: returns the retention times from the \code{Chromatogram}.
}
}

\section{Data subsetting and filtering}{

\itemize{
\item \code{filterRt}: filter/subset the \code{Chromatogram} to the specified retention
time range (defined with parameter \code{rt}).
\item \code{filterIntensity}: filter a \code{\link[=Chromatogram]{Chromatogram()}} object removing data
points with intensities below a user provided threshold. If \code{intensity}
is a \code{numeric} value, the returned chromatogram will only contain data
points with intensities > \code{intensity}. In addition it is possible to
provide a function to perform the filtering.
This function is expected to take the input \code{Chromatogram} (\code{object}) and
to return a logical vector with the same length then there are data points
in \code{object} with \code{TRUE} for data points that should be kept and \code{FALSE}
for data points that should be removed. See examples below.
}
}

\section{Data processing and manipulation}{

\itemize{
\item \code{alignRt}: Aligns chromatogram \code{x} against chromatogram \code{y}. The resulting
chromatogram has the same length (number of data points) than \code{y} and the
same retention times thus allowing to perform any pair-wise comparisons
between the chromatograms. If \code{x} is a \code{\link[=MChromatograms]{MChromatograms()}} object, each
\code{Chromatogram} in it is aligned against \code{y}. Additional parameters (\code{...})
are passed along to the alignment functions (e.g. \code{\link[=closest]{closest()}}).

Parameter \code{method} allows to specify which alignment method
should be used. Currently there are the following options:
\itemize{
\item \code{method = "closest"} (the default): match data points in the first
chromatogram (\code{x}) to those of the second (\code{y}) based on the difference
between their retention times: each data point in \code{x} is assigned to the
data point in \code{y} with the smallest difference in their retention times
if their difference is smaller than the minimum average difference
between retention times in \code{x} or \code{y} (parameter \code{tolerance} for the
call to the \code{\link[=closest]{closest()}} function).
By setting \code{tolerance = 0} only exact retention times are matched against
each other (i.e. only values are kept with exactly the same retention
times between both chromatograms).
\item \code{method = "approx"}: uses the base R \code{approx} function to approximate
intensities in \code{x} to the retention times in \code{y} (using linear
interpolation). This should only be used for chromatograms that were
measured in the same measurement run (e.g. MS1 and corresponding MS2
chromatograms from SWATH experiments).
}
\item \code{bin}: aggregates intensity values from a chromatogram in discrete bins
along the retention time axis and returns a \code{Chromatogram} object with
the retention time representing the mid-point of the bins and the
intensity the binned signal. Parameters \code{binSize} and \code{breaks} allow to
define the binning, \code{fun} the function which should be used to aggregate
the intensities within a bin.
\item \code{compareChromatograms}: calculates a similarity score between 2
chromatograms after aligning them. Parameter \code{ALIGNFUN} allows to define
a function that can be used to align \code{x} against \code{y} (defaults to
\code{ALIGNFUN = alignRt}). Subsequently, the similarity is calculated on the
aligned intensities with the function provided with parameter \code{FUN} which
defaults to \code{cor} (hence by default the Pearson correlation is calculated
between the aligned intensities of the two compared chromatograms).
Additional parameters can be passed to the \code{ALIGNFUN} and \code{FUN} with the
parameter \code{ALIGNFUNARGS} and \code{FUNARGS}, respectively.
\item \code{clean}: removes 0-intensity data points (and \code{NA} values). See \code{\link[=clean]{clean()}}
for details.
\item \code{normalize}, \code{normalise}: \emph{normalises} the intensities of a chromatogram by
dividing them either by the maximum intensity (\code{method = "max"}) or total
intensity (\code{method = "sum"}) of the chromatogram.
\item \code{transformIntensity}: allows to manipulate the intensity values of a
chromatogram using a user provided function. See below for examples.
}
}

\section{Data visualization}{

\itemize{
\item \code{plot}: plots a \code{Chromatogram} object.
}
}

\examples{

## Create a simple Chromatogram object.
ints <- abs(rnorm(100, sd = 100))
rts <- seq_len(length(ints))
chr <- Chromatogram(rtime = rts, intensity = ints)
chr

## Extract intensities
intensity(chr)

## Extract retention times
rtime(chr)

## Extract the mz range - is NA for the present example
mz(chr)

## plot the Chromatogram
plot(chr)

## Create a simple Chromatogram object based on random values.
chr <- Chromatogram(intensity = abs(rnorm(1000, mean = 2000, sd = 200)),
        rtime = sort(abs(rnorm(1000, mean = 10, sd = 5))))
chr

## Get the intensities
head(intensity(chr))

## Get the retention time
head(rtime(chr))

## What is the retention time range of the object?
range(rtime(chr))

## Filter the chromatogram to keep only values between 4 and 10 seconds
chr2 <- filterRt(chr, rt = c(4, 10))

range(rtime(chr2))

## Data manipulations:

## normalize a chromatogram
par(mfrow = c(1, 2))
plot(chr)
plot(normalize(chr, method = "max"))

## Align chromatograms against each other

chr1 <- Chromatogram(rtime = c(1, 2, 3, 4, 5, 6, 7, 8, 9, 10),
    intensity = c(3, 5, 14, 30, 24, 6, 2, 1, 1, 0))
chr2 <- Chromatogram(rtime = c(2.5, 3.42, 4.5, 5.43, 6.5),
    intensity = c(5, 12, 15, 11, 5))

plot(chr1, col = "black")
points(rtime(chr2), intensity(chr2), col = "blue", type = "l")

## Align chr2 to chr1 without interpolation
res <- alignRt(chr2, chr1)
rtime(res)
intensity(res)
points(rtime(res), intensity(res), col = "#00ff0080", type = "l")

## Align chr2 to chr1 with interpolation
res <- alignRt(chr2, chr1, method = "approx")
points(rtime(res), intensity(res), col = "#ff000080", type = "l")
legend("topright", col = c("black", "blue", "#00ff0080","#ff000080"),lty = 1,
    legend = c("chr1", "chr2", "chr2 matchRtime", "chr2 approx"))


## Compare Chromatograms. Align chromatograms with `alignRt` and
## method `"approx"`
compareChromatograms(chr2, chr1, ALIGNFUNARGS = list(method = "approx"))

## Data filtering

chr1 <- Chromatogram(rtime = c(1, 2, 3, 4, 5, 6, 7, 8, 9, 10),
    intensity = c(3, 5, 14, 30, 24, 6, 2, 1, 1, 0))

## Remove data points with intensities below 10
res <- filterIntensity(chr1, 10)
intensity(res)

## Remove data points with an intensity lower than 10\% of the maximum
## intensity in the Chromatogram
filt_fun <- function(x, prop = 0.1) {
    x@intensity >= max(x@intensity, na.rm = TRUE) * prop
}
res <- filterIntensity(chr1, filt_fun)
intensity(res)

## Remove data points with an intensity lower than half of the maximum
res <- filterIntensity(chr1, filt_fun, prop = 0.5)
intensity(res)

## log2 transform intensity values
res <- transformIntensity(chr1, log2)
intensity(res)
log2(intensity(chr1))
}
\seealso{
\link{MChromatograms} for combining \code{Chromatogram} in
a two-dimensional matrix (rows being mz-rt ranges, columns samples).
\verb{chromatogram()] for the method to extract chromatogram data from an }MSnExp\code{or}OnDiskMSnExp\verb{object. [clean()] for the method to *clean* a}Chromatogram` object.
}
\author{
Johannes Rainer
}
