#' @title Plot region trends for two-dimensional lipid features
#' @description Visualize lipid trend analysis results in two-dimensional
#' feature space (e.g., chain length vs. double bond count). The resulting
#' heatmap highlights regions with significant trends between groups based on
#' the smoothed permutation test.
#'
#' Each point in the heatmap represents a lipid defined by two continuous
#' features (such as chain length and double bond). Color indicates the
#' log2 fold-change between groups, and asterisks denote levels of marginal
#' significance. If \code{abund_weight=TRUE} during analysis, the point size
#' reflects the mean abundance of each lipid. If \code{abund_weight=FALSE}, all
#' points are displayed with equal size.
#' @param object A LipidTrendSE object generated by \code{analyzeLipidRegion()}
#' for two-dimensional analysis.
#' @param p_cutoff Numeric. Significance threshold for smoothed permutation
#' p-values. Default is \code{0.05}.
#' @param log2FC_cutoff Numeric. Threshold for absolute log2 fold-change used to
#'  truncate the color scale in the heatmap. Values exceeding this threshold
#' will be capped at the cutoff value for visualization purposes. Default is
#' \code{3}.
#' @return
#' If \code{split_chain=TRUE} in the analysis, the function returns a list with:
#' \enumerate{
#'      \item{even_result: A ggplot object for even-chain lipids,
#'      or \code{NULL} if none exist.}
#'      \item{odd_result: A ggplot object for odd-chain lipids, or \code{NULL}
#'      if none exist.}
#' }
#' If \code{split_chain=FALSE}, the function returns a single ggplot object
#' visualizing all lipids together.
#' @examples
#' data("lipid_se_2D")
#' res_se <- analyzeLipidRegion(
#'     lipid_se=lipid_se_2D, ref_group="sgCtrl", split_chain=FALSE,
#'     chain_col=NULL, radius=3, own_contri=0.5, test="t.test",
#'     abund_weight=TRUE, permute_time=100)
#' plot_2D <- plotRegion2D(res_se, p_cutoff=0.05)
#' @importFrom ggplot2 ggplot aes geom_point theme_bw scale_colour_gradient2
#'   geom_text scale_x_continuous scale_y_continuous labs scale_size
#'   geom_segment geom_tile scale_fill_gradient2 theme
#' @importFrom stats setNames
#' @importFrom ggnewscale new_scale_fill
#'
#' @export
#' @seealso
#' \code{\link{analyzeLipidRegion}} for generating the input LipidTrendSE object
setGeneric("plotRegion2D", function(
        object, p_cutoff=0.05, log2FC_cutoff=3)
    standardGeneric("plotRegion2D"))

#' @rdname plotRegion2D
#' @aliases plotRegion2D,LipidTrendSE-method
#' @exportMethod plotRegion2D
setMethod(
    "plotRegion2D", "LipidTrendSE",
    function(object, p_cutoff=0.05, log2FC_cutoff=3) {
    if (!is.numeric(p_cutoff) || p_cutoff < 0 || p_cutoff > 1) {
        stop("p_cutoff must be a numeric value between 0 and 1")
    }
    if (!is.numeric(log2FC_cutoff) || log2FC_cutoff <= 0) {
        stop("log2FC_cutoff must be a positive numeric value")
    }
    abund_weight <- .abund_weight(object)
    if (.split_chain(object)) {
        even_results <- even_chain_result(object)
        odd_results <- odd_chain_result(object)
        plots <- list(
            even_result=if(
                !is.null(even_results)) .trendPlot2D(
                    even_results, p_cutoff, log2FC_cutoff,
                    abund_weight) else NULL,
            odd_result=if(
                !is.null(odd_results)) .trendPlot2D(
                    odd_results, p_cutoff, log2FC_cutoff,
                    abund_weight) else NULL
        )
        return(plot2D=plots)
    } else {
        results <- result(object)
        plot <- if(!is.null(results)) .trendPlot2D(
            results, p_cutoff, log2FC_cutoff, abund_weight) else NULL
        return(plot2D=plot)
    }
})

.trendPlot2D <- function(res, p_cutoff, log2FC_cutoff, abund_weight){
    X.info <- res[, seq_len(2)]
    direction.int <- ifelse(res$direction == "+", 1, -1)
    smoothing.pval <- res$smoothing.pval.BH
    selected.regions <- list(
        high=ifelse(smoothing.pval<p_cutoff&direction.int>0, "High", "None"),
        low=ifelse(smoothing.pval<p_cutoff&direction.int<0, "Low", "None"))
    region_vec <- ifelse(
        smoothing.pval < p_cutoff & direction.int > 0, "High",
        ifelse(smoothing.pval < p_cutoff & direction.int < 0, "Low", "None"))
    dis_res <- .countDistance(X.info)
    x.distance <- dis_res$x_distance
    y.distance <- dis_res$y_distance
    buildAllWalls <- function(region_type) {
        walls <- lapply(seq_len(nrow(X.info)), function(x) {
            .buildWall(
                feature.idx=x, X.info, selected.regions[[region_type]],
                x.distance, y.distance)
        })
        walls_df <- as.data.frame(do.call(rbind, walls))
        colnames(walls_df)[seq_len(2)] <- c("x","y")
        split(walls_df[, seq_len(2)], walls_df$pos)
    }
    walls <- list(high=buildAllWalls("high"),low=buildAllWalls("low"))
    heatmap.df <- data.frame(
        v1=X.info[,1], v2=X.info[,2], avg.abund=res$avg.abund,
        log2.FC=pmin(pmax(res$log2.FC, -log2FC_cutoff), log2FC_cutoff),
        pval.annotate=vapply(
            res$marginal.pval.BH, function(pval) .pvalAnnotation(pval),
            FUN.VALUE=character(1)),
        signed.logp.smooth=direction.int*smoothing.pval) %>%
        mutate(
            region=region_vec,
            label_vjust=ifelse(pval.annotate == "-", 0.3, 0.7))
    heatmap <- .plotHeatmap(
        X.info, heatmap.df, walls, x.distance, y.distance, abund_weight)
    return(heatmap)
}

.buildWall <- function(
        feature.idx, X.info, selected.region, x.distance, y.distance) {
    if (!is.numeric(feature.idx) || length(feature.idx) != 1) {
        stop("feature.idx must be a single numeric value")
    }
    if (!is.numeric(x.distance) || !is.numeric(y.distance)) {
        stop("distance parameters must be numeric")
    }
    dis_res <- .countDistance(X.info)
    dist_matrix <- dis_res$normalized_matrix
    ref_point <- dist_matrix[feature.idx, ]
    neighbors <- lapply(
        c("right", "left", "top", "bottom"), function(direction) {
            .findNeighbor(ref_point, dist_matrix, direction)})
    names(neighbors) <- c("right", "left", "top", "bottom")
    wall_positions <- character()
    needsWall <- function(neighbor_indices, direction) {
        feature_region <- selected.region[feature.idx]
        n_neighbors <- sum(neighbor_indices)
        if (direction %in% c("left", "bottom")) {
            return(feature_region != "None" && n_neighbors == 0)
        }
        if (n_neighbors == 1) {
            neighbor_region <- selected.region[which(neighbor_indices)]
            return(feature_region != neighbor_region)
        } else {
            return(feature_region != "None")
        }
    }
    need_walls <- vapply(names(neighbors), function(direction) {
        needsWall(neighbors[[direction]], direction)
    }, logical(1))
    wall_positions <- names(neighbors)[need_walls]
    if (length(wall_positions) == 0) {
        wall_out <- data.frame(matrix(ncol = ncol(X.info) + 1, nrow = 0))
    } else {
        wall_out <- data.frame(do.call(
            rbind, replicate(
                length(wall_positions), as.numeric(X.info[feature.idx, ]),
                simplify = FALSE)), pos=wall_positions)
    }
    colnames(wall_out) <- c(colnames(X.info), "pos")
    return(wall_out)
}

.findNeighbor <- function(
        self, others, type=c("right", "left", "top", "bottom")) {
    type <- match.arg(type)
    x_diff <- others[, 1] - self[1]
    y_diff <- others[, 2] - self[2]
    switch(
        type,
        "right"  = y_diff == 0 & x_diff == 1,
        "left"   = y_diff == 0 & x_diff == -1,
        "top"    = x_diff == 0 & y_diff == 1,
        "bottom" = x_diff == 0 & y_diff == -1)
}

.pvalAnnotation <- function(pval) {
    if (length(pval) == 0 || is.na(pval)) return("")
    dplyr::case_when(
        pval < 0.001 ~ "***", pval < 0.01 ~ "**", pval < 0.05 ~ "*",
        pval < 0.1 ~ "-", TRUE ~ "")
}

.plotHeatmap <- function(
        X.info, heatmap.df, walls, x.distance, y.distance, abund_weight) {
    if (!all(
            c("v1", "v2", "avg.abund", "log2.FC", "pval.annotate", "region")
            %in% colnames(heatmap.df))) {
        stop("Required columns missing from heatmap.df")
    }
    if (!is.numeric(x.distance) || !is.numeric(y.distance)) {
        stop("Distance parameters must be numeric")
    }
    base_plot <- ggplot(heatmap.df, aes(x=v1, y=v2)) +
        theme_bw() + geom_tile(aes(fill=region), alpha=0.1, color=NA) +
        scale_fill_manual(
            name="Trend", breaks=c("High", "Low"), na.value=NA,
            labels=c("High"="Increase", "Low"="Decrease"),
            values=c("High"="#FF5151", "Low"="#4169E1", "None"=NA)) +
        theme(
            panel.grid.minor=element_blank(),
            panel.grid.major=element_line(color="grey95")) +
        scale_x_continuous(
            breaks=seq(min(X.info[,1]), max(X.info[,1]), x.distance)) +
        scale_y_continuous(
            breaks=seq(min(X.info[,2]), max(X.info[,2]), y.distance)) +
        labs(x=colnames(X.info)[1], y=colnames(X.info)[2])
    base_plot <- base_plot + new_scale_fill()
    final_plot <- .formingFinalPlot(base_plot, heatmap.df, abund_weight)
    regions <- c("high", "low")
    directions <- c("top", "right", "bottom", "left")
    colors <- c(high="#FF5151", low="#4169E1")
    segment_layers <- unlist(
        lapply(regions, function(region) {
            vapply(directions, function(direction) {
                layer <- .createSegmentLayer(
                    walls[[region]][[direction]], direction, colors[region],
                    x.distance, y.distance)
                if (is.null(layer)) {
                    return(list(NULL))
                } else {
                    return(list(layer))
                }
            }, FUN.VALUE=list(NULL))
        }), recursive=FALSE)
    if (length(segment_layers) > 0) {
        final_plot <- Reduce(`+`, segment_layers, init=final_plot)
    }
    return(final_plot)
}

.formingFinalPlot <- function(base_plot, heatmap.df, abund_weight) {
    if (abund_weight) {
        plot <- base_plot +
            geom_point(
                aes(size=avg.abund, fill=log2.FC), shape=21, color="grey70",
                stroke=0.3) +
            scale_size(name="Abundance", range=c(3,12))
    } else {
        plot <- base_plot +
            geom_point(
                aes(fill=log2.FC), size=7, shape=21, color="grey70", stroke=0.3)
    }
    plot <- plot +
        scale_fill_gradient2(
            name="Log2(FC)", high="#FF4500", mid="gray90", low="#4169E1",
            midpoint=0) +
        geom_text(aes(
            label=pval.annotate, vjust=label_vjust), color="#3C3C3C",
            size=3, hjust=0.5) +
        theme(
            axis.title=element_text(size=15), axis.text=element_text(size=14),
            legend.title=element_text(size=12),
            legend.text=element_text(size=11))
    return(plot)
}

.createSegmentLayer <- function(wall_data, direction, color, x_dist, y_dist){
    if (is.null(wall_data)) {return(NULL)}
    segments <- .calcSegment(wall_data, direction, x_dist, y_dist)
    segment_df <- data.frame(
        x=segments$x, xend=segments$xend, y=segments$y, yend=segments$yend)
    geom_segment(
        data=segment_df, mapping=aes(x=x, xend=xend, y=y, yend=yend),
        linewidth=0.8, colour=color, alpha=0.8)
}

.calcSegment <- function(wall_data, direction, x_dist, y_dist) {
    if (is.null(wall_data)) {return(NULL)}
    switch(
        direction,
        "top"=list(
            x=wall_data$x-x_dist/2, xend=wall_data$x+x_dist/2,
            y=wall_data$y+y_dist/2, yend=wall_data$y+y_dist/2),
        "right"=list(
            x=wall_data$x+x_dist/2, xend=wall_data$x+x_dist/2,
            y=wall_data$y-y_dist/2, yend=wall_data$y+y_dist/2),
        "bottom"=list(
            x=wall_data$x-x_dist/2, xend=wall_data$x+x_dist/2,
            y=wall_data$y-y_dist/2, yend=wall_data$y-y_dist/2),
        "left"=list(
            x=wall_data$x-x_dist/2, xend=wall_data$x-x_dist/2,
            y=wall_data$y-y_dist/2, yend=wall_data$y+y_dist/2))
}
