#' Send graph to Cytoscape via CyREST
#' @description View the KEGG pathway in Cytoscape. With either the 
#' 'expanded edges' or 'stacked nodes' layout, users can visualize and interact 
#' with the graphs [strictly] as they are documented in the most recent KGML
#' available from KEGG.  
#' This function is a modified version of the function send2cy(),
#'  which is part of the cyREST utility functions.
#' @export
#' @import methods
#' @import httr
#' @import RJSONIO
#' @param graph_object An igraph object such as the one generated by the 
#' function \code{\link{get_graph_object}}
#' @param title An optional title for the graph when it is in Cytoscape
#' @param edge_width_attribute The attribute that will be used for edge width; 
#' if data is not added or the attribute is not part of the graphing 
#' information, the edge width will default to 1. 
#' @param port.number The port address for Cytoscape 
#' @return A dynamic map in Cytoscape automatically formatted for convenient 
#' viewing.  
#' @examples 
#' p53_KGML <- get_KGML("hsa04115")
#' p53_KEGG_mappings <- expand_KEGG_mappings(p53_KGML, FALSE)
#' nodes <- node_mapping_info(p53_KEGG_mappings)
#' 
#' p53_edges <- expand_KEGG_edges(p53_KGML, p53_KEGG_mappings)
#' edges <- edge_mapping_info(p53_edges) 
#' 
#' p53_graph_object <- get_graph_object(nodes, edges)
#'                                       
#' @examples \dontrun{
#' cyto_vis(p53_graph_object, "Default p53 Graph [no data added]")
#' 
#' #Workflow to visualize graph with data-dependent attributes:
#' 
#' p53_KGML <- get_KGML("hsa04115")
#' p53_KEGG_mappings <- expand_KEGG_mappings(p53_KGML)
#' nodes <- node_mapping_info(p53_KEGG_mappings)
#' 
#' p53_edges <- expand_KEGG_edges(p53_KGML, p53_KEGG_mappings)
#' 
#' p53_HA1E_data <- overlap_info(p53_KGML, p53_KEGG_mappings, "HA1E", 
#'                                data_type = "100_bing")
#' p53_edges_plus_data <- add_edge_data(p53_edges, p53_KEGG_mappings, 
#'                                       p53_HA1E_data, c(3, 10,12),
#'                                       only_mapped = TRUE)
#' 
#' edges <- edge_mapping_info(p53_edges_plus_data, data_added = TRUE)
#'                                             
#' p53_plus_data_graph_object <- get_graph_object(nodes, edges)
#'                                                 
#' cyto_vis(p53_plus_data_graph_object, "p53 Graph: Mapped Edges + HA1E Data",
#'          edge_width_attribute = "UP")
#'          }

cyto_vis <-
    function(graph_object, title = "Cytoscape Graph Window",
            edge_width_attribute = "summary_score",
            port.number = 1234) {
    base.url = paste("http://localhost:", toString(port.number), "/v1", sep="")

    if (edge_width_attribute %in% names(igraph::edge_attr(graph_object))){
        min.summary_score <- min(abs(igraph::E(graph_object)$summary_score),
                                na.rm = TRUE)
        max.summary_score <- max(abs(igraph::E(graph_object)$summary_score), 
                                na.rm = TRUE)
        map_edge_width <- TRUE
    }
    else {
        min.summary_score <- NA
        max.summary_score <- NA
        map_edge_width <- FALSE
    }
    
    graph_object$name <- title
    cygraph <- toCytoscape(graph_object)
        
    network.url = paste(base.url, "networks", sep="/")
    res <- httr::POST(url=network.url, body=cygraph, encode="json")
    network.suid = unname(RJSONIO::fromJSON(rawToChar(res$content)))
    
    style.name = "myKEGGstyle"
    mappings <- generate_mappings(style.name, map_edge_width, 
                                    edge_width_attribute, 
                                    min_score = min.summary_score, 
                                    max_score = max.summary_score)
    style.url = paste(base.url, "styles", sep="/")
    style <- list(title=style.name, defaults = mappings[[1]],
                  mappings = mappings[[2]])
    style.JSON <- RJSONIO::toJSON(style)
    httr::POST(url=style.url, body=style.JSON, encode = "json")
    
    dependencies.url <- paste(style.url, style.name, "dependencies", sep="/")
    lock_style <- list(visualPropertyDependency="nodeSizeLocked", 
                       enabled = "false")
    lock_style.JSON <- toJSON(list(lock_style))
    PUT(url=dependencies.url, body=lock_style.JSON, encode="json")
    
    apply.style.url = paste(base.url, "apply/styles", style.name ,
                            toString(network.suid), sep="/")
    httr::GET(apply.style.url)
    
    fit_content.url <- paste(base.url, "apply/fit", network.suid, sep="/")
    httr::GET(url=fit_content.url)
    }
