#' Ibex Matrix Interface
#'
#' This function runs the Ibex algorithm to generate latent vectors from 
#' input data. The output can be returned as a matrix, with options to choose 
#' between deep learning autoencoders or geometric transformations based on 
#' the BLOSUM62 matrix.
#'
#' @examples
#' # Get Data
#' ibex_example <- get(data("ibex_example"))
#' 
#' # Using the encoder method with a variational autoencoder
#' ibex_values <- Ibex_matrix(ibex_example, 
#'                            chain = "Heavy",
#'                            method = "encoder",
#'                            encoder.model = "VAE",
#'                            encoder.input = "atchleyFactors")
#'
#' # Using the geometric method with a specified angle
#' ibex_values <- Ibex_matrix(ibex_example, 
#'                            chain = "Heavy",
#'                            method = "geometric",
#'                            geometric.theta = pi)
#'
#' @param input.data Input data, which can be:
#'   - A Single Cell Object in Seurat or SingleCellExperiment format
#'   - The output of [scRepertoire::combineBCR()] or [combineExpandedBCR()]
#' @param chain Character. Specifies which chain to analyze:
#'   - "Heavy" for the heavy chain
#'   - "Light" for the light chain
#' @param method Character. The algorithm to use for generating latent vectors:
#'   - "encoder" - Uses deep learning autoencoders
#'   - "geometric" - Uses geometric transformations based on the BLOSUM62 matrix
#' @param encoder.model Character. The type of autoencoder model to use:
#'   - "CNN" - CDR3 Convolutional Neural Network-based autoencoder
#'   - "VAE" - CDR3 Variational Autoencoder
#'   - "CNN.EXP" - CDR1/2/3 CNN
#'   - "VAE.EXP" - CDR1/2/3 VAE
#' @param encoder.input Character. Specifies the input features for the encoder model. Options include:
#'   - Amino Acid Properties: "atchleyFactors", "crucianiProperties", "kideraFactors", "MSWHIM","tScales", "zScales"
#'   - "OHE" for One Hot Encoding 
#' @param geometric.theta Numeric. Angle (in radians) for the geometric 
#' transformation. Only used when `method = "geometric"`.
#' @param species Character. Default is "Human" or "Mouse".
#' @param verbose Logical. Whether to print progress messages. Default is TRUE.
#' @return A matrix of latent vectors generated by the specified method.
#'
#' @export
#' @importFrom basilisk basiliskRun
#' @importFrom SeuratObject CreateDimReducObject
#' @importFrom immApex propertyEncoder onehotEncoder geometricEncoder getIR
#' @importFrom tensorflow tf
#' 
#' @seealso 
#' [immApex::propertyEncoder()], 
#' [immApex::geometricEncoder()]
Ibex_matrix <- function(input.data, 
                        chain = c("Heavy", "Light"), 
                        method = c("encoder", "geometric"),
                        encoder.model = c("CNN", "VAE", "CNN.EXP", "VAE.EXP"), 
                        encoder.input = c("atchleyFactors", "crucianiProperties", 
                                          "kideraFactors", "MSWHIM", "tScales", "OHE"),
                        geometric.theta = pi/3, 
                        species = "Human",
                        verbose = TRUE) {

  # Match arguments for better validation
  chain <- match.arg(chain)
  method <- match.arg(method)
  
  if (method == "encoder") {
    encoder.model <- match.arg(encoder.model)
    encoder.input <- match.arg(encoder.input)
    expanded.sequences <- grepl(".EXP", encoder.model)
  } else {
    expanded.sequences <- FALSE
  }
  
  # Define loci based on chain selection
  loci <- if (chain == "Heavy") "IGH" else c("IGK", "IGL")
  
  #Getting Sequences
  BCR <- getIR(input.data, chain, sequence.type = "aa")
  #BCR <- BCR[complete.cases(BCR[["cdr3_aa"]]), ]
  
  # Determine dictionary for sequence encoding
  if (expanded.sequences) {
    #Quick Check to see if there are - corresponding to CDR1-CDR2-CDR3
    if (all(grepl("-", BCR[,"cdr3_aa"]))) {
      stop("Expanded sequences are not properly formated, please use 
           combineExpandedBCR().")
    }
    BCR[,"cdr3_aa"] <- gsub("-", "_", BCR[,"cdr3_aa"])
    dictionary <- c(amino.acids, "_")
  } else  {
    dictionary <- amino.acids
  }
  # Ensure sequences meet length criteria
  checkLength(x = BCR[,"cdr3_aa"], expanded = expanded.sequences)
  length.to.use <- if (expanded.sequences) 90 else 45
  
  if (method == "encoder") {
    if (!.ibex_ensure_external_dir()) {
      stop("Basilisk external directory is not writable; cannot run encoder in this session.")
    }
    
    if (verbose) print("Encoding Sequences...")
    
    if(encoder.input == "OHE") {
      encoded.values <- suppressMessages(onehotEncoder(BCR[,"cdr3_aa"],
                                         max.length = length.to.use,
                                         convert.to.matrix = TRUE,
                                         sequence.dictionary = dictionary,
                                         padding.symbol = "."))[[2]]
    } else {
      encoded.values <- suppressMessages(propertyEncoder(BCR[,"cdr3_aa"], 
                                         max.length = length.to.use,
                                         property.set = encoder.input,
                                         convert.to.matrix = TRUE))[[2]]
    }
    if (verbose) print("Calculating Latent Dimensions...")
    # Getting Model
    model.path <- aa.model.loader(species      = species,  
                                  chain        = chain,
                                  encoder.input = encoder.input,
                                  encoder.model = encoder.model)
    #Getting Reduction 
    reduction <- basiliskRun(
      env = IbexEnv,
      fun = function(mpath, xmat) {
        keras <- reticulate::import("keras", delay_load = FALSE)
        model <- NULL 
        pred <- NULL  
        
        tryCatch({
          model <- keras$models$load_model(mpath)  
          pred  <- model$predict(xmat)
          as.array(pred) # This will be the return value
        }, finally = {
          rm(pred)
          rm(model)
          # Clear the Keras/TensorFlow backend session
          keras$backend$clear_session()
          gc()
        })
      },
      mpath = model.path,
      xmat  = encoded.values
    )
    
  } else if (method == "geometric") {
    if (verbose) print("Performing geometric transformation...")
    BCR[,"cdr3_aa"] <- gsub("-", "", BCR[,"cdr3_aa"])
    reduction <- suppressMessages(geometricEncoder(BCR[,"cdr3_aa"], theta = geometric.theta))[[3]]
  }
  reduction <- as.data.frame(reduction)
  barcodes <- BCR[,"barcode"]
  rownames(reduction) <- barcodes
  colnames(reduction) <- paste0("Ibex_", seq_len(ncol(reduction)))
  return(reduction)
}


