#' @title `AggrHiCExperiment` S4 class
#' 
#' @name AggrHiCExperiment
#' @rdname AggrHiCExperiment
#' 
#' @description
#' 
#' The `AggrHiCExperiment` extends `HiCExperiment` class.
#'
#' @slot fileName Path of Hi-C contact file
#' @slot resolutions Resolutions available in the Hi-C contact file.
#' @slot resolution Current resolution
#' @slot interactions Genomic Interactions extracted from the Hi-C contact file
#' @slot scores Available interaction scores. 
#' @slot slices Available interaction slices. 
#' @slot topologicalFeatures Topological features associated with the dataset 
#'   (e.g. loops (\<Pairs\>), borders (\<GRanges\>), 
#'   viewpoints (\<GRanges\>), etc...)
#' @slot pairsFile Path to the .pairs file associated with the Hi-C contact file
#' @slot metadata metadata associated with the Hi-C contact file.
#' 
#' @param file CoolFile or plain path to a Hi-C contact file
#' @param resolution Resolution to use with the Hi-C contact file
#' @param targets Set of chromosome coordinates for which 
#'   interaction counts are extracted from the Hi-C contact file, provided
#'   as a GRanges object (for diagnoal-centered loci) or as a GInteractions
#'   object (for off-diagonal coordinates).
#' @param flankingBins Number of bins on each flank of the bins containing 
#'   input targets.
#' @param metadata list of metadata
#' @param topologicalFeatures topologicalFeatures provided as a named SimpleList
#' @param pairsFile Path to an associated .pairs file
#' @param bed Path to regions file generated by HiC-Pro
#' @param maxDistance Maximum distance to use when compiling distance decay
#' @param BPPARAM BiocParallel parameters
#' 
#' @return An `AggrHiCExperiment` object.
#' 
#' @include HiCExperiment-class.R
#' @seealso [HiCExperiment()]
#' @examples
#' fpath <- HiContactsData::HiContactsData('yeast_wt', 'mcool')
#' data(centros_yeast)
#' x <- AggrHiCExperiment(
#'   file = fpath, 
#'   resolution = 8000,
#'   targets = centros_yeast[c(4, 7)]
#' )
#' x
#' slices(x, 'count')[1:10, 1:10, 1]
NULL

#' @rdname AggrHiCExperiment
#' @export

methods::setClass("AggrHiCExperiment", contains = c("HiCExperiment"), 
    slots = c(slices = "SimpleList")
)

#' @rdname AggrHiCExperiment
#' @export

AggrHiCExperiment <- function(
    file, 
    resolution = NULL, 
    targets, 
    flankingBins = 50, 
    metadata = list(), 
    topologicalFeatures = S4Vectors::SimpleList(), 
    pairsFile = NULL, 
    bed = NULL,
    maxDistance = NULL,  
    BPPARAM = BiocParallel::bpparam()
) {
    
    ## - Check input arguments
    file <- gsub('~', Sys.getenv('HOME'), file)
    stopifnot(file.exists(file))
    if (!is.null(resolution)) {
        resolution <- as.integer(resolution)
    }
    else {
        resolution <- ifelse({.is_cool(file) | .is_mcool(file)}, 
            .lsCoolResolutions(file)[1], 
            ifelse(.is_hic(file), .lsHicResolutions(file)[1], 
            GenomicRanges::width(.getHicproAnchors(bed))[1])
        )
    }
    if (.is_cool(file)) {
        .check_cool_file(file)
        .check_cool_format(file, NULL)
        resolutions <- .lsCoolResolutions(file)
    } 
    else if (.is_mcool(file)) {
        .check_cool_file(file)
        .check_cool_format(file, resolution)
        resolutions <- .lsCoolResolutions(file)
    }
    else if (.is_hic(file)) {
        .check_hic_file(file)
        .check_hic_format(file, resolution)
        resolutions <- .lsHicResolutions(file)
    }
    else if (.is_hicpro_matrix(file)) {
        .check_hicpro_files(file, bed)
        if (is.null(bed)) stop("Regions files not provided.")
        resolutions <- resolution
    }
    
    ## - Reformat targets
    if (is(targets, 'GRanges')) {
        # Need to check that targets are OK (unique width, greater than 0, ...)
        targets <- GenomicRanges::resize(
            targets, fix = 'center', width = resolution*{2*flankingBins}
        )
        pairs <- S4Vectors::Pairs(targets, targets)
    }
    else if (is(targets, 'GInteractions')) {
        # Need to check that pairs are OK (all width = 1)
        ans <- InteractionSet::anchors(targets)
        an1 <- ans[[1]]
        an1 <- resize(an1, fix = 'center', width = resolution*{2*flankingBins})
        an2 <- ans[[2]]
        an2 <- resize(an2, fix = 'center', width = resolution*{2*flankingBins})
        pairs <- InteractionSet::swapAnchors(InteractionSet::GInteractions(
            an1, an2
        ))
        pairs <- as(pairs, 'Pairs')
    }
    else {
        stop("Please provide `targets` as `GRanges` or `GInteractions`.")
    }

    ## - Compute aggregated scores
    gis <- .multi2DQuery(
        file, 
        resolution, 
        pairs = pairs, 
        bed = bed, 
        BPPARAM = BPPARAM, 
        maxDistance = maxDistance
    )
    mdata <- S4Vectors::metadata(gis)
    S4Vectors::metadata(gis) <- list()
    mcols <- GenomicRanges::mcols(gis)
    GenomicRanges::mcols(gis) <- NULL
    
    ## -- Create contact object
    x <- methods::new("AggrHiCExperiment", 
        fileName = as.character(file),
        focus = paste0(length(targets), " aggregated targets"), 
        resolutions = resolutions, 
        resolution = resolution, 
        interactions = gis, 
        scores = S4Vectors::SimpleList(
            'count' = as.numeric(mcols$count), 
            'balanced' = as.numeric(mcols$balanced), 
            'expected' = as.numeric(mcols$expected), 
            'detrended' = as.numeric(mcols$detrended)
        ),
        slices = mdata$slices, 
        topologicalFeatures = S4Vectors::SimpleList(
            c(targets = mdata$pairs, as.list(topologicalFeatures))
        ),
        pairsFile = pairsFile, 
        metadata = metadata
    )
    methods::validObject(x)
    return(x)

}
